/*
 * Copyright (c) 2001 Sonic Software. All Rights Reserved.
 */

package com.sonicsw.mf.common;

import com.sonicsw.mf.common.config.IBasicElement;
import com.sonicsw.mf.common.config.IBlob;
import com.sonicsw.mf.common.config.IElement;
import com.sonicsw.mf.common.config.IElementIdentity;
import com.sonicsw.mf.common.config.IIdentity;
import com.sonicsw.mf.common.config.INextVersionToken;
import com.sonicsw.mf.common.config.query.Query;
import com.sonicsw.mf.common.dirconfig.DirectoryServiceException;
import com.sonicsw.mf.common.dirconfig.IDeltaDirElement;
import com.sonicsw.mf.common.dirconfig.IDirElement;
import com.sonicsw.mf.common.dirconfig.IDirIdentity;
import com.sonicsw.mf.common.dirconfig.InvalidXMLException;
import com.sonicsw.mf.common.dirconfig.VersionOutofSyncException;
import com.sonicsw.mf.common.runtime.IBackupStatus;
import com.sonicsw.mf.common.view.IDeltaView;
import com.sonicsw.mf.common.view.IView;

/**
 * IDirectoryAdminService exposes API methods for creating and
 * maintaining directories and IDirElement objects; API methods for retrieving IDirElement and IIdentity objects;
 * And methods to import and export the directory information in XML format. IDirectoryAdminService methods are thread safe and
 * optimistic concurrency control is used to coordinate between multiple utilities updating the same IDirElement
 * objects concurrently. Each domain has a single IView object which describes a logical, easy to modify, organization
 * of the domain. Directory names cannot contain the '\' and ':' characters. Two identical names (of directories and elements)
 * where the only difference is the case should not be created - that doesn't work on MS platforms (e.g. '/containers/Ab'
 * and '/containers/ab'). The attachBlob(), getBlob() and detachBlob() methods can be used to handle large (up to 10M) binary
 * data. Binary data is stored separately and cannot be exported or imported through XML.
 */

public interface IDirectoryAdminService extends IDirectoryAdminNotifications, IBlobSource
{

    /**
     * Returns the version of this Directory Service instance.
     *
     * @return the version of this Directory Service instance
     */
    public int getDirectoryServiceVersion();

   /**
     * Returns the release version of this Directory Service instance.
     *
     * @return the release version of this Directory Service instance in a  MAJOR_VERSION.MINOR_VERSION.POINT_VERSION format
     */
    public String getDirectoryServiceReleaseVersion();

    /**
     * Returns the domain name of this directoryservice object
     *
     * @return String or IAttributeList or IAttributeSet
     */
    public String getDomain();

    /**
     * Returns this domain's view.
     *
     * @return this domain's view
     *
     * @exception DirectoryServiceException if there is a problem accessing the underlying storage
     *
     */
    public IView getView()
        throws DirectoryServiceException;

    /**
     * Updates this domain's view.
     *
     * @param view modification to the view (can be null)
     *
     *
     * @return the new view
     *
     * @exception DirectoryServiceException if there is a problem accessing the underlying storage
     * @exception VersionOutofSyncException if the view cannot be modified because of a conflict with a previous view update
     *                                      by another application
     *
     */
    public IView setView(IDeltaView view)
        throws DirectoryServiceException, VersionOutofSyncException;

    /**
     * Updates the view and returns a token to get the next view's version without shipping the entire view back to the
     * client.
     *
     * @param view modification to the view (can be null)
     *
     * @return a token to be used in IView.getNextVersion
     *
     * @exception DirectoryServiceException if there is a problem accessing the underlying storage
     * @exception VersionOutofSyncException if the view cannot be modified because of a conflict with a previous view update
     *                                      by another application
     *
     */
    public INextVersionToken setViewGetToken(IDeltaView view)
        throws DirectoryServiceException, VersionOutofSyncException;

    /**
     * Returns an element by name.
     *
     * @param elementName the name of the element
     * @param forUpdate true if the caller might update this element
     *
     * @return the element object (or null if the element is not in the directory)
     *
     * @exception DirectoryServiceException if there is a problem accessing the underlying storage
     *
     */
    public IDirElement getElement(String elementName, boolean forUpdate)
        throws DirectoryServiceException;

    /**
     * Returns the element if the caller does not have the correct version. Used by the launcher to get the container
     * configuration if not already in cache. if the container's cache backup version is different from the DS's
     * then the configuration element must be returned.
     *
     * @param callerBackupVersion the container's cache backup version
     * @param elementName the name of the element
     * @param id true if thed of the element already in cache - null if not in the cache
     *
     * @return a single array element that contains the element if the caller does not have the correct version or
     *         null if the caller does have the correct version or a single null element if the element was not
     *         found.
     *
     * @exception DirectoryServiceException if there is a problem accessing the underlying storage
     *
     */
    public IDirElement[] getElementIfUpdated(long callerBackupVersion, String elementName, IElementIdentity id)
      throws DirectoryServiceException;

    /**
     * Returns an element by name.
     *
     * @param elementName the name of the element
     * @param forUpdate true if the caller might update this element
     * @param getSubclassingDelta true if the caller wants the subclassing delta to be attached to the element
     *        in cases where elementName is a sub-classed element (use IDirElement.getSubclassingDelta to
     *        get the delta from the element).
     *
     * @return the element object (or null if the element is not in the directory)
     *
     * @exception DirectoryServiceException if there is a problem accessing the underlying storage
     *
     */
    public IDirElement getElement(String elementName, boolean forUpdate, boolean getSubclassingDelta)
        throws DirectoryServiceException;

    /**
     * Returns the elements that matches the FROM clause and the WHERE clause of the query. Only
     * attributes selected by the SELECT clause are fetched. All the attributes are fetched if the query
     * does not contain a SELECT clause.
     *
     * @param query with a SELECT clause (optional), a FROM clause (mandatory) and a WHERE clause (optional)
     * @param forUpdate true if the caller might update this element. forUpdate must be false if the
     *                  query contains a WHERE clause.
     * @param getSubclassingDelta true if the caller wants the subclassing delta to be attached to the elements
     *        in cases where elementName is a sub-classed element (use IDirElement.getSubclassingDelta to
     *        get the delta from the element).
     *
     * @return the elements that match the query (can be an empty list)
     *
     * @exception DirectoryServiceException if there is a problem accessing the underlying storage or the parametrs are invalid
     *
     * @see com.sonicsw.mf.common.config.query.Query
     *
     */
    public IDirElement[] getElements(Query query, boolean forUpdate, boolean getSubclassingDelta)
        throws DirectoryServiceException;

    /**
     * Returns the elements that matches the FROM clause and the WHERE clause of the query. Only
     * attributes selected by the SELECT clause are fetched. All the attributes are fetched if the query
     * does not contain a SELECT clause.
     *
     * @param query with a SELECT clause (optional), a FROM clause (mandatory) and a WHERE clause (optional)
     * @param forUpdate true if the caller might update this element. forUpdate must be false if the
     *                  query contains a WHERE clause.
     *
     * @return the elements that match the query (can be an empty list)
     *
     * @exception DirectoryServiceException if there is a problem accessing the underlying storage or the parametrs are invalid
     *
     * @see com.sonicsw.mf.common.config.query.Query
     *
     */
    public IDirElement[] getElements(Query query, boolean forUpdate)
        throws DirectoryServiceException;

    /**
     * Returns an identity of a directory or an element
     *
     * @param name the name of the element or the directory
     *
     * @return the identity (or null if there is no element or directory with this name)
     *
     * @exception DirectoryServiceException if there is a problem accessing the underlying storage
     *
     */
    public IIdentity getIdentity(String name)
        throws DirectoryServiceException;

    /**
     * Returns a list of all the elements under a specific directory
     *
     * @param dirName the name of the directory
     * @param forUpdate true if the caller might update these elements
     *
     * @return a list of the elements
     *
     * @exception DirectoryServiceException if there is a problem accessing the underlying storage
     *
     */
    public IDirElement[] getAllElements(String dirName, boolean forUpdate)
        throws DirectoryServiceException;

    /**
     * Returns a byte array of the <b>zipped, serialized</b> form of all the elements under a specific directory
     *
     * @param dirName the name of the directory
     * @param forUpdate true if the caller might update these elements
     *
     * @return a byte array of the <b>zipped, serialized</b> form of all the elements
     *
     * @exception DirectoryServiceException if there is a problem accessing the underlying storage or during zipping
     *
     */
    public IDirElement[] getAllElementsCompressed(String dirName, boolean forUpdate) throws DirectoryServiceException;

    /**
     * Returns the identities of all the elements and directories under a specific directory.
     *
     * @param dirName the name of the directory
     *
     * @return the list of identities
     *
     * @exception DirectoryServiceException if there is a problem accessing the underlying storage
     *
     */
    public IIdentity[] listAll(String dirName)
        throws DirectoryServiceException;

    /**
     * Returns the identities of all the directories under a specific directory.
     *
     * @param dirName the name of the directory
     *
     * @return the list of directory identities
     *
     * @exception DirectoryServiceException if there is a problem accessing the underlying storage
     *
     */
    public IDirIdentity[] listDirectories(String dirName)
        throws DirectoryServiceException;

    /**
     * Returns the identities of all the elements under a specific directory.
     *
     * @param dirName the name of the directory
     *
     * @return the list of element identities
     *
     * @exception DirectoryServiceException if there is a problem accessing the underlying storage
     *
     */
    public IElementIdentity[] listElements(String dirName)
        throws DirectoryServiceException;

    /**
     * Deletes a specific element and update the view as one operation.
     *
     * @param elementName the name of the element
     * @param view modification to the view (can be null)
     *
     * @exception DirectoryServiceException if there is a problem accessing the underlying storage.
     * @exception VersionOutofSyncException if the view cannot be modified because of a conflict with a previous view update
     *                                      by another application
     *
     * @return the identity of the deleted element or null if it doesn't exist.
     *
     */
    public IElementIdentity deleteElement(String elementName, IDeltaView view)
        throws DirectoryServiceException, VersionOutofSyncException;

    /**
     * Creates a new directory.
     *
     * @param dirName the name of the new directory (the parent directory must exist)
     *
     * @exception DirectoryServiceException if there is a problem accessing the underlying storage
     *
     */
    public void createDirectory(String dirName)
        throws DirectoryServiceException;

    /**
     * Deletes a directory (the directory must be empty).
     *
     * @param dirName the name of the new directory (the directory must exist)
     *
     * @exception DirectoryServiceException if there is a problem accessing the underlying storage or the directory does not exist
     *
     */
    public void deleteDirectory(String dirName)
        throws DirectoryServiceException;

    /**
     * Deletes a directory and its contents (cascading delete);
     *
     * @param dirName the name of the new directory (the directory must exist)
     * @param view modification to the view (can be null)
     *
     * @exception DirectoryServiceException if there is a problem accessing the underlying storage or the directory does not exist
     *
     */
    public void deleteDirectory(String dirName, IDeltaView view)
        throws DirectoryServiceException;

    /**
     * Applies a modification to a an element or creates a new one. VersionOutofSyncException is thrown if the element does not exist and
     * 'element' is not an INITIAL_VERSION element or if the element exists and 'element' is an INITIAL_VERSION
     * object or if the version of 'element' is not equal to the version of the element in the central repository.
     * A normal situation where a VersionOutofSyncException might be thrown is when two different applications
     * update the central repository concurrently and access the same element. The setElement method also allows to update the domain's
     * view.
     *
     * @param element the element or delta element to be applied to the  central repository
     * @param view modification to the view (can be null)
     *
     * @return A token to be passed to the IView.getNextVersion and IDirElement.getNextVersion methods
     *
     * @exception VersionOutofSyncException if there is a conflict in updating the element or the view because of a previous update by
     *                                      another application
     * @exception DirectoryServiceException if there is a problem accessing the underlying storage or the directory does not exist
     *
     */
    public INextVersionToken setElement(IBasicElement element, IDeltaView view)
        throws DirectoryServiceException, VersionOutofSyncException;

    /**
     * Sets and/or delete multiple elements under a single transaction. setElements is equivalent to a sequence of
     * deleteElement calls followed by a sequence of setElement calls and finally a setView call. setElements is different
     * from that sequence of calls being executed under a single transaction and a single remote call.
     *
     * @param elements[] the element or delta element - can be null
     * @param deleteList[] the names of elements to be deleted - can be null
     * @param view modification to the view (can be null)
     *
     * @return A token to be passed to the IView.getNextVersion and IDirElement.getNextVersion methods
     *
     * @exception VersionOutofSyncException if there is a conflict in updating the element or the view because of a previous update by
     *                                      another application
     * @exception DirectoryServiceException if there is a problem accessing the underlying storage or the directory does not exist
     *
     */
    public INextVersionToken setElements(IBasicElement[] elements, String[] deleteList, IDeltaView view)
        throws DirectoryServiceException, VersionOutofSyncException;

    /**
     * Clone an element to create a new element with a new name identical in content to the original. Super elements cannot be cloned.
     *
     * @param elementName the name of the element to be cloned
     * @param newName the name of the new element
     * @param view modification to the view (can be null)
     *
     * @return the new element
     *
     * @exception DirectoryServiceException if there is a problem accessing the underlying storage or the element does not exist
     *            or the element is a super element or the directory for the new element doesn't exist
     * @exception VersionOutofSyncException if there is a conflict in updating the element or the view because of a previous update by
     *                                      another application
     *
     */
    public IDirElement cloneElement(String elementName, String newName, IDeltaView view)
        throws DirectoryServiceException, VersionOutofSyncException;

    /**
     * Clone an element to create a new element with a new name and some modifications relative to the original content.
     * Super elements cannot be cloned.
     *
     * @param delta the delta returned from the IDirElement.doneUpdate() applied on the original element
     * @param newName the name of the new element
     * @param view modification to the view (can be null)
     *
     * @return the new element
     *
     * @exception DirectoryServiceException if there is a problem accessing the underlying storage or the element does not exist
     *            or the element is a super element or the directory for the new element doesn't exist
     * @exception VersionOutofSyncException if there is a conflict in applying the delta or modifying the view because of a previous update by
     *                                      another application
     *
     */
    public IDirElement cloneElement(IBasicElement delta, String newName, IDeltaView view)
        throws DirectoryServiceException,
        VersionOutofSyncException;

    /**
     * Clone an element to create a new element with a new name identical in content to the original. Super elements cannot be cloned.
     *
     * @param elementName the name of the element to be cloned
     * @param newName the name of the new element
     * @param createTemplate make the clone a template (if 'true')
     * @param view modification to the view (can be null)
     *
     * @return the new element
     *
     * @exception DirectoryServiceException if there is a problem accessing the underlying storage or the element does not exist
     *            or the element is a super element or the directory for the new element doesn't exist
     * @exception VersionOutofSyncException if there is a conflict in updating the element or the view because of a previous update by
     *                                      another application
     *
     */
    public IDirElement cloneElement(String elementName, String newName, boolean createTemplate, IDeltaView view)
        throws DirectoryServiceException, VersionOutofSyncException;

    /**
     * Clone an element to create a new element with a new name and some modifications relative to the original content.
     * Super elements cannot be cloned.
     *
     * @param delta the delta returned from the IDirElement.doneUpdate() applied on the original element
     * @param newName the name of the new element
     * @param createTemplate make the clone a template (if 'true')
     * @param view modification to the view (can be null)
     *
     * @return the new element
     *
     * @exception DirectoryServiceException if there is a problem accessing the underlying storage or the element does not exist
     *            or the element is a super element or the directory for the new element doesn't exist
     * @exception VersionOutofSyncException if there is a conflict in applying the delta or modifying the view because of a previous update by
     *                                      another application
     *
     */
    public IDirElement cloneElement(IBasicElement delta, String newName, boolean createTemplate, IDeltaView view)
        throws DirectoryServiceException, VersionOutofSyncException;

    /**
     * Exports an element into an XML document.
     *
     * @param elementName the name of the directory or element
     *
     * @return the XML format of the element (null if the element does not exist)
     *
     * @exception DirectoryServiceException if there is a problem accessing the underlying storage
     *
     */
    public String exportElementToXML(String elementName)
        throws DirectoryServiceException;
    
    /**
     * Exports the contents of the DS boot file as an XML document.
     *
     * @param elementName the name of the Directory Service element
     *
     * @return the XML format of the Directory Service boot file (null if the element does not exist)
     *
     * @exception DirectoryServiceException if there is a problem accessing the underlying storage
     *
     */
    public String exportDSBootFileString(String elementName)
        throws DirectoryServiceException;

    /**
     * Exports an entire directory into an XML document.
     *
     * @param directoryName the name of the directory
     *
     * @return the XML format of the directory (null if the directory does not not exist)
     *
     * @exception DirectoryServiceException if there is a problem accessing the underlying storage
     *
     */
    public String exportDirectoryToXML(String directoryName)
        throws DirectoryServiceException;

    /**
     * Dumps an entire directory into an XML document in the current working directory of the
     * container hosting the DS. The dump file will be named "dump.<domain name>.<n>.xml" where
     * <n> is a version number one greater than existing dump files (starting at 0).
     */
    public void dumpContentsToXML();

    /**
     * Import a directory or an element from an XML document.
     *
     * @param XMLDocument the XML format the element or directory
     *
     * @exception DirectoryServiceException if there is a problem accessing the underlying storage
     * @exception InvalidXMLException if the XMLDocument is not a valid XML document
     *
     */
    public void importFromXML(String XMLDocument)
        throws DirectoryServiceException, InvalidXMLException;

    /**
     * Stores a BLOB in the DS storage. The BLOB data is described by the element (the element's content is determined by the application).
     * The BLOB is stored in the file-system under <domain-name>/blobs  with a name
     * identical to the element's full name. For example, if the element's name is '/jars/broker' then the BLOB is stored under
     * '<domain-name>/blobs//jars/broker'.
     * attachBlob() can be called multiple times to replace the attachment. The maximum size of the BLOB is 10 MB.
     *
     * @param element the element the BLOB is attached to. Element can be a new IDirElement or IDeltaDirElement object returned
     * from IDirElement.doneUpdate()
     * @param blobStream the stream of bytes that makes the BLOB
     * @param view modification to the view (can be null)
     *
     * @exception DirectoryServiceException if there is a problem accessing the underlying storage
     * @exception VersionOutofSyncException if there is a conflict in applying the delta or modifying the view because of a previous update by
     *                                      another application
     *
     */
    public void attachBlob(IBasicElement element, java.io.InputStream blobStream, IDeltaView view)
        throws DirectoryServiceException, VersionOutofSyncException;

// should not be documented or added to DirectoryServiceProxy - used internally
    public void attachBlob(IBasicElement element, byte[] blob, IDeltaView view)
        throws DirectoryServiceException, VersionOutofSyncException;

    /**
     * Gets the BLOB from the DS storage.
     *
     * @param elementName the name of the element the BLOB is attached to.
     * @param forUpdate true if the caller might update the element and/or the attachment
     *
     * @return an IBlob object that contains references to the element and the InputStream object
     *
     * @exception DirectoryServiceException if there is a problem accessing the underlying storage
     *
     */
    public IBlob getBlob(String elementName, boolean forUpdate)
        throws DirectoryServiceException;

    /**
     * Deletes the BLOB attached to the element.
     *
     * @param delta a modification delta returned from IDirElement.doneUpdate().
     * @param view modification to the view (can be null)
     *
     * @exception DirectoryServiceException if there is a problem accessing the underlying storage
     * @exception VersionOutofSyncException if there is a conflict in applying the delta or modifying the view because of a previous update by
     *                                      another application
     *
     */
    public void detachBlob(IDeltaDirElement delta, IDeltaView view)
        throws DirectoryServiceException, VersionOutofSyncException;

    /**
     * Subclass an element from another (super) element.
     * The super element is fetched (for update), modified and then the returned value from superElement.doneUpdateForSubclassing()
     * is passed to the subclassElement() method. All modifications to the super element are automatically reflected in the subclassed
     * element. Attributes overwritten by the subclassed element are not affected when modified at the super element. The new
     * subclassed element cannot be further subclassed.
     *
     * Packed elements cannot be sub-classed. Currently packed elements are all the elements under _MfUsers directories.
     * Examples:  /security/_MFUsers,  /_MFUsers and  /security/domain1/_MFUsers.
     *
     * @param delta the delta returned from the IDirElement.doneUpdateForSubclassing() applied on the super element
     * @param newName the name of the new element
     * @param view modification to the view (can be null)
     *
     * @return the new subclassed element
     *
     * @exception DirectoryServiceException if there is a problem accessing the underlying storage or the super element does not exist
     *            or the name of the new element is invalid or the delta is not a product of a doneUpdateForSubclassing() class.
     * @exception VersionOutofSyncException if there is a conflict in applying the delta or modifying the view because of a previous update by
     *                                      another application or a 'newName' element already exists.
     *
     */
    public IDirElement subclassElement(IBasicElement delta, String newName, IDeltaView view)
        throws DirectoryServiceException,
        VersionOutofSyncException;

    /**
     * Removes the binding between the super element and a subclassed element. If elementName is already independent then unSubclassElement
     * throws an exception.
     *
     * @param elementName name  of the subclassed element
     * @param view modification to the view (can be null)
     *
     * @exception DirectoryServiceException if there is a problem accessing the underlying storage or elementName is invalid (or not sub-classed)
     * @exception VersionOutofSyncException if there is a conflict in modifying the view because of a previous update by another application.
     *
     */
    public void unSubclassElement(String elementName, IDeltaView view)
        throws DirectoryServiceException, VersionOutofSyncException;


    /**
     * Set a new release version in the Element's header. Used only by version upgrade programs.
     *
     * @param elementName
     * @param newVersion
     *
     * @exception DirectoryServiceException if there is a problem accessing the underlying storage.
     *
     */
    public void setElementReleaseVersion(String elementName, String newVersion)
        throws DirectoryServiceException;

    /**
     * This method is used by PASS feature.
     * <p>
     * @return list of External Domain descriptor that have been configured with
     * Management SPI. For Example, one of the element can be
     * <code>/authentication/domains/0/_MFDomainDescriptor </code>
     * <p>
     * @throws DirectoryServiceException
     */
    public String[] listExternalDomainWithManagementSPI()
        throws DirectoryServiceException;

    /**
     * This method is used by PASS
     * <p>
     * This method reloads the External Domain that matches the descriptor passed in.
     * For Example, passed in descriptor can be
     * <code> /authentication/domains/0/_MFDomainDescriptor </code>
     * <p>
     * @param mfDomainDescriptor
     * <p>
     * @return Boolean.TRUE if successful
     * <p>
     * @throws DirectoryServiceException
     */
    public Boolean reloadExternalAuthenticationDomain(String mfDomainDescriptor)
        throws DirectoryServiceException;

    /**
     * Upgrade a configuration, given its logical name and path to version 6.
     *
     * @param configType the lead element type
     * @param configLogicalPath the logical path of this configuration
     */
    public void upgrade5to6(String configType, String configLogicalPath)
        throws DirectoryServiceException;

    /**
     * Gets a list of elements through their logical names. The elements contain their storage identity.
     * <p>
     * @param logicalNames List of names (full paths)
     * <p>
     * @return An array of elements (with storage identity). Null is returned for each missing element.
     * <p>
     * @throws DirectoryServiceException if there is a problem accessing the underlying storage
     */
    public IElement[] getElementsByLogicalNames(String[] logicalNames) throws DirectoryServiceException;

    /**
     * Gets a blob through its logical name. The element n the bloc contains its storage identity.
     * <p>
     * @param logicalName full path
     * <p>
     * @return A blob. Null is returned if the blob is missing
     * <p>
     * @throws DirectoryServiceException if there is a problem accessing the underlying storage
     */
    public IBlob getBlobByLogicalName(String logicalName) throws DirectoryServiceException;

  /**
     * Gets a blob through its logical name. The element n the bloc contains its storage identity.
     * <p>
     * @param unusedSubscriber to support the IBlobSource interface
     * @param logicalName full path
     * <p>
     * @return A blob. Null is returned if the blob is missing
     * <p>
     * @throws DirectoryServiceException if there is a problem accessing the underlying storage
     */
    @Override
    public IBlob getBlobByLogicalName(String unusedSubscriber, String logicalName) throws DirectoryServiceException;

    /**
     * Initiate the creation of a backup copy of DS storage, while the DS
     * is running and operational.  This method returns immediately.
     * The backup operation itself executes asynchronously.
     * When the backup operation completes, a message is written to the DS
     * container's log file indicating success or failure.
     *
     * @param backupPath The location to put the backup copy of DS storage.
     *   It may be absolute or relative to the DS container's working directory.
     *   Examples: "C:\Sonic\MQ7.5\Domain1.backup" or "./Domain1.backup"
     *
     * @param overwrite If false, backupPath should not exist.
     *   If true, anything existing at backupPath location will be overwritten.
     *
     * @throws BackupAlreadyInProgess if a backup is already in progress
     *
     * @throws BackupPathExists if overwrite is false and backupPath exists
     *
     * @throws InvalidBackupPath if the parent of the backupPath does not
     *   exist or is not accessible
     */


    public void startBackup(String backupDir, boolean overwrite) throws DirectoryServiceException;

    /**
     * @return the status of the current or most recent DS backup operation. 
     * @see com..sonicsw.mf.common.runtime.IBackupStatus
     */
    public IBackupStatus getBackupStatus() throws DirectoryServiceException;



}

