/*
 * Copyright (c) 2001 Sonic Software. All Rights Reserved.
 */

package com.sonicsw.mf.common;

import com.sonicsw.mf.common.dirconfig.DirectoryServiceException;
import com.sonicsw.mf.common.dirconfig.IDirElement;
import com.sonicsw.mf.common.view.IView;

/**
 * Local DS access for efficient caching and bulk operations. These operations are supported only locally
 * (not from remote clients) and are used by MF internals. It's in the common package because some of the 
 * methods are called from the common.xml package.
 */

public interface IDirectoryCacheService
{
 /**
  * Returns an element and all the other elements that can be retrieved without any extra I/O. It is possible
  * that the requested element will not be found, but getElements still returns other elements that were
  * read while the element was serached for.
  *
  * @param elementName the name of the element
  * @param forUpdate true if the caller might update this element
  *
  * @return the elements list (an empty list if no elements were retrieved).
  *
  * @exception DirectoryServiceException if there is a problem accessing the underlying storage
  *
  */
  public IDirElement[] getElements(String elementName, boolean forUpdate) throws DirectoryServiceException;

 /**
  * Deletes elements under a single directory efficiently. Can be called only from the ImportManager only.
  *
  * @param elementNames the list of element names.
  *
  * @exception DirectoryServiceException if there is a problem accessing the underlying storage or if not all the element under the 
  *                                      same directory
  *
  */
  public void deleteElements(String[] elementNames) throws DirectoryServiceException;

 /**
  * Sets elements under a single directory efficiently. Can be called only from the ImportManager only.
  *
  * @param elements the list of elements to set
  *
  * @exception DirectoryServiceException if there is a problem accessing the underlying storage or if not all the element under the
  *                                      same directory
  */
  public void setElements(IDirElement[] elements) throws DirectoryServiceException;

 /**
  * Sets the view directly to storage without starting a transaction.
  *
  * @param view
  * @return a clean view object - without the modification history
  *
  * @exception DirectoryServiceException if there is a problem accessing the underlying storage or if not all the element under the
  *                                      same directory
  */
  public IView storeViewInternal(IView view) throws DirectoryServiceException;
}
