/*
 * Copyright (c) 2001 Sonic Software. All Rights Reserved.
 */

package com.sonicsw.mf.common;

import com.sonicsw.mf.common.config.IElementChangeHandler;
import com.sonicsw.mf.common.config.INameChangeHandler;

/**
 *  The interface is used by administration clients who wish to receive and handle configuration
 *  change notifications. It uses the folder/logical-names naming scheme.
 */
public interface IDirectoryFileSystemNotifications
{
    /**
     * Register a configuration change handler to receive changes relating to any configuration
     * stored in the DS.
     *
     * @param handler The handler to be invoked on a change
     *
     * @see #unregisterElementChangeHandler(IElementChangeHandler)
     */
    public void registerFSElementChangeHandler(IElementChangeHandler handler);

   /**
     * Register a name space change handler to receive changes relating to any new or renamed
     * folders or elements or meta-data attributes.
     *
     * The following naming events should be handled by the handler:
     *
     * CreateElement(String elementPath) - Will not be sent for complex configuration elements (under _MFUsers, for example)
     * DeleteElement(String elementPath)
     * CreateFolder(String folderPath)
     * DeleteFolder(String folderPath)
     * Rename(String oldName, String newName)
     * MetaAttributes(String path, HashMap attributes)
     *
     * @param handler The handler to be invoked on a change
     *
     * @see #unregisterNameChangeHandler(IElementChangeHandler)
     */
    public void registerNameChangeHandler(INameChangeHandler handler);


    /**
     * Register a configuration change handler to receive changes relating to the
     * named configuration stored in the DS.
     *
     * @param elementPath The element the handler is interested in
     * @param handler     The handler to be invoked on a change
     *
     * @see #unregisterElementChangeHandler(String, IElementChangeHandler)
     */
    public void registerFSElementChangeHandler(String elementPath, IElementChangeHandler handler);

    /**
     * Unregister a previously registered handler that expressed interest in changes to all
     * configuration stored in the DS.
     *
     * @param handler
     */
    public void unregisterFSElementChangeHandler(IElementChangeHandler handler);

    /**
     * Unregister a previously registered handler that expressed interest in changes relating to any new or renamed
     * folders or elements or meta-data attributes.
     *
     * @param handler
     */
    public void unregisterNameChangeHandler(INameChangeHandler handler);

    /**
     * Unregister a previously registered handler that expressed interest in changes to the
     * named configuration stored in the DS.
     *
     * @param elementPath The element the handler is interested in
     * @param handler     The handler to be invoked on a change
     */
    public void unregisterFSElementChangeHandler(String elementPath, IElementChangeHandler handler);


}
