/*
 * Copyright (c) 2001 Sonic Software. All Rights Reserved.
 */

package com.sonicsw.mf.common;

import java.util.ArrayList;
import java.util.HashMap;

import com.sonicsw.mf.common.config.IBasicElement;
import com.sonicsw.mf.common.config.IBlob;
import com.sonicsw.mf.common.config.IElementIdentity;
import com.sonicsw.mf.common.config.INextVersionToken;
import com.sonicsw.mf.common.config.query.AttributeName;
import com.sonicsw.mf.common.config.query.Query;
import com.sonicsw.mf.common.config.query.QueryBatch;
import com.sonicsw.mf.common.dirconfig.DirectoryServiceException;
import com.sonicsw.mf.common.dirconfig.IDeltaDirElement;
import com.sonicsw.mf.common.dirconfig.IDirElement;
import com.sonicsw.mf.common.dirconfig.VersionOutofSyncException;

/**
 * The interface is used by administration clients to access the DS through the logical name space.
 */
public interface IDirectoryFileSystemService
extends IDirectoryFileSystemNotifications
{

    /** *********************************************************************** */
    /** ****************** Name Space Manipulation Methods ******************** */
    /** *********************************************************************** */

    /**
     * Called once to set a list of folder String attributes. Typically set when the DS is seeded rather than using this
     * method. Note that attributes are not supported for folders in a complex configuration (such as
     * /na/usa/bedford/authDomain/_MFUsers
     *
     * @param attributeNames
     *            the list of folder attributes
     *
     * @exception DirectoryServiceException
     *                if there is a problem accessing the underlying storage or the attribute list was already set.
     *
     */
    public void defineFolderMetaAttributes(String[] attributeNames)
    throws DirectoryServiceException;

    /**
     * Gets the list of defined folder attributes
     *
     * @return the list of defined folder attributes
     *
     * @exception DirectoryServiceException
     *                if there is a problem accessing the underlying storage or the attribute list was already set.
     *
     */
    public String[] getDefinedFolderMetaAttributes()
    throws DirectoryServiceException;

    /**
     * Called once to set a list of element String attributes. Typically set when the DS is seeded rather than using
     * this method. Note that attributes are not supported for elements under a complex configuration folder.
     *
     * @param attributeNames
     *            the list of element attributes
     *
     * @exception DirectoryServiceException
     *                if there is a problem accessing the underlying storage or the attribute list was already set.
     *
     */
    public void defineElementMetaAttributes(String[] attributeNames)
    throws DirectoryServiceException;

    /**
     * Gets the list of defined element attributes
     *
     * @return the list of defined element attributes
     *
     * @exception DirectoryServiceException
     *                if there is a problem accessing the underlying storage or the attribute list was already set.
     *
     */
    public String[] getDefinedElementMetaAttributes()
    throws DirectoryServiceException;

    /**
     * Rename a folder or an element. Not supported for folders or elements in a complex configuration
     *
     * @param oldPath
     *            of folder or element
     * @param newPath
     *            of folder or element
     *
     * @exception DirectoryServiceException
     *                if there is a problem accessing the underlying storage or oldName does not exist or there is no
     *                folder for the new name
     *
     */
    public void rename(String oldPath, String newPath)
    throws DirectoryServiceException;

    /**
     * Rename a folder. Not supported for folders or elements in a complex configuration
     *
     * @param oldPath
     *            of folder
     * @param newPath
     *            of folder
     *
     * @exception DirectoryServiceException
     *                if there is a problem accessing the underlying storage or oldName does not exist or is not a folder or there is no
     *                folder for the new name
     *
     */
    public void renameFolder(String oldPath, String newPath)
    throws DirectoryServiceException;

    /**
     * Rename a file.
     *
     * @param oldPath
     *            of file
     * @param newPath
     *            of file
     *
     * @exception DirectoryServiceException
     *                if there is a problem accessing the underlying storage or oldName does not exist or is not a file or there is no
     *                folder for the new name
     *
     */
    public void renameFile(String oldPath, String newPath)
    throws DirectoryServiceException;


    /**
     * Create a new folder. The parent folder must exist.
     *
     * @param folderPath
     *            the name of the new folder
     *
     * @exception DirectoryServiceException
     *                if there is a problem accessing the underlying storage or the parent folder doesn't exist
     *
     */
    public void createFolder(String folderPath)
    throws DirectoryServiceException;

    /**
     * Create a new folder. The parent folder must exist.
     *
     * @param folderPath
     *            the name of the new folder
     *
     * @exception DirectoryServiceException
     *                if there is a problem accessing the underlying storage or the parent folder doesn't exist
     *
     */
    public void createFolder(String folderPath, boolean existingOk)
    throws DirectoryServiceException;

    /**
     * Copy the 'fromPath' file or folder into 'toPath'. Deep copy is performed if 'fromPath' is a folder. If 'toPath' is an existing
     * folder then 'fromPath' is copied into it. If 'toPath' does not exit then the content of 'fromPath' is copied into its parent.
     * An exception is thrown if the parent of 'toPath' does not exist.
     *
     * @param fromPath a file or a folder source
     * @param toPath the target path
     *
     * @exception DirectoryServiceException
     *                if there is a problem accessing the underlying storage or the parent folder of 'toPath' doesn't exit
     *
     */
    public void copyFiles(String fromPath, String toPath)
    throws DirectoryServiceException;

    /**
     * Delete a folder. Only folders that represent a complex configuration can be deleted, when not empty, with their
     * entire content.
     *
     * @param folderPath
     *            the name of the folder to be deleted
     *
     * @exception DirectoryServiceException
     *                if there is a problem accessing the underlying storage or the folder doesn't exit
     *
     */
    public void deleteFolder(String folderPath)
    throws DirectoryServiceException;

    /**
     * Lists all the folders under folderPath. Each HashMap object contains the folder name under _FOLDER_NAME as well
     * as the rest of the folder attributes. _IS_COMPLEX is true if it's a complex folder.
     *
     * @param folderPath
     *            the folder to list
     *
     * @return a list of HashMap objects - one for each listed folder
     *
     * @exception DirectoryServiceException
     *                if there is a problem accessing the underlying storage or the folder doesn't exit
     *
     */
    public HashMap[] listFolders(String folderPath)
    throws DirectoryServiceException;

    /**
     * Lists all the folders in the DS including the root. Each HashMap object contains the folder name under
     * _FOLDER_NAME as well as the rest of the folder attributes. _IS_COMPLEX is true if it's a complex folder.
     *
     * @return a list of HashMap objects - one for each listed folder
     *
     * @exception DirectoryServiceException
     *                if there is a problem accessing the underlying storage
     *
     */
    public HashMap[] listAllFolders()
    throws DirectoryServiceException;

    /**
     * Lists all the folders and elements under path. Each HashMap object contains the folder or element attributes with
     * _FOLDER_NAME String for folders and _ELEMENT_IDENTITY IElementIdentity for elements. _IS_COMPLEX is true if it's
     * a complex folder.
     *
     * @param path
     *            the folder to list
     *
     * @return a list of HashMap objects - one for each listed folder or element
     *
     * @exception DirectoryServiceException
     *                if there is a problem accessing the underlying storage or the path doesn't exit
     *
     */
    public HashMap[] listFSAll(String path)
    throws DirectoryServiceException;
    
    /**
     * Lists the folders and elements under path. Each HashMap object in the ArrayList contains the folder or element attributes with
     * _FOLDER_NAME String for folders and _ELEMENT_IDENTITY IElementIdentity for elements. _IS_COMPLEX is true if it's
     * a complex folder. Elements to be returned are filtered by name ending in the given extension.
     * If the user making the request does not have read permissions on the folder, the method throws an exception
     *
     * @param path
     *            the folder to list
     * @param getFolders If true, folders are returned in the ArrayList
     * @param extension Used a a filter to return elements with a logical name
     * ending in the extension. A null value means no filtering. Folder names are
     * not filtered.
     * @return a list of HashMap objects - one for each listed folder or element
     *
     * @exception DirectoryServiceException
     *                if there is a problem accessing the underlying storage or the path doesn't exit
     *
     */
    public HashMap[] listFSAll(String path, boolean getFolders, String extension)
    throws DirectoryServiceException;
    
    /**
     * Lists the folders and elements under path. Each HashMap object in the ArrayList contains the folder or element attributes with
     * _FOLDER_NAME String for folders and _ELEMENT_IDENTITY IElementIdentity for elements. _IS_COMPLEX is true if it's
     * a complex folder. Elements to be returned are filtered by name ending in the given extension.
     * Subfolders are traversed as well to return elements and folders.
     * If the user making the request does not have read permissions on a folder, the method 
     * bypasses the traversal of that folder. Because this method has been initially
     * implemented with SonicFSFileSystem in mind, it does not traverse folders which
     * are both folders and configurations, like broker and cluster "folders" 
     * ie. /Brokers/Broker1. If a different use is required, we could add a flag to
     * control that behavior.
     *
     * @param path
     *            the folder to list
     * @param getFolders If true, folders are returned in the ArrayList
     * @param getElements If true, names of elements are returned in the ArrayList
     * @param extension Used a a filter to return elements with a logical name
     * ending in the extension. A null value means no filtering. Folder names are
     * not filtered.
     * @return a list of HashMap objects - one for each listed folder or element
     *
     * @exception DirectoryServiceException
     *                if there is a problem accessing the underlying storage or the path doesn't exit
     *
     */
    public ArrayList<HashMap> recursiveList(String path, boolean getFolders, boolean getElements, String extension)
    throws DirectoryServiceException;

    /**
     * Lists all the elements under folderPath. Each HashMap object contains the IElementIdentity under
     * _ELEMENT_IDENTITY as well as the rest of the element attributes.
     *
     * @param folderPath
     *            the folder to list
     *
     * @return a list of HashMap objects - one for each listed element
     * @exception DirectoryServiceException
     *                if there is a problem accessing the underlying storage or the folder doesn't exit
     *
     */
    public HashMap[] listFSElements(String folderPath)
    throws DirectoryServiceException;

    /**
     * Update the attributes of a folder or an element. Note that only String attributes are supported. The
     * _ELEMENT_IDENTITY _FOLDER_NAME and _IS_COMPLEX attributes are ignored.
     *
     * @param path
     *            the name of the folder or the element
     * @param attributes
     *            the new attributes
     *
     * @exception DirectoryServiceException
     *                if there is a problem accessing the underlying storage or the folder doesn't exit
     *
     */
    public void setMetaAttributes(String path, HashMap attributes)
    throws DirectoryServiceException;

    /**
     * Get the attributes of a folder or an element. In addition the following attributes are added: _FOLDER_NAME String
     * for folders, _ELEMENT_IDENTITY IElementIdentity for elements, _IS_COMPLEX is true for complex folders.
     *
     * @param path
     *            the name of the folder or the element
     *
     * @exception DirectoryServiceException
     *                if there is a problem accessing the underlying storage
     *
     */
    public HashMap getMetaAttributes(String path)
    throws DirectoryServiceException;

    /** ******************************* ************************************** */
    /** ************************** Storage & Back reference Hints ************ */
    /** ********************************************************************** */

    /**
     * The list of types is typically set once through seeding or through this setBackReferenceTypes API call.
     * setBackReferenceTypes can be called several times to add more types. Back references are not created when
     * setBackReferenceTypes is called, so it should be called before any elements of these types are created (but the
     * DS doesn't enforce that). The only way to remove types from the list is through the resetBackReferences method
     * which removes all the types from the list and deletes all the back references.
     *
     * @param typeList
     *            the back reference definition list to add
     *
     * @exception DirectoryServiceException
     *                if there is a problem accessing the underlying storage.
     *
     */
    public void setBackReferenceTypes(String[] typeList)
    throws DirectoryServiceException;

    /**
     *
     * Returns the back reference definition list
     *
     * @return the back reference definition list
     *
     * @exception DirectoryServiceException
     *                if there is a problem accessing the underlying storage
     *
     */
    public String[] getBackReferenceTypes()
    throws DirectoryServiceException;

    /**
     *
     * Used to delete the type definition list and all the back references data.
     *
     * @exception DirectoryServiceException
     *                if there is a problem accessing the underlying storage.
     *
     */
    public void resetBackReferences()
    throws DirectoryServiceException;

    /**
     *
     * Rebuild all the back references according to the definition list.
     *
     * @exception DirectoryServiceException
     *                if there is a problem accessing the underlying storage.
     *
     */
    public void rebuildBackReferences()
    throws DirectoryServiceException;

    /**
     * Tells the DS where to store elements of that type. The directory must be explicitly created before the elements
     * are created. The hints are stored in the DS like any other data and can be imported/exported/seeded as XML
     * documents.
     *
     * @param elementType
     *            the type returned by IElementIdentity.getType()
     * @param directoryPath
     *            the directory under which element of this type should be stored
     *
     * @return the element object (or null if the element is not in the directory)
     *
     * @exception DirectoryServiceException
     *                if there is a problem accessing the underlying storage
     *
     */
    public void setStorageHint(String elementType, String directoryPath)
    throws DirectoryServiceException;

    /**
     * Tells the DS where to store elements of that type and that postfix. The directory must be explicitly created
     * before the elements are created. The hints are stored in the DS like any other data and can be
     * imported/exported/seeded as XML documents.
     *
     * Example setStorageHint("XQ_TYPE", "cbr", "/xq/cbrs");
     *
     * @param elementType
     *            the type returned by IElementIdentity.getType()
     * @param postfix
     *            the name postfix (e.g. xxx.cbr) of the element
     * @param directoryPath
     *            the directory under which element of this type should be stored
     *
     * @return the element object (or null if the element is not in the directory)
     *
     * @exception DirectoryServiceException
     *                if there is a problem accessing the underlying storage
     *
     */
    public void setStorageHint(String elementType, String postfix, String directoryPath)
    throws DirectoryServiceException;

    /**
     * Tells the DS where to store elements of that type. First creates a folder and then places the element under it.
     * Example: setComplexStorageHint("MQ_AUTHORIZATION_POLICY", "/policies"); setElement(<new element of type
     * MQ_AUTHORIZATION_POLICY called "NorthAmerica/Bedford/BedfordPolicies>);
     *
     * The DS will create the /NorthAmerica/Bedford/BedfordPolicies folder. Store the element under
     * /NorthAmerica/Bedford/BedfordPolicies/BedfordPolicies
     *
     * The DS, behind the scenes create the /policies/UID-123454555 directory and link it to
     * /NorthAmerica/Bedford/BedfordPolicies and create the /policies/UID-123454555/UID-123454555 element and link it to
     * /NorthAmerica/Bedford/BedfordPolicies/BedfordPolicies
     *
     * NOTE: Elements or folders under /NorthAmerica/Bedford/BedfordPolicies CANNOT BE RENAMED!
     * /NorthAmerica/Bedford/BedfordPolicies CAN BE RENAMED. For example, /NorthAmerica/Bedford/BedfordPolicies can be
     * renamed to /NorthAmerica/US/MA/Bedford/BedfordBillericaPolicies
     *
     *
     * The directory must be explicitly created before the elements are created. The hints are stored in the DS like any
     * other data and can be imported/exported/seeded as XML documents.
     *
     * @param elementType
     *            the type returned by IElementIdentity.getType()
     * @param directoryPath
     *            the directory under which element of this type should be stored
     *
     * @return the element object (or null if the element is not in the directory)
     *
     * @exception DirectoryServiceException
     *                if there is a problem accessing the underlying storage
     *
     */
    public void setComplexStorageHint(String elementType, String directoryPath)
    throws DirectoryServiceException;

    /** ******************************* ************************************** */
    /** ****************** Element Manipulation Methods ********************** */
    /** ******************************* ************************************** */
    /**
     * Returns the elements that matches the FROM clause and the WHERE clause of the query. Only attributes selected by
     * the SELECT clause are fetched. All the attributes are fetched if the query does not contain a SELECT clause.
     *
     * NOTE: FromDirectory from clause is not supported
     *
     * @param query
     *            with a SELECT clause (optional), a FROM clause (mandatory) and a WHERE clause (optional)
     * @param forUpdate
     *            true if the caller might update this element. forUpdate must be false if the query contains a WHERE
     *            clause.
     * @param getSubclassingDelta
     *            true if the caller wants the subclassing delta to be attached to the elements in cases where
     *            elementName is a sub-classed element (use IDirElement.getSubclassingDelta to get the delta from the
     *            element).
     *
     * @return the elements that match the query (can be an empty list)
     *
     * @exception DirectoryServiceException
     *                if there is a problem accessing the underlying storage or the parametrs are invalid
     *
     * @see com.sonicsw.mf.common.config.query.Query
     *
     */
    public IDirElement[] getFSElements(Query query, boolean forUpdate, boolean getSubclassingDelta)
    throws DirectoryServiceException;

    /**
     * Returns the elements that matches the FROM clause and the WHERE clause of the query. Only attributes selected by
     * the SELECT clause are fetched. All the attributes are fetched if the query does not contain a SELECT clause.
     *
     * NOTE: FromDirectory from clause is not supported
     *
     * @param query
     *            with a SELECT clause (optional), a FROM clause (mandatory) and a WHERE clause (optional)
     * @param forUpdate
     *            true if the caller might update this element. forUpdate must be false if the query contains a WHERE
     *            clause.
     *
     * @return the elements that match the query (can be an empty list)
     *
     * @exception DirectoryServiceException
     *                if there is a problem accessing the underlying storage or the parametrs are invalid
     *
     * @see com.sonicsw.mf.common.config.query.Query
     *
     */
    public IDirElement[] getFSElements(Query query, boolean forUpdate)
    throws DirectoryServiceException;

    /**
     * Returns the elements that matches the FROM clause and the WHERE clause of the query. Only attributes selected by
     * the SELECT clause are fetched. All the attributes are fetched if the query does not contain a SELECT clause. If 
     * the query container an OrderedBy clause, the elements will come back sorted. If QueryBatch is non-null,
     * a subset of the results will be returned.
     *
     * NOTE: FromDirectory from clause is not supported
     *
     * @param query
     *            with a SELECT clause (optional), a FROM clause (mandatory) and a WHERE clause (optional)
     * @param forUpdate
     *            true if the caller might update this element. forUpdate must be false if the query contains a WHERE
     *            clause.
     * @param getSubclassingDelta
     *            true if the caller wants the subclassing delta to be attached to the elements in cases where
     *            elementName is a sub-classed element (use IDirElement.getSubclassingDelta to get the delta from the
     *            element).
     *
     * @return QueryBatch object which contains the batch of elements.
     *
     * @exception DirectoryServiceException
     *                if there is a problem accessing the underlying storage or the parameters are invalid
     *
     * @see com.sonicsw.mf.common.config.query.Query
     *
     */
    public QueryBatch getFSElements(Query query, boolean forUpdate, boolean getSubclassingDelta, QueryBatch batch)
    throws DirectoryServiceException;
    
    /**
     * Returns an element by name.
     *
     * @param elementPath
     *            the name of the element
     * @param forUpdate 
     *            true if the caller might update this element
     *
     * @return the element object (or null if the element is not in the folder)
     *
     * @exception DirectoryServiceException
     *                if there is a problem accessing the underlying storage
     *
     */
    public IDirElement getFSElement(String elementPath, boolean forUpdate)
    throws DirectoryServiceException;

    /**
     * Returns the list of attributes that reference an element.
     *
     * @see com.sonicsw.mf.common.config.query.AttributeName
     *
     * Use String AttributeName.getElementName to get the referencing element Use int AttributeName.getComponentCount to
     * get the depth of the referencing attribute Use Object AttributeName.getComponent to get n-th component of the
     * referencing attribute - String is returned if it's an attribute and Integer if it's a list item.
     *
     * @param elementPath
     *            the name of the element
     *
     * @return the list of attributes that reference elementPath. The list is empty if there are no references.
     *
     * @exception DirectoryServiceException
     *                if there is a problem accessing the underlying storage
     *
     */
    public AttributeName[] getReferences(String elementPath)
    throws DirectoryServiceException;

    /**
     * Returns all the elements under a folder.
     *
     * @param folderPath
     *            the name of the folder
     * @param forUpdate
     *            true if the caller might update these elements
     *
     * @return all the elements under the folder
     *
     * @exception DirectoryServiceException
     *                if there is a problem accessing the underlying storage
     *
     */
    public IDirElement[] getFSElements(String folderPath, boolean forUpdate)
    throws DirectoryServiceException;

    /**
     * Returns an element by name.
     *
     * @param elementPath
     *            the name of the element
     * @param forUpdate
     *            true if the caller might update this element
     * @param getSubclassingDelta
     *            true if the caller wants the subclassing delta to be attached to the element in cases where
     *            elementPath is a sub-classed element (use IDirElement.getSubclassingDelta to get the delta from the
     *            element).
     *
     * @return the element object (or null if the element is not in the folder)
     *
     * @exception DirectoryServiceException
     *                if there is a problem accessing the underlying storage
     *
     */
    public IDirElement getFSElement(String elementPath, boolean forUpdate, boolean getSubclassingDelta)
    throws DirectoryServiceException;

    /**
     * Returns an identity of an element
     *
     * @param name
     *            the name of the element
     *
     * @return the identity (or null if there is no element with this name)
     *
     * @exception DirectoryServiceException
     *                if there is a problem accessing the underlying storage
     *
     */
    public IElementIdentity getFSIdentity(String elementPath)
    throws DirectoryServiceException;

    /**
     * Deletes a specific element.
     *
     * @param elementPath
     *            the name of the element
     *
     * @exception DirectoryServiceException
     *                if there is a problem accessing the underlying storage.
     * @exception VersionOutofSyncException
     *                if there is a conflict with a previous update by another application
     *
     * @return the identity of the deleted element or null if it doesn't exist.
     *
     */
    public IElementIdentity deleteFSElement(String elementPath)
    throws DirectoryServiceException, VersionOutofSyncException;

    /**
     * Creates a new element. VersionOutofSyncException is thrown if the element already exists.
     *
     * @param element
     *            the new element.
     *
     * @return A token to be passed to the and IDirElement.getNextVersion methods
     *
     * @exception VersionOutofSyncException
     *                if the element already exists.
     * @exception DirectoryServiceException
     *                if there is a problem accessing the underlying storage or the folder does not exist
     *
     */
    public INextVersionToken createFSElement(IDirElement element)
    throws DirectoryServiceException, VersionOutofSyncException;

    /**
     * Applies a modification to a an element . VersionOutofSyncException is thrown if the element does not exist or if
     * the version of 'element' is not equal to the version of the element in the DS.
     *
     * @param element
     *            the delta element to be applied to the DS.
     *
     * @return A token to be passed to the and IDirElement.getNextVersion methods
     *
     * @exception VersionOutofSyncException
     *                if there is a conflict in updating the element of a previous update by another application
     * @exception DirectoryServiceException
     *                if there is a problem accessing the underlying storage or the folder does not exist
     *
     */
    public INextVersionToken updateFSElement(IDeltaDirElement element)
    throws DirectoryServiceException, VersionOutofSyncException;

    /**
     * Clone an element to create a new element with a new name identical in content to the original. Super elements
     * cannot be cloned.
     *
     * @param elementPath
     *            the name of the element to be cloned
     * @param newElementPath
     *            the name of the new element
     *
     * @return the new element
     *
     * @exception DirectoryServiceException
     *                if there is a problem accessing the underlying storage or the element does not exist or the
     *                element is a super element or the folder for the new element doesn't exist
     * @exception VersionOutofSyncException
     *                if there is a conflict in updating the element because of a previous update by another application
     *
     */
    public IDirElement cloneFSElement(String elementPath, String newElementPath)
    throws DirectoryServiceException, VersionOutofSyncException;

    /**
     * Clone an element to create a new element with a new name and some modifications relative to the original content.
     * Super elements cannot be cloned.
     *
     * @param delta
     *            the delta returned from the IDirElement.doneUpdate() applied on the original element
     * @param newElementPath
     *            the name of the new element
     *
     * @return the new element
     *
     * @exception DirectoryServiceException
     *                if there is a problem accessing the underlying storage or the element does not exist or the
     *                element is a super element or the folder for the new element doesn't exist
     * @exception VersionOutofSyncException
     *                if there is a conflict in applying the delta because of a previous update by another application
     *
     */
    public IDirElement cloneFSElement(IBasicElement delta, String newElementPath)
    throws DirectoryServiceException, VersionOutofSyncException;

    /**
     * Clone an element to create a new element with a new name identical in content to the original. Super elements
     * cannot be cloned.
     *
     * @param elementPath
     *            the name of the element to be cloned
     * @param newElementPath
     *            the name of the new element
     * @param createTemplate
     *            make the clone a template (if 'true')
     *
     * @return the new element
     *
     * @exception DirectoryServiceException
     *                if there is a problem accessing the underlying storage or the element does not exist or the
     *                element is a super element or the folder for the new element doesn't exist
     * @exception VersionOutofSyncException
     *                if there is a conflict in updating the element because of a previous update by another application
     *
     */
    public IDirElement cloneFSElement(String elementPath, String newElementPath, boolean createTemplate)
    throws DirectoryServiceException, VersionOutofSyncException;

    /**
     * Clone an element to create a new element with a new name and some modifications relative to the original content.
     * Super elements cannot be cloned.
     *
     * @param delta
     *            the delta returned from the IDirElement.doneUpdate() applied on the original element
     * @param newElementPath
     *            the name of the new element
     * @param createTemplate
     *            make the clone a template (if 'true')
     *
     * @return the new element
     *
     * @exception DirectoryServiceException
     *                if there is a problem accessing the underlying storage or the element does not exist or the
     *                element is a super element or the folder for the new element doesn't exist
     * @exception VersionOutofSyncException
     *                if there is a conflict in applying the delta because of a previous update by another application
     *
     */
    public IDirElement cloneFSElement(IBasicElement delta, String newElementPath, boolean createTemplate)
    throws DirectoryServiceException, VersionOutofSyncException;

    /**
     * Stores a BLOB in the DS storage. The BLOB data is described by the element (the element's content is determined
     * by the application). The BLOB is stored in the file-system under <domain-name>/blobs with a name identical to the
     * element's full name. For example, if the element's name is '/jars/broker' then the BLOB is stored under '<domain-name>/blobs//jars/broker'.
     * attachBlob() can be called multiple times to replace the attachment. The maximum size of the BLOB is 10 MB.
     *
     * @param element
     *            the element the BLOB is attached to. Element can be a new IDirElement or IDeltaDirElement object
     *            returned from IDirElement.doneUpdate()
     * @param blobStream
     *            the stream of bytes that makes the BLOB
     *
     * @exception DirectoryServiceException
     *                if there is a problem accessing the underlying storage
     * @exception VersionOutofSyncException
     *                if there is a conflict in applying the delta because of a previous update by another application
     *
     */
    public void attachFSBlob(IBasicElement element, java.io.InputStream blobStream)
    throws DirectoryServiceException, VersionOutofSyncException;

    /**
     * Stores a BLOB in the DS storage. The BLOB data is described by the element (the element's content is determined
     * by the application). The BLOB is stored in the file-system under <domain-name>/blobs with a name identical to the
     * element's full name. For example, if the element's name is '/jars/broker' then the BLOB is stored under '<domain-name>/blobs//jars/broker'.
     * attachBlob() can be called multiple times to replace the attachment. The maximum size of the BLOB is 10 MB.
     *
     * @param element
     *            the element the BLOB is attached to. Element can be a new IDirElement or IDeltaDirElement object
     *            returned from IDirElement.doneUpdate()
     * @param blob
     *            the blob's byte array
     *
     * @exception DirectoryServiceException
     *                if there is a problem accessing the underlying storage
     * @exception VersionOutofSyncException
     *                if there is a conflict in applying the delta because of a previous update by another application
     *
     */
    public void attachFSBlob(IBasicElement element, byte[] blob)
    throws DirectoryServiceException, VersionOutofSyncException;

    /**
     * Clones a blob element and its blob.
     *
     * @param fromElementName
     *            The name of the blob element to be copied.
     * @param toElementName
     *            The name of the new blob element.
     * @return The cloned element
     * @throws DirectoryServiceException
     *             If there is a problem accessing the underlying storage
     * @throws VersionOutOfSynchException
     *             if there is a conflict in applying the delta because of a previous update by another application
     */

    public IDirElement cloneFSBlob(String fromElementName, String toElementName)
    throws DirectoryServiceException, VersionOutofSyncException;

    /**
     * Gets the BLOB from the DS storage.
     *
     * @param elementPath
     *            the name of the element the BLOB is attached to.
     * @param forUpdate
     *            true if the caller might update the element and/or the attachment
     *
     * @return an IBlob object that contains references to the element and the InputStream object
     *
     * @exception DirectoryServiceException
     *                if there is a problem accessing the underlying storage
     *
     */
    public IBlob getFSBlob(String elementPath, boolean forUpdate)
    throws DirectoryServiceException;
    
    /**
     * Return the blob for a zip file containing the files named in filenames
     * Each fileName is tested for read management permissions and only those which the 
     * current connected user can read will be returned
     * @param fileNames A list of DS file names
     * @return A blob representing the zip file created to contain all the files in fileNames
     * that the user is allowed to read. If no files are zipped - because of security,
     * for instance - the method returns null.
     * @throws DirectoryServiceException if there is a problem accessing the files
     */
    
    public IBlob getFiles(String[] fileNames) throws DirectoryServiceException;
    
    /**
     * Return the blob for a zip file containing the files obtained by traversing
     * path. If recurse is true, subfolders are traversed as well.
     * Each fileName is tested for read management permissions and only those which the 
     * current connected user can read will be returned.
     * If extension is set, only those elements/files with a name that ends in extension
     * will be returned.
     * @param path The starting point folder of the search
     * @param recurse If true, subfolders are traversed
     * @param extension If set, only those elements/files with a name that ends in the
     * extension will be returned.
     * @return A blob representing the zip file created to contain all the files in fileNames
     * that the user is allowed to read. If no files are zipped, the return value is 
     * null.
     * @throws DirectoryServiceException if there is a problem accessing the files
     */
    
    public IBlob getFiles(String path, boolean recurse, String extension) throws DirectoryServiceException;

    /**
     * Deletes the BLOB attached to the element.
     *
     * @param delta
     *            a modification delta returned from IDirElement.doneUpdate().
     *
     * @exception DirectoryServiceException
     *                if there is a problem accessing the underlying storage
     * @exception VersionOutofSyncException
     *                if there is a conflict in applying the delta because of a previous update by another application
     *
     */
    public void detachFSBlob(IDeltaDirElement delta)
    throws DirectoryServiceException, VersionOutofSyncException;

    /**
     * Subclass an element from another (super) element. The super element is fetched (for update), modified and then
     * the returned value from superElement.doneUpdateForSubclassing() is passed to the subclassElement() method. All
     * modifications to the super element are automatically reflected in the subclassed element. Attributes overwritten
     * by the subclassed element are not affected when modified at the super element. The new subclassed element cannot
     * be further subclassed.
     *
     * Packed elements cannot be sub-classed. Currently packed elements are all the elements under _MFUsers directories.
     * Examples: /security/_MFUsers, /_MFUsers and /security/domain1/_MFUsers.
     *
     * @param delta
     *            the delta returned from the IDirElement.doneUpdateForSubclassing() applied on the super element
     * @param newElementPath
     *            the name of the new element
     *
     * @return the new subclassed element
     *
     * @exception DirectoryServiceException
     *                if there is a problem accessing the underlying storage or the super element does not exist or the
     *                name of the new element is invalid or the delta is not a product of a doneUpdateForSubclassing()
     *                class.
     * @exception VersionOutofSyncException
     *                if there is a conflict in applying the delta because of a previous update by another application
     *                or a 'newElementPath' element already exists.
     *
     */
    public IDirElement subclassFSElement(IBasicElement delta, String newElementPath)
    throws DirectoryServiceException, VersionOutofSyncException;

    /**
     * Removes the binding between the super element and a subclassed element. If elementPath is already independent
     * then unSubclassElement throws an exception.
     *
     * @param elementPath
     *            name of the subclassed element
     *
     * @exception DirectoryServiceException
     *                if there is a problem accessing the underlying storage or elementPath is invalid (or not
     *                sub-classed)
     * @exception VersionOutofSyncException
     *                if there is a conflict because of a previous update by another application.
     *
     */
    public void unSubclassFSElement(String elementPath)
    throws DirectoryServiceException, VersionOutofSyncException;

    /**
     * Creates a new transaction.
     */
    public IDSTransaction createTransaction();

    /**
     * Multiple updates under a single transaction. The actions are performed in the following order: 1. All the storage
     * names for new elements are generated. 2. The actions are performed according their creation order.
     *
     * they were added to IDSTransaction.
     *
     * @param transaction
     *
     * @return A token to be passed to IDirElement.getNextVersion methods
     *
     * @exception VersionOutofSyncException
     *                if there is a conflict in updating the element because of a previous update by another application
     * @exception DirectoryServiceException
     *                if there is a problem accessing the underlying storage or the folder does not exist
     *
     */
    public INextVersionToken executeTransaction(IDSTransaction transaction)
    throws DirectoryServiceException, VersionOutofSyncException;

    /**
     * Takes the element named in elementName and makes the attributes named in attributes go back to the value in the
     * parent template element. Erases the history of modifications to the attributes in the subclassing delta. Saves
     * the modified element in the DS.
     *
     * @param elementName the name of the subclassed element to be modified @param attributes an array of attribute
     * names for the attributes which should be reverted to the template value
     *
     * @return the modified element
     *
     * @exception DirectoryServiceException, if any of the attributes named in attributes is not available in the
     * template element, if there is a problem accessing the underlying storage or the element does not exist or the
     * element is not a subclassed element
     */
    public IDirElement revertToTemplate(String elementName, AttributeName[] attributes)
    throws DirectoryServiceException;

    /** ******************************* ************************************** */
    /** ************************** Misc ************************************** */
    /** ******************************* ************************************** */

    /**
     * Returns the version of this Directory Service instance.
     *
     * @return the version of this Directory Service instance
     */
    public int getDirectoryServiceVersion();

    /**
     * Returns the domain name of this DirectoryService object
     *
     * @return String or IAttributeList or IAttributeSet
     */
    public String getDomain();

    /**
     * Translates the element's logical name to storage name.
     *
     * @param path the element's full logical path
     *
     * @return the storage name
     *
     * @exception DirectoryServiceException the path format is illegal or the path doesn't exist
     */
    public String logicalToStorage(String path)
    throws DirectoryServiceException;

    /**
     * Translates the element's storage name to logical name.
     *
     * @param name the element's storage name
     *
     * @return the logical full path
     *
     * @exception DirectoryServiceException the logical name is illegal or doesn't exist
     */
    public String storageToLogical(String storageName)
    throws DirectoryServiceException;

    /**
     * Scan the DS - excluding directory names which contain a string from the exclusion list - and reassign NO_STORAGE:<logical-name>
     * values. For example, if the exclusions list contains "/_MF" then /authentication/domains/0/_MFUSers and
     * /authentication/domains/0/_MFGroups will be excluded.
     *
     * @param exclusions List of directory names to bypass
     *
     * @exception DirectoryServiceException
     */
    public void repairReferences(String[] exclusions)
    throws DirectoryServiceException;

    /**
     * Close the DS if the DS is in this JVM (this implemnted by com.sonicsw.mf.framework.direcory.impl.DirectoryService). NOOP if the DS is remote
     *
     * @exception DirectoryServiceException
     */
    public void close()
    throws DirectoryServiceException;

    /**
     * Stop forwarding element change notifications to the MF container named by configID
     * This method is used by the configuration upgrade and is not meant to be used outside
     * of Sonic utilities - ie. not to be used by users writing mgmt applications.
     * @param configID The logical ID of the MF container
     * @param allowTypes Can be null. If not null, a list of types for which notifications
     * can still be forward to the container. For instance "MF_USER", "MF_GROUP", "MF_AUTHENTICATION_DOMAIN"
     * if the caller wants the container to receive authentication domain change information.
     * @throws DirectoryServiceException
     */
    public void suspendChangeNotifications(String configID, String[] allowTypes) throws DirectoryServiceException;
    public void resumeChangeNotifications(String configID) throws DirectoryServiceException;
    public void resumeAllChangeNotifications() throws DirectoryServiceException;
}
