package com.sonicsw.mf.common.archive;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.jar.JarFile;
import java.util.zip.ZipEntry;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;

import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.SAXException;

/**
 * Wraps the information provided by a Sonic archive's descriptor, hiding the parsing and
 * structure of the META-INF/sonic.xml file.
 */
public class SonicArchive
{
    /*
    // for testing
    public static void main(String[] args)
    {
        try
        {
            System.out.println(new SonicArchive(new File(args[0])).getName());
        } catch(Exception e) { e.printStackTrace(); }
    }
    */

    private String m_name;

    /**
     * For use by the tools to evaluate information based on the meta data described in the
     * archive's sonic.sml descriptor.
     *
     * @param archive The path to the unexpanded or expanded archive
     * @throws ParserConfigurationException
     * @throws IOException
     */
    public SonicArchive(File archive)
    throws ParserConfigurationException, SAXException, IOException
    {
        if (!archive.exists())
        {
            throw new IOException("Archive path does not exist: " + archive);
        }

        InputStream sonicXML = null;

        // is this an expanded archive?
        if (archive.isDirectory())
        {
            File descriptor = new File(archive, "META-INF/sonic.xml");
            if (!descriptor.exists())
            {
                throw new IOException("Archive descriptor does not exist: " + descriptor.toString());
            }
            sonicXML = new FileInputStream(descriptor);
        }
        else // its a jar format
        {
            JarFile jar = new JarFile(archive);
            ZipEntry descriptor = jar.getEntry("META-INF/sonic.xml");
            if (descriptor == null)
            {
                throw new IOException("Archive descriptor does not exist: META-INF/sonic.xml");
            }
            sonicXML = jar.getInputStream(descriptor);
        }

        DocumentBuilder builder = DocumentBuilderFactory.newInstance().newDocumentBuilder();
        Document document = builder.parse(sonicXML);

        NodeList nodes = document.getDocumentElement().getChildNodes();
        for (int i = nodes.getLength() - 1; i >= 0; i--)
        {
            Node node = nodes.item(i);
            if (node.getNodeType() == Node.ELEMENT_NODE)
            {
                if (((Element)node).getTagName().equals(IDescriptorTags.NAME_TAG))
                {
                    m_name = node.getFirstChild().getNodeValue();
                    if (m_name == null || m_name.length() < 1)
                    {
                        throw new IOException("Invalid " + IDescriptorTags.NAME_TAG + " name");
                    }
                }
            }
        }
    }

    public String getName()
    {
        return m_name;
    }
}