/*
 * Copyright (c) 2001 Sonic Software. All Rights Reserved.
 */

package com.sonicsw.mf.common.config;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Date;

import com.sonicsw.mf.common.config.query.AttributeName;


/**
 * The IAttributeList interface provides access to a list of basic type, IAttributeSet or IAttributeList
 * items. The items in the list don't have to be of the same type (for example, some of the
 * items can be Strings and some IAttributeSets). Items in the list can be deleted or modified.
 * New item can be inserted in the middle of the list or appended to the end. Locations in the list
 * are 0 based (the first item is in position 0).
 */
public interface IAttributeList extends ITypeCollection
{

  /**
   * @return the number of elements in the list
   */
  public int getCount();

 /**
  * Returns the attribute specified.
  *
  * @param attributeName the attribute name (could be nested)
  *
  * @return the attribute specified
  *
  */
  public Object getAttribute(AttributeName attributeName);


 /**
  * Returns an item in the list.
  *
  * @param position the location of the item in the list
  *
  * @return basic type or IAttributeList or IAttributeSet
  *
  * @exception IndexOutOfBoundsException if the position is invalid
  *
  */
  public Object getItem(int position) throws IndexOutOfBoundsException;

 /**
  * Creates and sets a new IAttributeList item in a specific position in the list.
  *
  * @param position the location of the item in the list
  *
  * @return the new IAttributeList object
  *
  * @exception IndexOutOfBoundsException if the position is invalid
  * @exception ReadOnlyException if it's a read-only list
  *
  */
  public IAttributeList setNewAttributeListItem(int position) throws IndexOutOfBoundsException, ReadOnlyException;

 /**
  * Creates and inserts a new IAttributeList item before the specified position.
  *
  * @param position the new item is inserted before this location
  *
  * @return the new IAttributeList object
  *
  * @exception IndexOutOfBoundsException if the position is invalid
  * @exception ReadOnlyException if it's a read-only list
  *
  */
  public IAttributeList insertAttributeListItem(int position) throws IndexOutOfBoundsException, ReadOnlyException;

 /**
  * Creates and appends to the end of the list a new IAttributeList item
  *
  * @return the new IAttributeList object
  *
  * @exception ReadOnlyException if it's a read-only list
  *
  */
  public IAttributeList addNewAttributeListItem() throws ReadOnlyException;

 /**
  * Creates and sets a new IAttributeSet item in a specific position in the list.
  *
  * @param position the location of the item in the list
  *
  * @return the new IAttributeSet object
  *
  * @exception IndexOutOfBoundsException if the position is invalid
  * @exception ReadOnlyException if it's a read-only list
  *
  */
  public IAttributeSet setNewAttributeSetItem(int position) throws IndexOutOfBoundsException, ReadOnlyException;

 /**
  * Creates and inserts a new IAttributeSet item before the specified position.
  *
  * @param position the new item is inserted before this location
  *
  * @return the new IAttributeSet object
  *
  * @exception IndexOutOfBoundsException if the position is invalid
  * @exception ReadOnlyException if it's a read-only list
  *
  */
  public IAttributeSet insertAttributeSetItem(int position) throws IndexOutOfBoundsException, ReadOnlyException;

 /**
  * Creates and appends to the end of the list a new IAttributeSet item
  *
  * @return the new IAttributeSet object
  *
  * @exception ReadOnlyException if it's a read-only list
  *
  */
  public IAttributeSet addNewAttributeSetItem() throws ReadOnlyException;

 /**
  * Creates and sets a new IAttributeSet item of a specific type in a specific position in the list.
  *
  * @param position the location of the item in the list
  * @param type the type of the IAttributeSet object
  *
  * @return the new IAttributeSet object
  *
  * @exception IndexOutOfBoundsException if the position is invalid
  * @exception ReadOnlyException if it's a read-only list
  * @exception AttributeSetTypeException if the type is not defined
  *
  */
  public IAttributeSet setNewAttributeSetItem(int position, IAttributeSetType type) throws IndexOutOfBoundsException, ReadOnlyException,
      AttributeSetTypeException;

 /**
  * Creates and inserts a new IAttributeSet item of a specific type before the specified position.
  *
  * @param position the new item is inserted before this location
  * @param type the type of the IAttributeSet object
  *
  * @return the new IAttributeSet object
  *
  * @exception IndexOutOfBoundsException if the position is invalid
  * @exception ReadOnlyException if it's a read-only list
  * @exception AttributeSetTypeException if the type is not defined
  *
  */
  public IAttributeSet insertAttributeSetItem(int position, IAttributeSetType type) throws IndexOutOfBoundsException, ReadOnlyException,
      AttributeSetTypeException;

 /**
  * Creates and appends to the end of the list a new IAttributeSet item of a specific type
  *
  * @param type the type of the IAttributeSet object
  *
  * @return the new IAttributeSet object
  *
  * @exception ReadOnlyException if it's a read-only list
  * @exception AttributeSetTypeException if the type is not defined
  *
  */
  public IAttributeSet addNewAttributeSetItem(IAttributeSetType type) throws ReadOnlyException, AttributeSetTypeException;


 /**
  * Deletes an item in the list.
  *
  * @param position the location of the item in the list
  *
  * @return the deleted object - basic type or IAttributeList or IAttributeSet
  *
  * @exception IndexOutOfBoundsException if the position is invalid
  * @exception ReadOnlyException if it's a read-only list
  *
  */
  public Object deleteAttributeItem(int position) throws IndexOutOfBoundsException , ReadOnlyException;

 /**
  * Returns references to all the items in the list.
  *
  * @return all the items in the list
  *
  */
  public ArrayList getItems();

 /**
  * Returns the item's meta data object.
  *
  * @param position the location of the item in the list
  *
  * @return the item's meta data object.
  *
  * @exception IndexOutOfBoundsException if the position is invalid
  *
  */
  public IAttributeMetaData getItemMetaData(int position) throws IndexOutOfBoundsException;


 /**
  * Sets a new item in a specific position in the list. Setting null is equivalent to deletion.
  *
  * @param position the location of the item in the list
  * @param value the new item
  *
  * @exception IndexOutOfBoundsException if the position is invalid
  * @exception ReadOnlyException if it's a read-only list
  *
  */
  public void setNewStringItem(int position, String value) throws IndexOutOfBoundsException, ReadOnlyException;

 /**
  * Sets a new item in a specific position in the list. Setting null is equivalent to deletion.
  *
  * @param position the location of the item in the list
  * @param value the new item
  *
  * @exception IndexOutOfBoundsException if the position is invalid
  * @exception ReadOnlyException if it's a read-only list
  *
  */
  public void setNewIntegerItem(int position, Integer value) throws IndexOutOfBoundsException, ReadOnlyException;

 /**
  * Sets a new item in a specific position in the list. Setting null is equivalent to deletion.
  *
  * @param position the location of the item in the list
  * @param value the new item
  *
  * @exception IndexOutOfBoundsException if the position is invalid
  * @exception ReadOnlyException if it's a read-only list
  *
  */
  public void setNewLongItem(int position, Long value) throws IndexOutOfBoundsException, ReadOnlyException;

 /**
  * Sets a new item in a specific position in the list. Setting null is equivalent to deletion.
  *
  * @param position the location of the item in the list
  * @param value the new item
  *
  * @exception IndexOutOfBoundsException if the position is invalid
  * @exception ReadOnlyException if it's a read-only list
  *
  */
  public void setNewDecimalItem(int position, BigDecimal value) throws IndexOutOfBoundsException, ReadOnlyException;

 /**
  * Sets a new item in a specific position in the list. Setting null is equivalent to deletion.
  *
  * @param position the location of the item in the list
  * @param value the new item
  *
  * @exception IndexOutOfBoundsException if the position is invalid
  * @exception ReadOnlyException if it's a read-only list
  *
  */
  public void setNewBytesItem(int position, byte[] value) throws IndexOutOfBoundsException, ReadOnlyException;

 /**
  * Sets a new item in a specific position in the list. Setting null is equivalent to deletion.
  *
  * @param position the location of the item in the list
  * @param value the new item
  *
  * @exception IndexOutOfBoundsException if the position is invalid
  * @exception ReadOnlyException if it's a read-only list
  *
  */
  public void setNewBooleanItem(int position, Boolean value) throws IndexOutOfBoundsException, ReadOnlyException;

 /**
  * Sets a new item in a specific position in the list. Setting null is equivalent to deletion.
  *
  * @param position the location of the item in the list
  * @param value the new item
  *
  * @exception IndexOutOfBoundsException if the position is invalid
  * @exception ReadOnlyException if it's a read-only list
  *
  */
  public void setNewReferenceItem(int position, Reference value) throws IndexOutOfBoundsException, ReadOnlyException;

 /**
  * Sets a new item in a specific position in the list. Setting null is equivalent to deletion.
  *
  * @param position the location of the item in the list
  * @param value the new item
  *
  * @exception IndexOutOfBoundsException if the position is invalid
  * @exception ReadOnlyException if it's a read-only list
  *
  */
  public void setNewDateItem(int position, Date value) throws IndexOutOfBoundsException, ReadOnlyException;



 /**
  * Inserts a new item before the specified position.  Inserting null has no effect on the list.
  *
  * @param position the new item is inserted before this location
  * @param value the new item
  *
  * @exception IndexOutOfBoundsException if the position is invalid
  * @exception ReadOnlyException if it's a read-only list
  *
  */
  public void insertStringItem(int position, String value) throws IndexOutOfBoundsException, ReadOnlyException;

 /**
  * Inserts a new item before the specified position.  Inserting null has no effect on the list.
  *
  * @param position the new item is inserted before this location
  * @param value the new item
  *
  * @exception IndexOutOfBoundsException if the position is invalid
  * @exception ReadOnlyException if it's a read-only list
  *
  */
  public void insertIntegerItem(int position, Integer value) throws IndexOutOfBoundsException, ReadOnlyException;

 /**
  * Inserts a new item before the specified position.  Inserting null has no effect on the list.
  *
  * @param position the new item is inserted before this location
  * @param value the new item
  *
  * @exception IndexOutOfBoundsException if the position is invalid
  * @exception ReadOnlyException if it's a read-only list
  *
  */
  public void insertLongItem(int position, Long value) throws IndexOutOfBoundsException, ReadOnlyException;

 /**
  * Inserts a new item before the specified position.  Inserting null has no effect on the list.
  *
  * @param position the new item is inserted before this location
  * @param value the new item
  *
  * @exception IndexOutOfBoundsException if the position is invalid
  * @exception ReadOnlyException if it's a read-only list
  *
  */
  public void insertDecimalItem(int position, BigDecimal value) throws IndexOutOfBoundsException, ReadOnlyException;

 /**
  * Inserts a new item before the specified position.  Inserting null has no effect on the list.
  *
  * @param position the new item is inserted before this location
  * @param value the new item
  *
  * @exception IndexOutOfBoundsException if the position is invalid
  * @exception ReadOnlyException if it's a read-only list
  *
  */
  public void insertBytesItem(int position, byte[] value) throws IndexOutOfBoundsException, ReadOnlyException;

 /**
  * Inserts a new item before the specified position.  Inserting null has no effect on the list.
  *
  * @param position the new item is inserted before this location
  * @param value the new item
  *
  * @exception IndexOutOfBoundsException if the position is invalid
  * @exception ReadOnlyException if it's a read-only list
  *
  */
  public void insertBooleanItem(int position, Boolean value) throws IndexOutOfBoundsException, ReadOnlyException;

 /**
  * Inserts a new item before the specified position.  Inserting null has no effect on the list.
  *
  * @param position the new item is inserted before this location
  * @param value the new item
  *
  * @exception IndexOutOfBoundsException if the position is invalid
  * @exception ReadOnlyException if it's a read-only list
  *
  */
  public void insertReferenceItem(int position, Reference value) throws IndexOutOfBoundsException, ReadOnlyException;

 /**
  * Inserts a new item before the specified position.  Inserting null has no effect on the list.
  *
  * @param position the new item is inserted before this location
  * @param value the new item
  *
  * @exception IndexOutOfBoundsException if the position is invalid
  * @exception ReadOnlyException if it's a read-only list
  *
  */
  public void insertDateItem(int position, Date value) throws IndexOutOfBoundsException, ReadOnlyException;



 /**
  * Appends a new basic item to the end of the list. Adding null has no effect on the list.
  *
  * @param value the new appended item. Supported: String, Integer, Long, Date, BigDecimal, Reference, byte array and Boolean.
  *
  * @exception ReadOnlyException if it's a read-only list
  *
  */
  public void addObjectItem(Object value) throws ReadOnlyException;

 /**
  * Appends a new item to the end of the list. Adding null has no effect on the list.
  *
  * @param value the new appended item.
  *
  * @exception ReadOnlyException if it's a read-only list
  *
  */
  public void addStringItem(String value) throws ReadOnlyException;

 /**
  * Appends a new item to the end of the list. Adding null has no effect on the list.
  *
  * @param value the new appended item
  *
  * @exception ReadOnlyException if it's a read-only list
  *
  */
  public void addIntegerItem(Integer value) throws ReadOnlyException;

 /**
  * Appends a new item to the end of the list. Adding null has no effect on the list.
  *
  * @param value the new appended item
  *
  * @exception ReadOnlyException if it's a read-only list
  *
  */
  public void addLongItem(Long value) throws ReadOnlyException;

 /**
  * Appends a new item to the end of the list. Adding null has no effect on the list.
  *
  * @param value the new appended item
  *
  * @exception ReadOnlyException if it's a read-only list
  *
  */
  public void addDecimalItem(BigDecimal value) throws ReadOnlyException;

 /**
  * Appends a new item to the end of the list. Adding null has no effect on the list.
  *
  * @param value the new appended item
  *
  * @exception ReadOnlyException if it's a read-only list
  *
  */
  public void addBytesItem(byte[] value) throws ReadOnlyException;

 /**
  * Appends a new item to the end of the list. Adding null has no effect on the list.
  *
  * @param value the new appended item
  *
  * @exception ReadOnlyException if it's a read-only list
  *
  */
  public void addBooleanItem(Boolean value) throws ReadOnlyException;

 /**
  * Appends a new item to the end of the list. Adding null has no effect on the list.
  *
  * @param value the new appended item
  *
  * @exception ReadOnlyException if it's a read-only list
  *
  */
  public void addReferenceItem(Reference value) throws ReadOnlyException;

 /**
  * Appends a new item to the end of the list. Adding null has no effect on the list.
  *
  * @param value the new appended item
  *
  * @exception ReadOnlyException if it's a read-only list
  *
  */
  public void addDateItem(Date value) throws ReadOnlyException;

 /**
  * Register a handler for modifications in this attribute list
  *
  * @param context the context of the container which retrieves elements from the DS (such as IComponenetContext at the MF container)
  * @param handler the modifications handler
  *
  */
  public void registerAttributeChangeHandler(Object context, IAttributeChangeHandler handler);

 /**
  * Unregister the modifications of this list (if a handler was registered before)
  * @param context the context of the container which retrieves elements from the DS (such as IComponenetContext at the MF container)
  *
  */
  public void unregisterAttributeChangeHandler(Object context);

 /**
  * Register a handler for modifications in a specific item of this attribute list
  *
  * @param context the context of the container which retrieves elements from the DS (such as IComponenetContext at the MF container)
  * @param position the position of the monitored list item
  * @param handler the modifications handler
  *
  */
  public void registerAttributeChangeHandler(Object context, int position, IAttributeChangeHandler handler);

 /**
  * Unregister a handler for modifications in a specific item of this attribute list (if a handler was registered before)
  *
  * @param context the context of the container which retrieves elements from the DS (such as IComponenetContext at the MF container)
  * @param position the position of the monitored list item
  *
  */
  public void unregisterAttributeChangeHandler(Object context, int position);
}
