/*
 * Copyright (c) 2001 Sonic Software. All Rights Reserved.
 */

package com.sonicsw.mf.common.config;

import java.io.Serializable;
import java.math.BigDecimal;
import java.util.Date;
import java.util.HashMap;

import com.sonicsw.mf.common.config.query.AttributeName;


/**
 * The IAttributeSet interface provides access to a set of {name,value} pairs. The values can
 * be of type basic type, IAttributeSet or IAttributeList. Attributes can be added, deleted or modified.
 * A null value is equivalent to a non defined attribute. Attribute names cannot contain the '\' and ':' characters.
 */

public interface IAttributeSet extends ITypeCollection, Serializable
{
 /**
  * Returns the value of this attribute.
  *
  * @param attributeName the name of the attribute (cannot be nested)
  *
  * @return basic type or IAttributeList or IAttributeSet (or null if the attribute is not defined)
  *
  */
  public Object getAttribute(String attributeName);

 /**
  * Returns the attribute specified.
  *
  * @param attributeName the attribute name (could be nested).
  *
  * @return the attribute specified
  *
  */
  public Object getAttribute(AttributeName attributeName);

 /**
  * Returns the attribute's meta data object.
  *
  * @param attributeName the attribute name (cannot be nested)
  *
  * @return the attribute's meta data object (or null if the attribute does not exist)
  *
  */
  public IAttributeMetaData getAttributeMetaData(String attributeName);

 /**
  * Returns the list of deleted attributes (relative to the template) in this attribute set in this subclassed element.
  * getDeletedAttributesInThisSubclassed can be called only if this element is fresh from the DS - not new or modified.
  *
  * @return the list of deleted attributes relative to the template. null is returned if the entire attribute set
  *         in the subclassed element is replaced (this attribute set is under a modified attribute list, for example).
  *         An Error is thrown if the element is modified or new. An empty list is returned if the element is not subclassed.
  *
  */
  public String[] getDeletedAttributesInThisSubclassed();


 /**
  * Creates and sets a new IAttributeList object for the given attribute
  *
  * @param attributeName the name of the attribute
  *
  * @return the new IAttributeList object
  *
  * @exception AttributeSetTypeException if this is a typed IAttributeSet and attributeName is not defined
  * @exception ReadOnlyException if it's a read-only set
  * @exception ConfigException illegal parameters (such as a null attributeName)
  *
  */
  public IAttributeList createAttributeList(String attributeName) throws ReadOnlyException, AttributeSetTypeException, ConfigException;

 /**
  * Creates and sets a new IAttributeSet object for the given attribute
  *
  * @param attributeName the name of the attribute
  *
  * @return the new IAttributeSet object
  *
  * @exception AttributeSetTypeException is it is a typed IAttributeSet and attributeName is not defined
  * @exception ReadOnlyException if it's a read-only set
  * @exception ConfigException illegal parameters (such as a null attributeName)
  *
  */
  public IAttributeSet createAttributeSet(String attributeName) throws ReadOnlyException, AttributeSetTypeException, ConfigException;


 /**
  * Creates and sets a new typed IAttributeSet object for the given attribute
  *
  * @param attributeName the name of the attribute
  * @param type the type of the IAttributeSet object
  *
  * @return the new IAttributeSet object
  *
  * @exception AttributeSetTypeException is it is a typed IAttributeSet and attributeName is not defined
  * @exception ReadOnlyException if it's a read-only set
  * @exception ConfigException illegal parameters (such as a null attributeName)
  *
  */
  public IAttributeSet createAttributeSet(String attributeName, IAttributeSetType type) throws ReadOnlyException, AttributeSetTypeException, ConfigException;

 /**
  * Deletes a {attribute,value} pair. It is equivalent to setting the attribute to null. Note that an attribute
  * in a typed IAttributeSet is allowed to be deleted (since it is allowed to have a null value).
  *
  * @param attributeName the name of the attribute
  *
  * @return the deleted object - basic type or IAttributeList or IAttributeSet or null
  *
  * @exception ReadOnlyException if it's a read-only set
  * @exception ConfigException illegal parameters (such as a null attributeName)
  *
  */
  public Object deleteAttribute(String attributeName) throws ReadOnlyException, ConfigException;

 /**
  * Returns references to all the {attribute,value} pairs in a HashMap object
  *
  * @return all the items in the set
  *
  */
  public HashMap getAttributes();

 /**
  * Sets a new basic value for the given attribute.
  *
  * @param attributeName the name of the attribute
  * @param value the new value of the attribute. Supported: String, Integer, Long, Date, BigDecimal, Reference, byte array, Boolean and null.
  *
  * @exception AttributeSetTypeException if this IAttributeSet is typed and attributeName is not defined
  * @exception ReadOnlyException if it's a read-only set
  * @exception ConfigException illegal parameters (such as a null attributeName)
  *
  */
  public void setObjectAttribute(String attributeName, Object value) throws ReadOnlyException, AttributeSetTypeException, ConfigException;

 /**
  * Sets a new value for the given attribute
  *
  * @param attributeName the name of the attribute
  * @param value the new value of the attribute
  *
  * @exception AttributeSetTypeException if this IAttributeSet is typed and attributeName is not defined
  * @exception ReadOnlyException if it's a read-only set
  * @exception ConfigException illegal parameters (such as a null attributeName)
  *
  */
  public void setIntegerAttribute(String attributeName, Integer value) throws ReadOnlyException, AttributeSetTypeException, ConfigException;

 /**
  * Sets a new value for the given attribute
  *
  * @param attributeName the name of the attribute
  * @param value the new value of the attribute
  *
  * @exception AttributeSetTypeException if this IAttributeSet is typed and attributeName is not defined
  * @exception ReadOnlyException if it's a read-only set
  * @exception ConfigException illegal parameters (such as a null attributeName)
  *
  */
  public void setLongAttribute(String attributeName, Long value) throws ReadOnlyException, AttributeSetTypeException, ConfigException;

 /**
  * Sets a new value for the given attribute
  *
  * @param attributeName the name of the attribute
  * @param value the new value of the attribute
  *
  * @exception AttributeSetTypeException if this IAttributeSet is typed and attributeName is not defined
  * @exception ReadOnlyException if it's a read-only set
  * @exception ConfigException illegal parameters (such as a null attributeName)
  *
  */
  public void setStringAttribute(String attributeName, String value) throws ReadOnlyException, AttributeSetTypeException, ConfigException;

 /**
  * Sets a new value for the given attribute
  *
  * @param attributeName the name of the attribute
  * @param value the new value of the attribute
  *
  * @exception AttributeSetTypeException if this IAttributeSet is typed and attributeName is not defined
  * @exception ReadOnlyException if it's a read-only set
  * @exception ConfigException illegal parameters (such as a null attributeName)
  *
  */
  public void setDecimalAttribute(String attributeName, BigDecimal value) throws ReadOnlyException, AttributeSetTypeException, ConfigException;

 /**
  * Sets a new value for the given attribute
  *
  * @param attributeName the name of the attribute
  * @param value the new value of the attribute
  *
  * @exception AttributeSetTypeException if this IAttributeSet is typed and attributeName is not defined
  * @exception ReadOnlyException if it's a read-only set
  * @exception ConfigException illegal parameters (such as a null attributeName)
  *
  */
  public void setBytesAttribute(String attributeName, byte[] value) throws ReadOnlyException, AttributeSetTypeException, ConfigException;

 /**
  * Sets a new value for the given attribute
  *
  * @param attributeName the name of the attribute
  * @param value the new value of the attribute
  *
  * @exception AttributeSetTypeException if this IAttributeSet is typed and attributeName is not defined
  * @exception ReadOnlyException if it's a read-only set
  * @exception ConfigException illegal parameters (such as a null attributeName)
  *
  */
  public void setBooleanAttribute(String attributeName, Boolean value) throws ReadOnlyException, AttributeSetTypeException, ConfigException;

 /**
  * Sets a new value for the given attribute
  *
  * @param attributeName the name of the attribute
  * @param value the new value of the attribute
  *
  * @exception AttributeSetTypeException if this IAttributeSet is typed and attributeName is not defined
  * @exception ReadOnlyException if it's a read-only set
  * @exception ConfigException illegal parameters (such as a null attributeName)
  *
  */
  public void setDateAttribute(String attributeName, Date value) throws ReadOnlyException, AttributeSetTypeException, ConfigException;


 /**
  * Sets a new value for the given attribute
  *
  * @param attributeName the name of the attribute
  * @param value the new value of the attribute
  *
  * @exception AttributeSetTypeException if this IAttributeSet is typed and attributeName is not defined
  * @exception ReadOnlyException if it's a read-only set
  * @exception ConfigException illegal parameters (such as a null attributeName)
  *
  */
  public void setReferenceAttribute(String attributeName, Reference value) throws ReadOnlyException, AttributeSetTypeException, ConfigException;

 /**
  * Returns the name of the AttributeSetType of this attribute set object.
  *
  * @return the type name of this attribute set object (or null if it's not a typed attribute set)
  *
  */
  public String getTypeName();

 /**
  * Register a handler for modifications in this attribute set
  *
  * @param context the context of the container which retrieves elements from the DS (such as IComponenetContext at the MF container)
  * @param handler the modifications handler
  *
  */
  public void registerAttributeChangeHandler(Object context, IAttributeChangeHandler handler);

 /**
  * Unregister the modifications of this set (if a handler was registered before)
  *
  * @param context the context of the container which retrieves elements from the DS (such as IComponenetContext at the MF container)
  */
  public void unregisterAttributeChangeHandler(Object context);

 /**
  * Register a handler for modifications in a specific attribute of this attribute set
  *
  * @param context the context of the container which retrieves elements from the DS (such as IComponenetContext at the MF container)
  * @param attributeName the name of the monitored attribute
  * @param handler the modifications handler
  *
  */
  public void registerAttributeChangeHandler(Object context, String attributeName, IAttributeChangeHandler handler);

 /**
  * Unregister a handler for modifications in a specific attribute of this attribute set (if a handler was registered before)
  *
  * @param context the context of the container which retrieves elements from the DS (such as IComponenetContext at the MF container)
  * @param attributeName the name of the monitored attribute
  *
  */
  public void unregisterAttributeChangeHandler(Object context, String attributeName);

 /**
  * Mark an attribute as deleted. This method doesn't change the state of the element locally or in the DS. It is meaningful
  * only to a sparse merge operation performed by MergeUtil. The sparse merge, in addition to applying the modifications from the
  * source element to the destination element, will also delete from the destination element attributes marked for deletion.
  *
  * @param attributeName the name of the attribute marked for deletion
  *
  */
  public void mergeUtilHint_markAttributeDeleted(String attributeName);

 /**
  * Return the list of attributes marked for deletion by mergeUtilHint_markAttributeDeleted calls.
  *
  * @return The list of attributes marked for deletion
  *
  */
  public String[] mergeUtilHint_getAttributesMarkedDeleted();
}

