/*
 * Copyright (c) 2001 Sonic Software. All Rights Reserved.
 */

package com.sonicsw.mf.common.config;

import com.sonicsw.mf.common.config.query.Query;
import com.sonicsw.mf.common.dirconfig.DirectoryServiceException;
import com.sonicsw.mf.common.dirconfig.IDirElement;
import com.sonicsw.mf.common.dirconfig.IDirIdentity;


/**
 * This interface provides Directory Service  read/write access to validation trigger code. 
 */
public interface ITriggerDSContext
{

    /**
     * Returns an element by name for update.
     * 
     * @param elementName
     *            the name of the element
     * 
     * @return the element object (or null if the element is not in the directory)
     * 
     * @exception DirectoryServiceException
     *                if there is a problem accessing the underlying storage
     * 
     */
    public IDirElement getElement(String elementName)
    throws DirectoryServiceException;

    /**
     * Returns the elements that matches the FROM clause and the WHERE clause of the query. Only attributes selected by
     * the SELECT clause are fetched. All the attributes are fetched if the query does not contain a SELECT clause.
     * 
     * @param query
     *            with a SELECT clause (optional), a FROM clause (mandatory) and a WHERE clause (optional)
     * @param forUpdate
     *            should be true if the element should be updateable (so SELECTclause is allowed in that case)
     * 
     * @return the elements that match the query (can be an empty list)
     * 
     * @exception DirectoryServiceException
     *                if there is a problem accessing the underlying storage or the parametrs are invalid
     * 
     * @see com.sonicsw.mf.common.config.query.Query
     * 
     */
    public IDirElement[] getElements(Query query, boolean forUpdate)
    throws DirectoryServiceException;

    /**
     * Applies a modification returned from IDirElement.donaUpdate(). Note that the object returned from
     * IDirElement.doneUpdate() should be cast to IDeltaElement.
     * <p>
     * With this version triggers will be fired.
     * 
     * @param delta
     *            the delta element to be applied to the DS
     * 
     * @exception DirectoryServiceException
     *                if there is a problem accessing the underlying storage
     * 
     */
    public void setElement(IDeltaElement delta)
    throws DirectoryServiceException;

    /**
     * Applies a modification returned from IDirElement.donaUpdate(). Note that the object returned from
     * IDirElement.doneUpdate() should be cast to IDeltaElement.
     * 
     * @param delta
     *            the delta element to be applied to the DS
     * @param fireTriggers
     *            indicates that triggers should be fired
     * 
     * @exception DirectoryServiceException
     *                if there is a problem accessing the underlying storage
     * 
     */
    public void setElement(IDeltaElement delta, boolean fireTriggers)
    throws DirectoryServiceException;

    /**
     * Deletes an element. This version is only applicable for elements that do not have a representation
     * in the view.
     * <p>
     * With this version triggers will be fired.
     * 
     * @param element
     *            the element to be deleted from the DS
     * 
     * @exception DirectoryServiceException
     *                if there is a problem accessing the underlying storage
     * 
     */
    public void deleteElement(String elementName)
    throws DirectoryServiceException;

    /**
     * Returns true if the directory exists.
     * 
     * @param directoryName
     *            the name of the directory in question
     * 
     * @return true if the directory exists
     * 
     * @exception DirectoryServiceException
     *                if there is a problem accessing the underlying storage
     * 
     * @see com.sonicsw.mf.common.config.query.Query
     * 
     */
    public boolean directoryExists(String directoryName)
    throws DirectoryServiceException;

    /**
     * Returns the list of elements stored in the directory.
     * 
     * @param directoryName
     *            the name of the directory
     * 
     * @return the list of elements (can be empty)
     * 
     * @exception DirectoryServiceException
     *                if there is a problem accessing the underlying storage
     * 
     */
    public IElementIdentity[] listElements(String directoryName)
    throws DirectoryServiceException;

    /**
     * Returns the list of directories stored in a given directory.
     * 
     * @param directoryName
     *            the name of the directory
     * 
     * @return the list of directories (can be empty)
     * 
     * @exception DirectoryServiceException
     *                if there is a problem accessing the underlying storage
     * 
     */
    public IDirIdentity[] listDirectories(String directoryName)
    throws DirectoryServiceException;

    /**
     * Deletes the given directory.
     * 
     * @param directoryName
     *            the name of the directory
     * 
     * @exception DirectoryServiceException
     *                if there is a problem accessing the underlying storage
     * 
     */
    public void deleteDirectory(String directoryName)
    throws DirectoryServiceException;

    /**
     * Log a message to the system log mechanism.
     *
     * @see #registerErrorCondition(String, int)
     */
    public void logMessage(String message, int severityLevel);

    /**
     * Log a message and associated exception to the system log mechanism.
     * 
     * @param message
     *            The description associated with the error condition.
     * @param exception
     *            The Exception thrown
     * @param severity
     *            The severity of the error - see Level for values.
     */
    public void logMessage(String message, Throwable throwable, int severityLevel);
}
