/*
 * Copyright (c) 2001 Sonic Software. All Rights Reserved.
 */

package com.sonicsw.mf.common.config;


/**
 * Defines a collection of attribute set types. The interface allows to
 * to efficiently maintain a large amount of attribute sets that share the same type.
 * the scope of the types is the object that implements the ITypeCollection interface.
 * @see com.sonicsw.mf.common.config.IAttributeSetType
 * @see com.sonicsw.mf.common.config.IAttributeSet
 * @see com.sonicsw.mf.common.config.IAttributeList
 */

public interface ITypeCollection
{
 /**
  * Defines a new type of IAttributeSet objects. Attribute definitions should be added in
  * the returned IAttributeSetType object before creating new IAttributeSet objects of that type.
  *
  * @param typeName the name of the IAttributeSet type
  *
  * @return a new (empty) IAttributeSetType object
  *
  * @exception AttributeSetTypeException if there is already an IAttributeSetType object with this name
  * @exception ReadOnlyException if its a read-only object 
  * @exception ConfigException illegal parameters (such as a null attributeName)
  *
  * @see com.sonicsw.mf.common.config.IAttributeSetType
  */
  public IAttributeSetType createAttributeSetType(String typeName) throws AttributeSetTypeException, ReadOnlyException, ConfigException;

 /**
  * Deletes the IAttributeSetType object. deleteAttributeSetType can be called only if there are no IAttributeSet
  * objects of that type.
  *
  * @param typeName the name of the IAttributeSet type
  *
  * @return the deleted IAttributeSetType object (null if does not exist)
  *
  * @exception AttributeSetTypeException if there is still a IAttributeSet of this type.
  * @exception ReadOnlyException if its a read-only object 
  * @exception ConfigException illegal parameters (such as a null attributeName)
  *
  * @see com.sonicsw.mf.common.config.IAttributeSetType
  *
  */
  public IAttributeSetType deleteAttributeSetType(String typeName) throws AttributeSetTypeException, ReadOnlyException, ConfigException;

 /**
  * returns the IAttributeSetType object.
  *
  * @param typeName the name of the IAttributeSet type
  *
  * @return the IAttributeSetType typeName object (null if does not exist)
  *
  * @see com.sonicsw.mf.common.config.IAttributeSetType
  *
  */
  public IAttributeSetType getAttributeSetType(String typeName);

 /**
  * returns the names of all the defined types
  *
  * @return the names of all the defined types
  *
  * @see com.sonicsw.mf.common.config.IAttributeSetType
  *
  */
  public String[] getAllTypeNames();

 /**
  * returns true if the other collection is equal to this one.
  *
  * @param otherCollection the ITypeCollection object to compare to
  *
  * @return true if otherCollection is equal to this one
  *
  */
  public boolean typesEqual(ITypeCollection otherCollection);

}
