/*
 * Copyright (c) 2001 Sonic Software. All Rights Reserved.
 */

package com.sonicsw.mf.common.config;
import com.sonicsw.mf.common.dirconfig.DirectoryServiceException;
import com.sonicsw.mf.common.dirconfig.IDirElement;


/**
 * This interface is implemented by component writers who wish to validate their configuration elements modifications
 * and/or to run a delete trigger. The implementation is then plugged into the DS.
 * The onDelete() method is excuted first followed by the validate() method. All the modification
 * performed by the transaction are passed to validate(), including any modifications performed by onDelete().
 * Once implemented, the class that implements this interface must be registered in an MF_VALIDATOR element under the
 * /_MFLibrary/validators directory. Example:
 *
 *      <ConfigElement>
 *        <ElementID version="1" releaseVersion="2.0" name="/_MFLibrary/validators/MQ_BROKER.5.0" creationTimestamp="991951719840" type="MF_VALIDATOR" />
 *        <AttributeSet>
 *          <Attribute name="TYPE" type="string" value="MQ_BROKER" />
 *          <Attribute name="RELEASE_VERSION" type="string" value="5.0" />
 *          <Attribute name="VALIDATION_CLASS" type="string" value="progress.message.config.Validator" />
 *          <AttributeList>
 *            <AttributeName name="VALIDATOR_ARCHIVES" />
 *            <ListItem type="string" value="ds:///MQ_BROKER/5.0/mq_config.jar" />
 *          </AttributeList>
 *        </AttributeSet>
 *      </ConfigElement>
 *
 * In the example above, progress.message.config.Validator implementes IValidator. The class is stored in the
 * DS archive - ds:///MQ_BROKER/5.0/mq_config.jar. The mq broker's release version is 5.0. Note that a
 * progress.message.config.Validator of another mq broker version could coexist with this one - it would be registered
 * by another MF_VALIDATOR element under the /_MFLibrary/validators directory:
 *
 *      <ConfigElement>
 *        <ElementID version="1" releaseVersion="2.0" name="/_MFLibrary/validators/MQ_BROKER.4.0" creationTimestamp="991951719840" type="MF_VALIDATOR" />
 *        <AttributeSet>
 *          <Attribute name="TYPE" type="string" value="MQ_BROKER" />
 *          <Attribute name="RELEASE_VERSION" type="string" value="4.0" />
 *          <Attribute name="VALIDATION_CLASS" type="string" value="progress.message.config.Validator" />
 *          <AttributeList>
 *            <AttributeName name="VALIDATOR_ARCHIVES" />
 *            <ListItem type="string" value="/sonic/sonicmq/lib/mq_config.jar" />
 *          </AttributeList>
 *        </AttributeSet>
 *      </ConfigElement>
 *
 * In the above example, the mq_config.jar file is stored in the file system rather than in the DS.
 *
 *
 */
public interface IValidator
{
 /**
  * Returns the release version this IValidator class validates. The version returned must match the RELEASE_VERSION attribute
  * in the MF_VALIDATOR element that registers this validator.
  *
  * @return the release version this IValidator class validates
  *
  */
  public String getReleaseVersion();

 /**
  * Returns the list of element types validated by this class. Whenever an element of a matching type and release version is
  * modified, the DS passes the modification to the validate() mthod of this class for approval.
  *
  * @return the list of element types this calss validates.
  *
  */
  public String[] getElementTypesValidated();

 /**
  * The validate method is passed a list of modifications for approval. All the modifications that occur under a single
  * transaction are passed in a single validate call. For example, an XML import is performed under a single transaction.
  * When the method is called, the modifications  where already applied to the DS, but not committed. If validate() throws
  * an exception then those modifications are  rolled back by the ds. Otherwise, the modifications are committed.
  * in the MF_VALIDATOR element that registers this validator.
  *
  * It's okay to have an empty implementation body (if for example only onDelete() is required).
  *
  * @param dsContext the validating code can use the IValidationDSContext interface for a read-only access to the DS.
  * @param changes the list of changes performed by the current transaction
  *
  * @exception Exception should thrown if any of the modifications in the list is invalid. Note that the administration
  *            client that that attempts an invalid modification would have the
  * com.sonicsw.mf.common.dirconfig.DirectoryServiceValidationException thrown. The exception (or Throwable) thrown from
  * this validate() call is accessible from the DirectoryServiceValidationException exception on the client
  * (if it is serializable).
  *
  */
  public void validate(IValidationDSContext dsContext, IValidationElementChange[] changes) throws Exception;

 /**
  * The onDelete() method can perform any DS cleanup based on the list of deleted elements. For example, onDelete() will call
  * dsContext.getElements(...) to find all the elements that reference the deleted element, remove the dangling references
  * and will call dsContext.setElement(...) to set the modifications in the DS. All the modifications performed are part of the
  * same transaction in which the elements where deleted. onDelete() is not called for XML imports (since elements are deleted only
  * to be replaced). Modification performed in onDelete are validated (in validate()) and reported to clients like any other modifications
  * performed directly by DS clients.
  *
  * It's okay to have an empty implementation body (if for example only validate() is required).
  *
  * @param dsContext the validating code can use the ITriggerDSContext interface for a limitted read/write access to the DS.
  * @param changes the list of elements (their before deletion image) deleted in the current transaction
  *
  * @exception DirectoryServiceException might be thrown when onDelete() uses dsContext to access the DS. But normally, if the trigger
  *            is correct and there is no catastrophic failure in the DS - no exception should be thrown from onDelete.
  *            The client would receive a dirconfig.DirectoryServiceTriggerException exception if any Throwable was thrown from onDelete().
  */
  public void onDelete(ITriggerDSContext dsContext, IDirElement[] beforeImages) throws DirectoryServiceException;
  
  public void onUpdate(ITriggerDSContext dsContext, IDirElement[] beforeImages, IDeltaElement[] afterImages) throws DirectoryServiceException;
  
  public void onCreate(ITriggerDSContext dsContext, IDirElement[] afterImages) throws DirectoryServiceException;
}
