/*
 * Copyright (c) 2001 Sonic Software. All Rights Reserved.
 */

package com.sonicsw.mf.common.config;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map.Entry;
import java.util.concurrent.locks.ReadWriteLock;
import java.util.concurrent.locks.ReentrantReadWriteLock;

import com.sonicsw.mx.util.IEmptyArray;

import com.sonicsw.mf.common.config.impl.EntityName;
import com.sonicsw.mf.common.config.impl.NameMapper;

/**
 * Maintains a two way, thread safe, translation map between logical and storage names
 */
public final class LogicalStorageNameMapper implements java.io.Serializable
{

    public static void main(String[] args) throws Exception
    {
        LogicalStorageNameMapper mapper = new LogicalStorageNameMapper();

        String E1 = "/a1/b1/l1";
        String E2 = "/a1/b1/l2";
        String E3 = "/a1/b2/l3";
        String oldEr = "/a1";
        String newEr = "/x1";

        mapper.set(E1, "/V1");
        mapper.set(E2, "/V2");
        mapper.set(E3, "/V3");

        HashMap<String, String> corrections = new HashMap<String, String>();
        corrections.put("/V1", "/xxx");
        mapper.applyCorrections(corrections);
        corrections = new HashMap<String, String>();
        corrections.put("/V1", E1);
        mapper.applyCorrections(corrections);

        System.out.println(mapper.storageToLogical("/V1") + " logical: " +  mapper.logicalToStorage(E1));
        System.out.println(mapper.storageToLogical("/V2") + " logical: " +  mapper.logicalToStorage(E2));
        System.out.println(mapper.storageToLogical("/V3") + " logical: " +  mapper.logicalToStorage(E3));

        String[] fullList = mapper.scan("/a1/b1");
        for (int i = 0; i < fullList.length; i++)
        {
            System.out.println(fullList[i]);
        }

        System.out.println();

        String[] renamedList = mapper.rename(oldEr, newEr);
        for (int i = 0; i < renamedList.length; i++)
        {
            System.out.println(renamedList[i]);
        }

        System.out.println();
        System.out.println("REMOVE: " + mapper.deleteByStorageName("/V1"));
        System.out.println();
        fullList = mapper.scan("/");
        for (int i = 0; i < fullList.length; i++)
        {
            System.out.println(fullList[i]);
        }

        System.out.println();
        System.out.println("REMOVE: " + mapper.deleteByStorageName("/V2"));
        System.out.println();
        fullList = mapper.scan("/");
        for (int i = 0; i < fullList.length; i++)
        {
            System.out.println(fullList[i]);
        }

        System.out.println();
        System.out.println("REMOVE: " + mapper.deleteByStorageName("/V3"));
        System.out.println();
        fullList = mapper.scan("/");
        for (int i = 0; i < fullList.length; i++)
        {
            System.out.println(fullList[i]);
        }

        System.out.println();
        E1 = "/a1/b1/l1";
        E2 = "/a1/b1/l2";
        E3 = "/a2/b2/l3";


        mapper.set(E1, "/V1");
        mapper.set(E2, "/V2");
        mapper.set(E3, "/V3");

        System.out.println(mapper.logicalToStorage(E1) + " " + mapper.storageToLogical("/V1"));
        System.out.println(mapper.logicalToStorage(E2) + " " + mapper.storageToLogical("/V2"));
        System.out.println(mapper.logicalToStorage(E3) + " " + mapper.storageToLogical("/V3"));

        System.out.println();
        fullList = mapper.scan("/");
        for (int i = 0; i < fullList.length; i++)
        {
            System.out.println(fullList[i] + " " + mapper.storageToLogical(fullList[i]));
        }

        System.out.println();
        oldEr = "/a1/b1/l1";
        newEr = "/a1/b1/x1";
        renamedList = mapper.rename(oldEr, newEr);
        for (int i = 0; i < renamedList.length; i++)
        {
            System.out.println("RENAMED: " + renamedList[i] + " " + mapper.storageToLogical(renamedList[i]));
        }

        System.out.println();
        System.out.println("REMOVING...");
        fullList = mapper.scan("/a1/b1");
        for (int i = 0; i < fullList.length; i++)
        {
            System.out.println("NAME: " + fullList[i] + " " + mapper.storageToLogical(fullList[i]));
        }

        System.out.println();
        renamedList =  mapper.deleteByLogicalName("/a1/b1");
        for (int i = 0; i < renamedList.length; i++)
        {
            System.out.println("REMOVED ITEM: " + renamedList[i]);
        }
        System.out.println();
        fullList = mapper.scan("/");
        for (int i = 0; i < fullList.length; i++)
        {
            System.out.println("NAME: " + fullList[i] + " " + mapper.storageToLogical(fullList[i]));
        }
    }

    private static final long serialVersionUID = 0L;

    private HashMap<String, String> m_storageToLogical;
    private transient NameMapper m_logicalToStorage;
    private transient ReadWriteLock m_lock;
    private static final String NEWLINE = System.getProperty("line.separator");

    /**
     * Constructor for an empty map.
     */
    public LogicalStorageNameMapper()
    {
    	m_lock = new ReentrantReadWriteLock();
        init();
    }

    /**
     * Returns the number of mappings the mapper holds.
     */
    public int size() {
        m_lock.readLock().lock();
        try
        {
            return m_storageToLogical == null ? 0 : m_storageToLogical.size();
        }
        finally
        {
            m_lock.readLock().unlock();
        }
    }

    private void init()
    {
        m_storageToLogical = new HashMap<String, String>();
        m_logicalToStorage = new NameMapper();
    }

    /**
     * Gets the storage-to-logical map. Used by the container to send the map's content to the DS for reconciliation
     *
     * @returns the storage-to-logical map
     */
    @SuppressWarnings("unchecked")
    public HashMap<String, String> getStorageToLogicalMap()
    {
        m_lock.readLock().lock();
        try
        {
            return (HashMap<String, String>)m_storageToLogical.clone();
        }
        finally
        {
            m_lock.readLock().unlock();
        }
    }

    /**
     * Makes corrections to the map.  Used by the container to reconcile the map with DS changes.
     *
     * @param corrections the corrections table
     */
    public void applyCorrections(HashMap<String, String> corrections)
    {
        m_lock.writeLock().lock();
        try
        {
            HashMap<String, String> saveStorageToLogical = m_storageToLogical;
    
            Iterator<Entry<String, String>> storageIterator = corrections.entrySet().iterator();
    
            // First correct the storage-to-logical list
            while (storageIterator.hasNext())
            {
                Entry<String, String> entry = storageIterator.next();
                String storageName = entry.getKey();
                String newLogicalName = entry.getValue();
                if (newLogicalName != null)
                {
                    saveStorageToLogical.put(storageName, newLogicalName);
                }
                else
                {
                    saveStorageToLogical.remove(storageName);
                }
            }
    
            // Reinitialize and construct from the new storage-to-logical
            init();
            constructFromStorageLogicalMap(saveStorageToLogical);
        }
        finally
        {
            m_lock.writeLock().unlock();
        }
    }
    
    /**
     * Sets a new entry pair. A one-to-one relation is enforced.
     *
     * @param logicalPath
     * @param storagePath
     *
     */
    
    public void set(String logicalPath, String storagePath)
    {
    	set(logicalPath, storagePath, false);
    }


    /**
     * Sets a new entry pair. A one-to-one relation is enforced.
     *
     * @param logicalPath
     * @param storagePath
     * @param replaceMap true if a previous existing mapping between the logical name and
     * a different storage name existed and should be overwritten. This is the case when 
     * archives are deleted and re-imported into the DS while a container is not running.
     * When the file is brought into the container cache during phase 1, the code will instruct
     * the cache to replace the mapping that might have existed through the replaceMap flag
     *
     */
    public void set(String logicalPath, String storagePath, boolean replaceMap)
    {
        m_lock.writeLock().lock();
        try
        {
          String insertMessage = NEWLINE + "Inserting LOGICAL: " + logicalPath + " STORAGE: " + storagePath + NEWLINE;
          if (logicalPath == null || storagePath == null)
        {
            throw new RuntimeException("Storage and logical must not be null. " + insertMessage);
        }
  
          EntityName logicalPathE = getEntityName(logicalPath);
          String existingLogical = m_storageToLogical.get(storagePath);
          String existingStorage = (String)m_logicalToStorage.get(logicalPathE);
  
          if (replaceMap || (existingLogical == null && existingStorage == null))
          {
              // take care of a previous mapping for the logical name. This happens
              // in a container cache, for instance, when a file is deleted from the 
              // DS and reimported while the container is not running. Because storage names
              // are not reused in the same way as logical names, checking for existingLogical!= null
              // probably doesn't make sense
              //
              // remove it *before* setting anew since we might have the same name
              if (replaceMap && existingStorage != null)
            {
                m_storageToLogical.remove(existingStorage);
            }
  
              m_storageToLogical.put(storagePath, logicalPath);
              m_logicalToStorage.set(logicalPathE, storagePath);
          }
          else if (existingLogical != null && existingStorage != null)
          {
              if (existingLogical.equalsIgnoreCase(logicalPath) && existingStorage.equals(storagePath))
              {
                  //NOOP
              }
            else
            {
                throw new RuntimeException(insertMessage + existingErrorMessage(existingLogical, existingStorage));
            }
          }
        else
        {
            throw new RuntimeException(insertMessage + existingErrorMessage(existingLogical, existingStorage));
        }
        }
        finally
        {
            m_lock.writeLock().unlock();
        }
    }

    /**
     * Renames a logical subtree
     *
     * @param oldPath
     * @param newPath
     *
     * @return the storage apth list of renamed elements
     */
    public String[] rename (String oldPath, String newPath)
    {
        m_lock.writeLock().lock();
        try
        {
            INamedPayload[] list = m_logicalToStorage.rename(getEntityName(oldPath), getEntityName(newPath));
            if (list == null)
            {
                return IEmptyArray.EMPTY_STRING_ARRAY;
            }
            String[] result = new String[list.length];
            for (int i = 0; i < list.length; i++)
            {
                result[i] = (String)list[i].getPayload();
                m_storageToLogical.put(result[i], list[i].getName()); // Fix the storage-to-logical mapping
            }
            return result;
        }
        finally
        {
            m_lock.writeLock().unlock();
        }
    }

    /**
     * Deletes a pair through the storage path
     *
     * @param storagePath
     *
     * return the remove logical path
     */
    public String deleteByStorageName (String storagePath)
    {
        m_lock.writeLock().lock();
        try
        {
            String logicalPath = m_storageToLogical.remove(storagePath);
            if (logicalPath != null)
            {
                m_logicalToStorage.remove(getEntityName(logicalPath));
            }
            return logicalPath;
        }
        finally
        {
            m_lock.writeLock().unlock();
        }
    }

    /**
     * Deletes using logical path (a single pair or a subtree)
     *
     * @param logicalPath
     *
     * @return the storage path list of removed elements
     *
     */
    public String[] deleteByLogicalName (String logicalPath)
    {
        m_lock.writeLock().lock();
        try
        {
            INamedPayload[] list = m_logicalToStorage.remove(getEntityName(logicalPath));
            if (list == null)
            {
                return IEmptyArray.EMPTY_STRING_ARRAY;
            }
            String[] result = new String[list.length];
            for (int i = 0; i < list.length; i++)
            {
                result[i] = (String)list[i].getPayload();
                m_storageToLogical.remove(result[i]);
            }
            return result;
        }
        finally
        {
            m_lock.writeLock().unlock();
        }
    }

    /**
     * Returns the logical path given the storage path.
     *
     * @param storagePath
     *
     * @return the logical path associated with this storage path
     */
    public String storageToLogical(String storagePath)
    {
        m_lock.readLock().lock();
        try
        {
            return m_storageToLogical.get(storagePath);
        }
        finally
        {
            m_lock.readLock().unlock();
        }
    }

    /**
     * Returns the storage path given the logical path (null is returned if logicalPath is a folder).
     *
     * @param logicalPath
     *
     * @return the storage path associated with this logical path (null if logicalPath is a folder).
     */
    public String logicalToStorage(String logicalPath)
    {
        m_lock.readLock().lock();
        try
        {
            Object node = m_logicalToStorage.get(getEntityName(logicalPath));
            if (node instanceof String)
            {
                return (String)node;
            }
            else
            {
                return null;
            }
        }
        finally
        {
            m_lock.readLock().unlock();
        }
    }

    /**
     * Returns the list of storage names under a logical subtree (a folder or an element).
     *
     * @param logicalPath
     *
     * @return the list of storage names under a logical subtree (a folder or an element)
     */
    public String[] scan(String logicalPath)
    {
        m_lock.readLock().lock();
        try
        {
            INamedPayload[] list = m_logicalToStorage.scan(getEntityName(logicalPath));
            if (list == null)
            {
                return IEmptyArray.EMPTY_STRING_ARRAY;
            }
            String[] result = new String[list.length];
            for (int i = 0; i < list.length; i++)
            {
                result[i] = (String)list[i].getPayload();
            }
            return result;
        }
        finally
        {
            m_lock.readLock().unlock();
        }
    }

    @Override
    public Object clone()
    {
        m_lock.readLock().lock();
        try
        {
            LogicalStorageNameMapper lsnm = new LogicalStorageNameMapper();
            lsnm.constructFromStorageLogicalMap((HashMap<String, String>)m_storageToLogical.clone());
            return lsnm;
        }
        finally
        {
            m_lock.readLock().unlock();
        }
    }

    private void writeObject(java.io.ObjectOutputStream s) throws java.io.IOException
    {
        s.writeObject(m_storageToLogical);
    }

    private void readObject(java.io.ObjectInputStream s) throws java.io.IOException, ClassNotFoundException
    {
    	m_lock = new ReentrantReadWriteLock();
        init();
        constructFromStorageLogicalMap((HashMap<String, String>)s.readObject());
    }

    private void constructFromStorageLogicalMap(HashMap<String, String> storageToLogical)
    {
        // Builds up the logical to storage map
        m_logicalToStorage = new NameMapper();
        Iterator<Entry<String, String>> iterator = storageToLogical.entrySet().iterator();
        while (iterator.hasNext())
        {
            Entry<String, String> entry = iterator.next();
            set(entry.getValue(), entry.getKey());
        }
        m_storageToLogical = storageToLogical;
    }

    private EntityName getEntityName(String path)
    {
        try
        {
            return new EntityName(path);
        }
        catch (ConfigException e)
        {
            throw new RuntimeException(e.toString());
        }
    }

    private static String existingErrorMessage(String existingLogical, String existingStorage)
    {
        return "One-to-One relation violation - existing LOGICAL:  " + existingLogical + " existing STORAGE: " + existingStorage;
    }
}
