package com.sonicsw.mf.common.config;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Set;

import com.sonicsw.mf.common.config.impl.Util;
import com.sonicsw.mf.common.dirconfig.IDirElement;

public class MergeUtil
{
    /**
     * A method that will merge the changes made to the data specified in the
     * modifiedElements into the specified originalElement.  Added/remove attributes
     * will be added and removed from the originalElement.  Altered values in the
     * modified element will be altered in the original element.
     *
     * @param originalElement The element to merge modified values into.
     * @param modifiedElement The element with the modifications to merge.
     * @throws ReadOnlyException If an attempt is made to alter a readonly attribute.
     * @throws AttributeSetTypeException If the attribute type is incorrect.
     * @throws ConfigException When there is a configuration error.
     */
    public static void mergeModifications( IDirElement originalElement, IDirElement modifiedElement )
        throws ReadOnlyException, AttributeSetTypeException, ConfigException
    {
        mergeModifications(originalElement, modifiedElement, false);
    }

    /**
     * A sparse version of the above mergeModifications(IDirElement originalElement, IDirElement modifiedElement)
     * I missing attribute in modifiedElement is interpreted as not modified rather than deleted.
     * Note that a modified attribute list cannot be sparse - the entire list must be present in modifiedElement.
     * Note sparse mode update does not work that typed attribute sets (we don't use typed attribute sets in the product
     * and don't plan to use them).
     *
     * @param originalElement The element to merge modified values into.
     * @param modifiedElement The element with the modifications to merge.
     * @param sparseMode if true ignores missing attributes in modifiedElement
     * @throws ReadOnlyException If an attempt is made to alter a readonly attribute.
     * @throws AttributeSetTypeException If the attribute type is incorrect.
     * @throws ConfigException When there is a configuration error.
     */
    public static void mergeModifications( IDirElement originalElement, IDirElement modifiedElement, boolean sparseMode )
        throws ReadOnlyException, AttributeSetTypeException, ConfigException
    {
        // Retrieve the respective attribute sets
        IAttributeSet modifiedElements = (IAttributeSet)modifiedElement.getAttributes();
        IAttributeSet originalElements = (IAttributeSet)originalElement.getAttributes();

        // Phase 1, traverse the attributes in the modified elements.  If there are
        // any attributes in the modified elements that are not in the original elements,
        // then add them.  If there are any attributes in the modifiedElements that have
        // been changed, then set them in the original elements.
        mergeAddAndSet( originalElements, modifiedElements);

        // Phase 2, traverse the attributes in the original elements.  If there are
        // any attributes in the original elements that are not located in the modified
        // elements, then the value has been deleted.  Delete those values from the
        // original elements.
        if (!sparseMode)
        {
            deleteDeletedElements( originalElements, modifiedElements );
        }
    }


    // This method assumes "mergeAddAndSet" has already been run.  After mergeAddAndSet
    // completes, there should be not type mismatches.  In addition, all lists
    // that have has changes would have been reset.
    //
    // This method recursively traverses the "original" attribute set looking for
    // attributes that are not located in the modified attribute set.  If an attribute is
    // not found, it is deleted from the original attribute set.

    private static void deleteDeletedElements( IAttributeSet original, IAttributeSet modified )
        throws ReadOnlyException, AttributeSetTypeException, ConfigException
    {
        // Get the keys associated with the original attribute set.
        HashMap map = original.getAttributes();
        Set keys = map.keySet();

        Iterator iter = keys.iterator();
        while( iter.hasNext() )
        {
            String key = (String)iter.next();

            if (key.equals("_MF_SYSTEM_ATTRIBUTES"))
            {
                continue;
            }

            // Get the modified value
            Object modifiedValue = modified.getAttribute( key );

            // Get the original value
            Object originalValue = original.getAttribute( key );

            if( modifiedValue == null )
            {
                // If the modified value is null, then the value has been
                // deleted.  Remove the attribute from the original.
                original.deleteAttribute( key );
            }
            else
            if( modifiedValue instanceof IAttributeSet )
            {
                // If the modified value is an attribute set, then traverse
                deleteDeletedElements( (IAttributeSet)originalValue, (IAttributeSet)modifiedValue );
            }

            // If the modified value is an attribute list, then skip it.  It
            // should not contain any "deleted" elements after "mergeAndSet" has been
            // run.
        }

    }

    // This method recursively traverse the "modified" attribute set looking for
    // values that has been changed or added.
    public static void mergeAddAndSet( IAttributeSet original, IAttributeSet modified)
        throws ReadOnlyException, AttributeSetTypeException, ConfigException
    {
        // Retrieve the attribute keys for the modified attribute set.
        HashMap map = modified.getAttributes();
        Set keys = map.keySet();

        Iterator iter = keys.iterator();
        while( iter.hasNext() )
        {
            // retrieve the original and modified values.
            String key = (String)iter.next();
            Object modifiedValue = map.get( key );
            Object originalValue = original.getAttribute( key );

            if( originalValue == null )
            {
                // This would occur if the modified value was a new value
                // that does not exist in the original

                if( modifiedValue instanceof IAttributeSet )
                {
                    // If the new value is an AttributeSet, then
                    // create a new AttributeSet in the original
                    IAttributeSet mset = (IAttributeSet)modifiedValue;
                    IAttributeSet nset = createNewSet( key, original, mset,
                        original.getAttributeSetType( mset.getTypeName() ) );

                    setAttributeSet( nset, mset );
                }
                else
                if( modifiedValue instanceof IAttributeList )
                {
                    // If the modified value is an attribute list, then
                    // create a new attribute list in the original.  Set
                    // all the elements in the modified list into the original.
                    IAttributeList mlist = (IAttributeList)modifiedValue;
                    IAttributeList nlist = createNewList( key, original, mlist );

                    setAttributeList( nlist, mlist );
                }
                else
                {
                    // Just call setObjectAttribute() on the original.  If the
                    // values are equivalent, this should be a no-op.
                    original.setObjectAttribute( key, modifiedValue);
                }

                continue;
            }

            // If the original value exists, then check its type

            if( modifiedValue instanceof IAttributeSet )
            {
                if( originalValue instanceof IAttributeSet )
                {
                    if ( ((ITypeCollection)originalValue).typesEqual((ITypeCollection)modifiedValue))
                    {
                        // Recursively inspect the new attribute set, merging and
                        // changing values.
                        mergeAddAndSet( (IAttributeSet)originalValue, (IAttributeSet)modifiedValue);
                    }
                    else
                    {
                        original.deleteAttribute( key );

                        // If the new value is an AttributeSet, then
                        // create a new AttributeSet in the original
                        IAttributeSet mset = (IAttributeSet)modifiedValue;
                        IAttributeSet nset = createNewSet( key, original, mset,
                            original.getAttributeSetType( mset.getTypeName() ) );

                        setAttributeSet( nset, mset );
                    }
                }
                else
                {
                    // If the types are different, delete the original and
                    // add the new set.

                    original.deleteAttribute( key );

                    // If the new value is an AttributeSet, then
                    // create a new AttributeSet in the original
                    IAttributeSet mset = (IAttributeSet)modifiedValue;
                    IAttributeSet nset = createNewSet( key, original, mset,
                        original.getAttributeSetType( mset.getTypeName() ) );

                    setAttributeSet( nset, mset );
                }
            }
            else
            if( modifiedValue instanceof IAttributeList )
            {
                if( originalValue instanceof IAttributeList )
                {
                    if( !isEqual( (IAttributeList)modifiedValue, (IAttributeList)originalValue ) )
                    {
                        // If the lists are not equivalent - replace the list with the new one
                        IAttributeList ml = (IAttributeList)modifiedValue;

                        IAttributeList ol = createNewList(key, original, ml);
                        setAttributeList( ol, ml );
                    }
                }
                else
                {
                    // If the types are different, delete the original and
                    // add the new list.

                    original.deleteAttribute( key );

                    IAttributeList mlist = (IAttributeList)modifiedValue;
                    IAttributeList nlist = createNewList( key, original, mlist );

                    setAttributeList( nlist, mlist );
                }
            }
            else
            {
                // If the modified value is a primative, then call setObjectAttribute().
                // This method should be a no-op for equivalent values.
                original.setObjectAttribute( key, modifiedValue );
            }
        }

        String[] markedForDeletion = modified.mergeUtilHint_getAttributesMarkedDeleted();
        for (int i = 0; i < markedForDeletion.length; i++)
        {
            original.deleteAttribute(markedForDeletion[i]);
        }
    }

    // This method recursively changes if two lists are equivalent.
    private static boolean isEqual( IAttributeList original, IAttributeList modified )
    {
        if (!original.typesEqual((ITypeCollection)modified))
        {
            return false;
        }

        ArrayList originalList = original.getItems();
        ArrayList modifiedList = modified.getItems();

        int originalSize = originalList.size();

        // If the sizes are different, return false.
        if( originalSize != modifiedList.size() )
        {
            return false;
        }

        for( int count = 0; count < originalSize; count++ )
        {
            Object originalValue = originalList.get( count );
            Object modifiedValue = modifiedList.get( count );

            if( !equalHelper( originalValue, modifiedValue ) )
            {
                return false;
            }
        }

        return true;
    }

    // This method recursively changes if two sets are equivalent.
    public static boolean isEqual( IAttributeSet original, IAttributeSet modified )
    {
        if (!original.typesEqual((ITypeCollection)modified))
        {
            return false;
        }

        String originalTypeName = original.getTypeName();
        String modifiedTypeName = modified.getTypeName();

        if (originalTypeName == null)
        {
            if (modifiedTypeName != null)
            {
                return false;
            }
        }
        else if (modifiedTypeName == null)
        {
            if (originalTypeName != null)
            {
                return false;
            }
        }
        else if (!originalTypeName.equals(modifiedTypeName))
        {
            return false;
        }

        HashMap map = modified.getAttributes();
        Set keys = map.keySet();

        Iterator iter = keys.iterator();
        while( iter.hasNext() )
        {
            String key = (String)iter.next();
            Object modifiedValue = map.get( key );
            Object originalValue = original.getAttribute( key );

            if( originalValue == null )
            {
                return false;
            }

            if( !equalHelper( originalValue, modifiedValue ) )
            {
                return false;
            }
        }

        return true;
    }

    // A helper method used by isEqual() methods in the context of AttributeSets
    // and AttributeLists
    private static boolean equalHelper( Object originalValue, Object modifiedValue )
    {
        if( modifiedValue instanceof IAttributeSet )
        {
            if( originalValue instanceof IAttributeSet )
            {
                return isEqual( (IAttributeSet)originalValue, (IAttributeSet)modifiedValue );
            }
            else
            {
                return false;
            }
        }
        else
        if( modifiedValue instanceof IAttributeList )
        {
            if( originalValue instanceof IAttributeList )
            {
                return isEqual( (IAttributeList)modifiedValue, (IAttributeList)originalValue );
            }
            else
            {
                return false;
            }
        }
        else
        if( !Util.atomicAndEqual( originalValue, modifiedValue ) )
        {
            return false;
        }

        return true;
    }

    // This method recursively sets all the attributes in the modified set into
    // the new set.
    private static void setAttributeSet( IAttributeSet newSet, IAttributeSet modifiedSet )
        throws ReadOnlyException, AttributeSetTypeException, ConfigException
    {
        HashMap map = modifiedSet.getAttributes();
        Set keys = map.keySet();
        Iterator iter = keys.iterator();

        while( iter.hasNext() )
        {
            String key = (String)iter.next();
            Object value = modifiedSet.getAttribute( key );

            if( value instanceof IAttributeSet )
            {
                IAttributeSet mset = (IAttributeSet)value;
                IAttributeSet nset = createNewSet( key, newSet, (IAttributeSet)value,
                    newSet.getAttributeSetType( mset.getTypeName() ) );

                setAttributeSet( nset, mset );
            }
            else
            if( value instanceof IAttributeList )
            {
                IAttributeList mlist = (IAttributeList)value;
                IAttributeList nlist = createNewList( key, newSet, mlist );

                setAttributeList( nlist, mlist );
            }
            else
            {
                newSet.setObjectAttribute( key, value );
            }
        }
    }

    // This method recursively sets all the attributes in the modified list into
    // the new list.
    private static void setAttributeList( IAttributeList newList, IAttributeList modifiedList )
        throws ReadOnlyException, AttributeSetTypeException, ConfigException
    {
        ArrayList list = modifiedList.getItems();
        int size = list.size();

        for( int count = 0; count < list.size(); count++ )
        {
            Object value = list.get( count );
            if( value instanceof IAttributeSet )
            {
                IAttributeSet mset = (IAttributeSet)value;
                IAttributeSet nset = createNewSet( newList, mset,
                    newList.getAttributeSetType( mset.getTypeName() ) );

                setAttributeSet( nset, mset );
            }
            else
            if( value instanceof IAttributeList )
            {
                IAttributeList mlist = (IAttributeList)value;
                IAttributeList nlist = createNewList( newList, mlist );

                setAttributeList( nlist, mlist );
            }
            else
            {
                newList.addObjectItem( value );
            }
        }
    }

    private static IAttributeList createNewList( String key, IAttributeSet newSet, IAttributeList mlist )
        throws ReadOnlyException, AttributeSetTypeException, ConfigException
    {
        IAttributeList nlist = newSet.createAttributeList( key );

        return nlist;
    }

    private static IAttributeList createNewList( IAttributeList newList, IAttributeList mlist )
        throws ReadOnlyException, AttributeSetTypeException, ConfigException
    {
        IAttributeList nlist = newList.addNewAttributeListItem();

        return nlist;
    }

    private static IAttributeSet createNewSet( String key, IAttributeSet newSet, IAttributeSet mset, IAttributeSetType type )
        throws ReadOnlyException, AttributeSetTypeException, ConfigException
    {
        IAttributeSet nset = null;

        nset = newSet.createAttributeSet(
            key,
            type );

        createNewSetType( nset, mset );

        return nset;
    }

    private static IAttributeSet createNewSet( IAttributeList newList, IAttributeSet mset, IAttributeSetType type )
        throws ReadOnlyException, AttributeSetTypeException, ConfigException
    {
        IAttributeSet nset = null;

        nset = newList.addNewAttributeSetItem( type );

        createNewSetType( nset, mset );

        return nset;
    }

    private static void createNewSetType( ITypeCollection newCollection, ITypeCollection modifiedCollection )
        throws ReadOnlyException, AttributeSetTypeException, ConfigException
    {
        String[] typeNames = modifiedCollection.getAllTypeNames();

        for( int count = typeNames.length - 1; count >=0; count-- )
        {
            IAttributeSetType nType = newCollection.createAttributeSetType( typeNames[ count ] );
            IAttributeSetType mType = modifiedCollection.getAttributeSetType( typeNames[ count ] );

            String[] typeTypeNames = mType.getAttributeNames();
            for( int ttnCount = typeTypeNames.length - 1; ttnCount >= 0; ttnCount-- )
            {
                nType.addAttributeName( typeTypeNames[ ttnCount ] );
            }
        }
    }

}
