/*
 * Copyright (c) 2001 Sonic Software. All Rights Reserved.
 */

package com.sonicsw.mf.common.config.impl;
import java.util.HashMap;
import java.util.Iterator;

import com.sonicsw.mf.common.config.query.AttributeName;
import com.sonicsw.mf.common.config.query.impl.Util;

// Associate element attributes with objects (such at modification handlers)
public final class AttributeAssociationTree
{
    private AttributeSetNode m_tree;
 
    public int size()
    {
        return m_tree.size();
    }

    // Constructs a new tree
    public AttributeAssociationTree()
    {
         m_tree = new AttributeSetNode();
    }

    // Constructs a tree from an existing node
    public AttributeAssociationTree(AttributeSetNode node)
    {
        m_tree = node;
    }

    public Object getSubtree(Object name)
    {
        Object node = m_tree.get(name);
        if (node == null)
        {
            return null;
        }
        else if (node instanceof AttributeSetNode)
        {
            return new AttributeAssociationTree((AttributeSetNode)node);
        }
        else if (node instanceof Payload)
        {
            return ((Payload)node).getValue();
        }
        else
        {
            throw new IllegalArgumentException();
        }
    }

    public void insertPayload(AttributeName attName, Object payLoad)
    {
         if (attName.getComponentCount() == 0)
        {
            m_tree.insertPayload(payLoad);
        }
        else
        {
            m_tree.insertPayload(attName.getComponent(0), Util.removeFirstComponent(attName), payLoad);
        }
    }

    public void removeNode(AttributeName attName)
    {
         if (attName.getComponentCount() == 0)
        {
            m_tree.clear();
        }
        else
        {
            m_tree.removeNode(attName.getComponent(0), Util.removeFirstComponent(attName));
        }
    }

    // remove the payload and delete the whole node if no pay loads under it 
    public Object removePayload(AttributeName attName)
    {
         if (attName.getComponentCount() == 0)
        {
            return m_tree.removePayload();
        }
        else
        {
            return m_tree.removePayload(attName.getComponent(0), Util.removeFirstComponent(attName));
        }
    }

    // Returns null if there is no payload for attName
    public Object getPayload(AttributeName attName)
    {
         if (attName.getComponentCount() == 0)
        {
            return m_tree.getPayload();
        }
        else
        {
            return m_tree.getPayload(attName.getComponent(0), Util.removeFirstComponent(attName));
        }
    }

    private class AttributeSetNode extends HashMap
    {
        private int m_level;
        private Object m_payLoad;

        AttributeSetNode()
        {
            this(0);
        }

        AttributeSetNode(int level)
        {
            super();
            m_level = (level);
            m_payLoad = null;
        }

        @Override
        public String toString()
        {
            Iterator iterator = keySet().iterator();
            String result = m_payLoad != null ? "P " : "NP ";
            String spaces = "";
            for (int i = 0; i < m_level; i++)
            {
                spaces += "  ";
            }
            while (iterator.hasNext())
            {
                Object attName = iterator.next();
                result += "\n"+ spaces + attName;
                Object node = get(attName);
                if (node instanceof AttributeSetNode)
                {
                    result += node.toString();
                }
            }
            return result;
        }

        void insertPayload(Object payLoad)
        {
            m_payLoad = payLoad;
        }

        Object removePayload()
        {
            Object payLoad = m_payLoad;
            m_payLoad = null;
            return payLoad;
        }

        Object getPayload()
        {
            return m_payLoad;
        }

        Object removePayload(Object attName, AttributeName nameLowerLevels)
        {
            // The to level of an element is never an AttributeList
            if (!correctType(attName))
            {
                throw new IllegalArgumentException();
            }

            Object nextLevel = get(attName);

            if (nextLevel == null)
            {
                return null;
            }

            // The attribute selected is in this level
            if (nameLowerLevels.getComponentCount() == 0)
            {
                if (nextLevel instanceof AttributeSetNode)
                {
                    return ((AttributeSetNode)nextLevel).removePayload();
                }
                else
                {
                    return ((Payload)remove(attName)).getValue();
                }
            }

            if (!(nextLevel instanceof AttributeSetNode))
            {
                throw new IllegalArgumentException();
            }

            return ((AttributeSetNode)nextLevel).removePayload(nameLowerLevels.getComponent(0), Util.removeFirstComponent(nameLowerLevels));
        }

        Object getPayload(Object attName, AttributeName nameLowerLevels)
        {
            // The to level of an element is never an AttributeList
            if (!correctType(attName))
            {
                throw new IllegalArgumentException();
            }

            Object nextLevel = get(attName);

            if (nextLevel == null)
            {
                return null;
            }

            // The attribute selected is in this level
            if (nameLowerLevels.getComponentCount() == 0)
            {
                if (nextLevel instanceof AttributeSetNode)
                {
                    return ((AttributeSetNode)nextLevel).getPayload();
                }
                else
                {
                    return ((Payload)nextLevel).getValue();
                }
            }


            if (!(nextLevel instanceof AttributeSetNode))
            {
                throw new IllegalArgumentException();
            }

            return ((AttributeSetNode)nextLevel).getPayload(nameLowerLevels.getComponent(0), Util.removeFirstComponent(nameLowerLevels));
        }

        void removeNode(Object attName, AttributeName nameLowerLevels)
        {
            // The to level of an element is never an AttributeList
            if (!correctType(attName))
            {
                throw new IllegalArgumentException();
            }



            // The attribute selected is in this level
            if (nameLowerLevels.getComponentCount() == 0)
            {
                remove(attName);
            }

            Object nextLevel = get(attName);

            if (nextLevel == null)
            {
                return;
            }
            else if (!(nextLevel instanceof AttributeSetNode))
            {
                throw new IllegalArgumentException();
            }

            ((AttributeSetNode)nextLevel).removeNode(nameLowerLevels.getComponent(0), Util.removeFirstComponent(nameLowerLevels));
        }


        void insertPayload(Object attName, AttributeName nameLowerLevels, Object payLoad)
        {
            // The to level of an element is never an AttributeList
            if (!correctType(attName))
            {
                throw new IllegalArgumentException();
            }

            Object insertedBefore = get(attName);

            // The attribute selected is in this level
            if (nameLowerLevels.getComponentCount() == 0)
            {
                if (insertedBefore != null && insertedBefore instanceof AttributeSetNode)
                {
                    ((AttributeSetNode)insertedBefore).insertPayload(payLoad);
                }
                else
                {
                    put(attName, new Payload(payLoad));
                }
                return;
            }

            Object nextComponent = nameLowerLevels.getComponent(0);

            if (insertedBefore == null)
            {
                insertedBefore = (nextComponent instanceof Integer) ? new AttributeListNode(m_level+1) : new AttributeSetNode(m_level+1);
                put(attName, insertedBefore);
            }
            else if (insertedBefore instanceof Payload) 
            {
                 AttributeSetNode replacement = (nextComponent instanceof Integer) ? new AttributeListNode(m_level+1) 
                                                                                   : new AttributeSetNode(m_level+1);
                 replacement.insertPayload(((Payload)insertedBefore).getValue());
                 put(attName, replacement);
                 insertedBefore = replacement;
            }

            ((AttributeSetNode)insertedBefore).insertPayload(nextComponent, Util.removeFirstComponent(nameLowerLevels), payLoad);
        }

        boolean correctType(Object nameComponent)
        {
            return (nameComponent instanceof String);
        }

    }

    private final class AttributeListNode extends AttributeSetNode
    {
        AttributeListNode()
        {
            this(0);
        }

        AttributeListNode(int level)
        {
            super(level);
        }

        @Override
        boolean correctType(Object nameComponent)
        {
            return (nameComponent instanceof Integer);
        }
    }

    private final class Payload
    {
        Object m_value;

        Payload(Object value)
        {
            m_value = value;
        }

        Object getValue()
        {
            return m_value;
        }
    }

}
