/*
 * Copyright (c) 2001 Sonic Software. All Rights Reserved.
 */

package com.sonicsw.mf.common.config.impl;
import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Set;

import com.sonicsw.mf.common.config.AttributeSetTypeException;
import com.sonicsw.mf.common.config.ConfigException;
import com.sonicsw.mf.common.config.IAttributeChangeHandler;
import com.sonicsw.mf.common.config.IAttributeList;
import com.sonicsw.mf.common.config.IAttributeMetaData;
import com.sonicsw.mf.common.config.IAttributeSet;
import com.sonicsw.mf.common.config.IAttributeSetType;
import com.sonicsw.mf.common.config.ITypeCollection;
import com.sonicsw.mf.common.config.ReadOnlyException;
import com.sonicsw.mf.common.config.Reference;
import com.sonicsw.mf.common.config.query.AttributeName;
import com.sonicsw.mf.common.view.ILogicalNameSpace;


final class AttributeList
extends ElementNode
implements IAttributeList, java.io.Serializable, ICanReplaceRef
{
  private static final long serialVersionUID = 0L;
  private final static int SERIALIZATION_VERSION = 3;

  private static final int DELETED_ITEM = 1;
  private static final int MODIFIED_ITEM = 2;

  private TypeCollection m_typeCollection;
  private ArrayList m_list;

  private transient HashMap m_modHistory;
  private transient OrdinalMap m_newOldMap;

  AttributeList(String name, ElementNode parent)
  {
      super(name, parent);

      m_typeCollection = null;
      m_list = new ArrayList();
      m_modHistory = null;
      m_newOldMap = null;
  }

  int estimateSize()
  {
      int estimate = Util.ELEMENT_NODE_SIZE;
      if (m_typeCollection != null)
    {
        estimate += m_typeCollection.estimateSize();
    }

      return estimate + Util.estimateArrayListSize(m_list);

  }

  // The new list is returned if the lists are not identical - the specific modifications are lost
  Object createDelta(AttributeList list)
  {
      int thisLength = getCount();
      int thatLength = list.getCount();

      if (thisLength != thatLength)
    {
        return list;
    }

      for (int i = 0; i < thisLength; i++)
      {
          Object thisValue = getItem(i);
          Object thatValue = list.getItem(i);
          Object delta = null;
          if ((thisValue instanceof AttributeSet) && (thatValue instanceof AttributeSet))
        {
            delta = AttributeSet.doTypedDelta(thisValue, thatValue,  getTypeCollection());
        }
        else
        {
            delta = Util.createDelta(thisValue, null, thatValue, null);
        }

           if (delta != null)
        {
            return list;
        }
      }

      return null;
  }

  private void writeObject(java.io.ObjectOutputStream s) throws java.io.IOException
  {
       s.writeInt(SERIALIZATION_VERSION);
       s.writeObject(m_typeCollection);
       s.writeObject(m_list);
  }

  private void readObject(java.io.ObjectInputStream s) throws java.io.IOException, ClassNotFoundException
  {
        int version = s.readInt();
        if (version != SERIALIZATION_VERSION)
        {
            Util.throwSerialVersionMismatch(version, SERIALIZATION_VERSION);
        }
        m_typeCollection = (TypeCollection)s.readObject();
        m_list = (ArrayList)s.readObject();

        m_modHistory = new HashMap();
        m_newOldMap = new OrdinalMap(m_list.size());
  }



  @Override
public void addObjectItem(Object value) throws ReadOnlyException
  {
      addItem(value, true);
  }

  void addItem(Object value, boolean checkType) throws ReadOnlyException
  {
      if (m_readOnly)
    {
        throw new ReadOnlyException();
    }

      if (value == null)
    {
        return;
    }

      if (checkType && !Util.validSimpleValue(value))
    {
        throw new IllegalArgumentException(value.getClass().getName() + " is an invalid type.");
    }

      m_list.add(value);

      if (!isNew())
    {
        markModified(null, false, false);
    }
  }

  private IAttributeSetType modifyAttributeSetType(String typeName, boolean create) throws AttributeSetTypeException, ReadOnlyException, ConfigException
  {
       if (typeName == null || typeName.length() == 0)
    {
        throw new ConfigException("Attribute name cannot be null or 0 length.");
    }

       if (m_readOnly)
    {
        throw new ReadOnlyException();
    }

       IAttributeSetType type = null;
       if (create)
    {
        type = getTypeCollection().createAttributeSetType(typeName);
    }
    else
    {
        type =  getTypeCollection().deleteAttributeSetType(typeName);
    }

       if (isNew())
    {
        return type;
    }

       markTreeModified();

       // Types changed - we will use the entire object rather then delta
       removeDeltaHistory();

       return type;
  }

  private TypeCollection getTypeCollection()
  {
       if (m_typeCollection == null)
    {
        m_typeCollection = new TypeCollection();
    }
       return m_typeCollection;
  }

  private void setItem(int position, Object value, boolean checkEquality) throws IndexOutOfBoundsException, ReadOnlyException
  {
     if (m_readOnly)
    {
        throw new ReadOnlyException();
    }

      validatePosition(position);

      if (value == null)
    {
        deleteAttributeItem(position);
    }

      // If we update with an atomic value equal to the old one then it's a NOOP
      if (checkEquality)
      {
          if (Util.atomicAndEqual(value,  m_list.get(position)))
        {
            return;
        }
      }

      m_list.set(position, value);

      if (!isNew())
    {
        markModified((new Integer(position)).toString(),  m_newOldMap.isOldItem(position), false);
    }
  }


  private void insertItem(int position, Object value) throws IndexOutOfBoundsException, ReadOnlyException
  {
     if (m_readOnly)
    {
        throw new ReadOnlyException();
    }

      validatePosition(position);

      if (value == null)
    {
        return;
    }

      m_list.add(position, value);

      // We don't want to keep too complex delta. So we stop keeping delta history
      // and consider the list as new
      removeDeltaHistory();
  }

  // No resources to delete
  @Override
void delete(){}

  @Override
Object getNameFromParent(Object listItem)
  {
      int position = m_list.indexOf(listItem);
      if (m_newOldMap != null)
    {
        position = m_newOldMap.getOldPosition(position);
    }
      return new Integer(position);
  }


  // Mark as modified this element and all the nodes up the tree
  // elementNum is null if an element is appended to the end of the list
  @Override
void markModified(String elementNum, boolean oldIsModified, boolean deletion)
  {

      // If this list is new then no need to keep history or mark the parents
      if (isNew())
    {
        return;
    }

      boolean isAdded = elementNum == null;

      // No need to add a history event if we append to the list or if we modify or delete
      // an item we just added
      if (!isAdded && oldIsModified)
      {

           // We keep the history using the old oridinal numbers of the list items
           int position = (new Integer(elementNum)).intValue();
           int oldPosition = m_newOldMap.getOldPosition(position);

           if (deletion)
        {
            m_newOldMap.remove(position);
        }

           m_modHistory.put(new Integer(oldPosition), new Integer(deletion ? DELETED_ITEM : MODIFIED_ITEM));

      }

      markTreeModified();
  }

  @Override
public Object getAttribute(AttributeName attributeName)
  {
       int componentCount = attributeName.getComponentCount();
       if (componentCount == 0)
    {
        return null;
    }

       Object firstComponenet = attributeName.getComponent(0);

       // Only Integers are used with AttributeLists
       if (firstComponenet instanceof java.lang.String)
    {
        return null;
    }

       Object thisLevel = null;
       try
       {
           thisLevel = getItem(((Integer)firstComponenet).intValue());
       }
       catch (IndexOutOfBoundsException e)
       {
           return null;
       }

       if (componentCount == 1 || thisLevel == null)
    {
        return thisLevel;
    }

       AttributeName nextLevelsname = com.sonicsw.mf.common.config.query.impl.Util.removeFirstComponent(attributeName);

       if (thisLevel instanceof AttributeSet)
    {
        return ((AttributeSet)thisLevel).getAttribute(nextLevelsname);
    }
    else if (thisLevel instanceof AttributeList)
    {
        return ((AttributeList)thisLevel).getAttribute(nextLevelsname);
    }
    else
    {
        return null;
    }

  }

  @Override
public void removeDeltaHistory()
  {
      m_isNew = true;
      m_newOldMap = null;
      m_modHistory = null;
      m_list.trimToSize();
  }

  private void validatePosition(int position) throws IndexOutOfBoundsException
  {
      if (position < 0 || position >= m_list.size())
    {
        throw new IndexOutOfBoundsException(getFullName() + " list size is: " + m_list.size());
    }
  }


  @Override
public void applyDelta(IDelta delta) throws AttributeSetTypeException
  {

       if (!(delta instanceof DeltaAttributeList))
    {
        throw new Error("Not an attribute list delta.");
    }

        DeltaAttributeList deltaList = (DeltaAttributeList) delta;

        HashMap modifiedItems = deltaList.getModifiedItems();
        int[] deletedItem = deltaList.getDeletedItemNumbers();
        Object[] newItems = deltaList.getNewItems();

        // Add the new items
        for (int i = 0; i < newItems.length; i++)
        {
            Object newValue = newItems[i];

            // We have to create a new set object references our local type object
            if (newValue instanceof AttributeSet)
            {
                ((AttributeSet)newValue).createLocalTypedSet(m_typeCollection);
            }

            if (newValue instanceof ElementNode)
            {
                ((ElementNode)newValue).setNewParent(this);
            }

            m_list.add(newValue);
        }

        // Modify items
        Set set =  modifiedItems.keySet();
        Iterator iterator = set.iterator();
        while (iterator.hasNext())
        {
            Integer position = (Integer)iterator.next();
            Object newValue = modifiedItems.get(position);
            Object oldValue = m_list.get(position.intValue());

            if (newValue instanceof IDelta)
            {
                if (oldValue == null || !(oldValue instanceof IDeltaBookKeeper))
                {
                    throw new Error("Cannot apply the delta to the old value.");
                }
                ((IDeltaBookKeeper)oldValue).applyDelta((IDelta)newValue);
                continue;
            }

            // We have to create a new set object references our local type object
            if (newValue instanceof AttributeSet)
            {
                ((AttributeSet)newValue).createLocalTypedSet(m_typeCollection);
            }

            if (newValue instanceof ElementNode)
            {
                ((ElementNode)newValue).setNewParent(this);
            }

            m_list.set(position.intValue(), newValue);

            if (oldValue instanceof ElementNode)
            {
                ((ElementNode)oldValue).delete();
            }
        }

        // Delete items
        for (int i = 0; i < deletedItem.length; i++)
        {
            Object oldValue = m_list.set(deletedItem[i], null);

            if (oldValue instanceof ElementNode)
            {
                ((ElementNode)oldValue).delete();
            }
        }

        for (int i = m_list.size() - 1; i >= 0 ; i--)
        {
            if (m_list.get(i) == null)
            {
                m_list.remove(i);
            }
        }

        m_list.trimToSize();
  }

  @Override
public int getCount()
  {
      return m_list.size();
  }

  @Override
boolean underModifiedList()
  {
      if (m_isModified)
    {
        return true;
    }
    else
    {
        return super.underModifiedList();
    }
  }

  @Override
public IAttributeMetaData getItemMetaData(int position) throws IndexOutOfBoundsException
  {
      // Will throw IndexOutOfBoundsException if the position is invalid
      getItem(position);

     boolean fromTamplate;

      if (isNew() || m_isModified || underModifiedList())
    {
          // If the list is modified then the whole list is over written from the template  
        fromTamplate = false;
    }
    else
    {
        fromTamplate = checkSubclassingDelta(getCompoundName().setNextComponent(position));
    }

      return new AttributeMetaData(fromTamplate);
  }

  @Override
public Object getItem(int position) throws IndexOutOfBoundsException
  {
      return m_list.get(position);
  }

  @Override
public IAttributeList setNewAttributeListItem(int position) throws IndexOutOfBoundsException, ReadOnlyException
  {
      IAttributeList newList = new AttributeList(null, this);
      setItem(position, newList, false);
      return newList;
  }

  @Override
public IAttributeList insertAttributeListItem(int position) throws IndexOutOfBoundsException, ReadOnlyException
  {
      IAttributeList newList = new AttributeList(null, this);
      insertItem(position, newList);
      return newList;
  }

  @Override
public IAttributeList addNewAttributeListItem() throws ReadOnlyException
  {
      int position = m_list.size();
      IAttributeList newList = new AttributeList(null, this);
      addItem(newList, false);
      return newList;
  }

  @Override
public IAttributeSet setNewAttributeSetItem(int position) throws IndexOutOfBoundsException, ReadOnlyException
  {
      try
      {
          return setNewAttributeSetItem(position, null);
      } catch (AttributeSetTypeException e) {throw new Error();}
  }

  @Override
public IAttributeSet insertAttributeSetItem(int position) throws IndexOutOfBoundsException, ReadOnlyException
  {
      try
      {
          return insertAttributeSetItem(position, null);
      } catch (AttributeSetTypeException e) {throw new Error();}
  }

  @Override
public IAttributeSet addNewAttributeSetItem() throws ReadOnlyException
  {
      try
      {
          return addNewAttributeSetItem(null);
      } catch (AttributeSetTypeException e) {throw new Error();}
  }

  @Override
public IAttributeSet setNewAttributeSetItem(int position, IAttributeSetType type) throws IndexOutOfBoundsException, ReadOnlyException ,
      AttributeSetTypeException
  {
        // Makes sure the type was defined in the scope of this list
        if (type != null)
        {
            m_typeCollection.verifyType((AttributeSetType)type, getFullName());
        }

        // We want to create the new AttributeSet object only after we know we can successfully set the value
        setItem(position, "PLACE HOLDER", false);

        // setAttributeObject didn't throw an exception so we can create a new AttributeSet and set it
        AttributeSet set = new AttributeSet (null, this, (AttributeSetType)type);
        m_list.set(position, set);

        return set;
  }


  @Override
public IAttributeSet insertAttributeSetItem(int position, IAttributeSetType type) throws IndexOutOfBoundsException, ReadOnlyException ,
      AttributeSetTypeException
  {
        // Makes sure the type was defined in the scope of this list
        if (type != null)
        {
            m_typeCollection.verifyType((AttributeSetType)type, getFullName());
        }

        // We want to create the new AttributeSet object only after we know we can successfully set the value
        insertItem(position, "PLACE HOLDER");

        // setAttributeObject didn't throw an exception so we can create a new AttributeSet and set it
        AttributeSet set = new AttributeSet (null, this, (AttributeSetType)type);
        m_list.set(position, set);

        return set;
  }


  @Override
public IAttributeSet addNewAttributeSetItem(IAttributeSetType type) throws ReadOnlyException, AttributeSetTypeException
  {
        // Makes sure the type was defined in the scope of this list
        if (type != null)
        {
            m_typeCollection.verifyType((AttributeSetType)type, getFullName());
        }

        // We want to create the new AttributeSet object only after we know we can successfully set the value
        addItem("PLACE HOLDER", false);

        int position = m_list.size() - 1;

        // setAttributeObject didn't throw an exception so we can create a new AttributeSet and set it
        AttributeSet set = new AttributeSet (null, this, (AttributeSetType)type);
        m_list.set(position, set);

        return set;
  }

  @Override
public Object deleteAttributeItem(int position) throws IndexOutOfBoundsException , ReadOnlyException
  {
     if (m_readOnly)
    {
        throw new ReadOnlyException();
    }

      validatePosition(position);

      Object oldItem = m_list.get(position);

      if (!isNew())
    {
        markModified((new Integer(position)).toString(), m_newOldMap.isOldItem(position), true);
    }

      m_list.remove(position);

      if (oldItem instanceof ElementNode)
    {
        ((ElementNode)oldItem).removeFromTree();
    }

      return oldItem;
   }

   @Override
public ArrayList getItems()
   {
       return (ArrayList)m_list.clone();
   }

    @Override
    public IDelta createDelta(boolean forSubclassing)
    {
         if (isNew())
        {
            throw new Error();
        }

         int[] deletedArray = new int[m_newOldMap.getDeletedCount()];
         HashMap deltaVals = new HashMap();

         Set set =  m_modHistory.keySet();
         Iterator iterator = set.iterator();
         int deletedIndex = 0;
         while (iterator.hasNext())
         {
             Integer oldPosition = (Integer)iterator.next();
             int historyEvent = ((Integer)m_modHistory.get(oldPosition)).intValue();
             if (historyEvent == DELETED_ITEM)
            {
                deletedArray[deletedIndex++] = oldPosition.intValue();
            }
            else
             {
                 int newPosition = m_newOldMap.getNewPosition(oldPosition.intValue());
                 Object newValue = m_list.get(newPosition);
                 if (newValue instanceof IDeltaBookKeeper)
                 {
                      IDeltaBookKeeper deltaKeeper = (IDeltaBookKeeper)newValue;

                      // We don't support DeltaAttributeList for subclassing, instead we just overwrite the whole list
                      if (forSubclassing && (newValue instanceof AttributeList))
                    {
                        deltaKeeper.removeDeltaHistory();
                    }

                      if (deltaKeeper.isNew())
                      {
                          if (deltaKeeper instanceof ElementNode)
                        {
                            deltaKeeper = (IDeltaBookKeeper)((ElementNode)deltaKeeper).cloneWithoutParent();
                        }
                          deltaVals.put(oldPosition, deltaKeeper);
                      }
                    else
                    {
                        deltaVals.put(oldPosition, deltaKeeper.createDelta(forSubclassing));
                    }
                 }
                 else
                 {
                     if (newValue instanceof ElementNode)
                    {
                        newValue = ((ElementNode)newValue).cloneWithoutParent();
                    }
                   deltaVals.put(oldPosition, newValue);
                 }
             }
         }

         int numNewItems = m_list.size() - m_newOldMap.numOldLeft();
         Object[] newItems = new Object[numNewItems];

         int newIndex = 0;
         for (int i = m_list.size() - numNewItems; i <  m_list.size(); i++)
        {
            newItems[newIndex++] = m_list.get(i);
        }

         return new DeltaAttributeList(deletedArray, deltaVals, newItems);

    }

    @Override
    public void setReadOnly(boolean readOnly)
    {
        m_readOnly = readOnly;

        if (m_typeCollection != null)
        {
            m_typeCollection.setReadOnly(readOnly);
        }

        if (readOnly)
        {
            m_list.trimToSize();
        }
        for (int i = 0; i < m_list.size(); i++)
        {
            Object value = m_list.get(i);
            if (value instanceof ElementNode)
            {
                ((ElementNode)value).setReadOnly(readOnly);
            }
        }

    }

    @Override
    public IAttributeSetType createAttributeSetType(String typeName) throws AttributeSetTypeException, ReadOnlyException , ConfigException
    {
         return modifyAttributeSetType(typeName, true);
    }

    @Override
    public IAttributeSetType deleteAttributeSetType(String typeName) throws AttributeSetTypeException, ReadOnlyException, ConfigException
    {
        return modifyAttributeSetType(typeName, false);
    }


    @Override
    public boolean typesEqual(ITypeCollection c)
    {
        if (!(c instanceof AttributeList))
        {
            return false;
        }
        return getTypeCollection().typesEqual(((AttributeList)c).getTypeCollection());
    }

    @Override
    public String[] getAllTypeNames()
    {
        return getTypeCollection().getAllTypeNames();
    }

    @Override
    public IAttributeSetType getAttributeSetType(String typeName)
    {
        return getTypeCollection().getAttributeSetType(typeName);
    }


    // Map between old and new list items oridinal number
    static class OrdinalMap
    {
        private ArrayList newToOldMap;
        private int[] oldToNewMap;
        private int lastOldItem;

        OrdinalMap(int sizeofOld)
        {
            newToOldMap = new ArrayList(sizeofOld);
            for (int i = 0; i < sizeofOld; i++)
            {
                newToOldMap.add(new Integer(i));
            }

            oldToNewMap = new int[sizeofOld];
            for (int i = 0; i < sizeofOld; i++)
            {
                oldToNewMap[i] = i;
            }

             lastOldItem = sizeofOld - 1;
        }

        int getOldPosition(int newPosition)
        {
            return ((Integer)newToOldMap.get(newPosition)).intValue();
        }

        int getNewPosition(int oldPosition)
        {
            return oldToNewMap[oldPosition];
        }

        int getDeletedCount()
        {
            return oldToNewMap.length - lastOldItem - 1;
        }

        int numOldLeft()
        {
            return oldToNewMap.length - getDeletedCount();
        }

        boolean isOldItem(int position)
        {
            return position <= lastOldItem;
        }

        void remove(int position)
        {
            lastOldItem--;


            // Update new-to-old map
            newToOldMap.remove(position);

            // Update old-to-new map
            boolean deleted = false;
            for (int i = 0; i < oldToNewMap.length; i++)
            {
                if (deleted)
                {
                    oldToNewMap[i]--;
                }

                if (position == oldToNewMap[i])
                {
                    oldToNewMap[i] = -1;
                    deleted = true;
                }

            }
        }
    }

  @Override
public void setNewStringItem(int position, String value) throws IndexOutOfBoundsException, ReadOnlyException
  {
      setItem(position, value, true);
  }

  @Override
public void setNewIntegerItem(int position, Integer value) throws IndexOutOfBoundsException, ReadOnlyException
  {
      setItem(position, value, true);
  }

  @Override
public void setNewLongItem(int position, Long value) throws IndexOutOfBoundsException, ReadOnlyException
  {
      setItem(position, value, true);
  }

  @Override
public void setNewDecimalItem(int position, BigDecimal value) throws IndexOutOfBoundsException, ReadOnlyException
  {
      // Temporary code to catch the code that enters a junk BigDecimal object toString() will throw an exception
      try
      {
          if (value != null)
        {
            value.toString();
        }
      }
      catch (Throwable t)
      {
          t.printStackTrace();
          //throw new Error(t.toString());
      }
       

      setItem(position, value, true);
  }

  @Override
public void setNewBytesItem(int position, byte[] value) throws IndexOutOfBoundsException, ReadOnlyException
  {
      setItem(position, value, true);
  }

  @Override
public void setNewBooleanItem(int position, Boolean value) throws IndexOutOfBoundsException, ReadOnlyException
  {
      setItem(position, value, true);
  }

  @Override
public void setNewReferenceItem(int position, Reference value) throws IndexOutOfBoundsException, ReadOnlyException
  {
      setItem(position, value, true);
  }

  @Override
public void setNewDateItem(int position, Date value) throws IndexOutOfBoundsException, ReadOnlyException
  {
      setItem(position, value, true);
  }


  @Override
public void insertStringItem(int position, String value) throws IndexOutOfBoundsException, ReadOnlyException
  {
      insertItem(position, value);
  }

  @Override
public void insertIntegerItem(int position, Integer value) throws IndexOutOfBoundsException, ReadOnlyException
  {
      insertItem(position, value);
  }

  @Override
public void insertLongItem(int position, Long value) throws IndexOutOfBoundsException, ReadOnlyException
  {
      insertItem(position, value);
  }

  @Override
public void insertDecimalItem(int position, BigDecimal value) throws IndexOutOfBoundsException, ReadOnlyException
  {
      // Temporary code to catch the code that enters a junk BigDecimal object toString() will throw an exception
      try
      {
          if (value != null)
        {
            value.toString();
        }
      }
      catch (Throwable t)
      {
          t.printStackTrace();
          //throw new Error(t.toString());
      }

      insertItem(position, value);
  }

  @Override
public void insertBytesItem(int position, byte[] value) throws IndexOutOfBoundsException, ReadOnlyException
  {
      insertItem(position, value);
  }

  @Override
public void insertBooleanItem(int position, Boolean value) throws IndexOutOfBoundsException, ReadOnlyException
  {
      insertItem(position, value);
  }

  @Override
public void insertReferenceItem(int position, Reference value) throws IndexOutOfBoundsException, ReadOnlyException
  {
      insertItem(position, value);
  }

  @Override
public void insertDateItem(int position, Date value) throws IndexOutOfBoundsException, ReadOnlyException
  {
      insertItem(position, value);
  }


  @Override
public void addStringItem(String value) throws ReadOnlyException
  {
      addItem(value, false);
  }

  @Override
public void addIntegerItem(Integer value) throws ReadOnlyException
  {
      addItem(value, false);
  }

  @Override
public void addLongItem(Long value) throws ReadOnlyException
  {
      addItem(value, false);
  }

  @Override
public void addDecimalItem(BigDecimal value) throws ReadOnlyException
  {
      // Temporary code to catch the code that enters a junk BigDecimal object toString() will throw an exception
      try
      {
          if (value != null)
        {
            value.toString();
        }
      }
      catch (Throwable t)
      {
          t.printStackTrace();
          //throw new Error(t.toString());
      }

      addItem(value, false);
  }

  @Override
public void addBytesItem(byte[] value) throws ReadOnlyException
  {
      addItem(value, false);
  }

  @Override
public void addBooleanItem(Boolean value) throws ReadOnlyException
  {
      addItem(value, false);
  }

  @Override
public void addReferenceItem(Reference value) throws ReadOnlyException
  {
      addItem(value, false);
  }

  @Override
public void addDateItem(Date value) throws ReadOnlyException
  {
      addItem(value, false);
  }

  @Override
public void registerAttributeChangeHandler(Object context, IAttributeChangeHandler handler)
  {
      validateRegistrationContext(context);

      ((IChangeRegistration)context).registerAttributeChangeHandler(getCompoundName(), handler);
  }

  @Override
public void unregisterAttributeChangeHandler(Object context)
  {
      validateRegistrationContext(context);

      ((IChangeRegistration)context).unregisterAttributeChangeHandler(getCompoundName());
  }

  @Override
public void registerAttributeChangeHandler(Object context, int position, IAttributeChangeHandler handler)
  {
      validateRegistrationContext(context);

      // Validation  - Will throw IndexOutOfBoundsException if there is no item in this position
      m_list.get(position);

      ((IChangeRegistration)context).registerAttributeChangeHandler(getCompoundName().setNextComponent(position), handler);
  }

  @Override
public void unregisterAttributeChangeHandler(Object context, int position)
  {
      validateRegistrationContext(context);

      ((IChangeRegistration)context).unregisterAttributeChangeHandler(getCompoundName().setNextComponent(position));
  }

  private void validateRegistrationContext(Object context)
  {
      if (context == null || !(context instanceof IChangeRegistration))
    {
        throw new IllegalArgumentException("The context object is invalid.");
    }
  }

  @Override
public boolean replaceReferences(boolean isSystemAtts, IReplaceRef replaceSrvc)
  {
       boolean retVal = true;
       for (int i = 0; i < getCount(); i++)
       {
           Object value = getItem(i);
           if (value instanceof ICanReplaceRef)
           {
               boolean ret = ((ICanReplaceRef)value).replaceReferences(false, replaceSrvc);
               if (!ret)
            {
                retVal = false;
            }
           }
           else if (value instanceof Reference)
           {
                Reference newRef = replaceSrvc.replace((Reference)value);
                if (newRef.getElementName().startsWith(ILogicalNameSpace.NO_STORAGE_LABEL))
                {
                    retVal = false;
                }
                m_list.set(i,  newRef);
           }
        else
        {
            continue;
        }
       }
       return retVal;
  }



}
