/*
 * Copyright (c) 2001 Sonic Software. All Rights Reserved.
 */

package com.sonicsw.mf.common.config.impl;
import java.math.BigDecimal;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Set;

import com.sonicsw.mf.common.config.AttributeSetTypeException;
import com.sonicsw.mf.common.config.ConfigException;
import com.sonicsw.mf.common.config.IAttributeChangeHandler;
import com.sonicsw.mf.common.config.IAttributeList;
import com.sonicsw.mf.common.config.IAttributeMetaData;
import com.sonicsw.mf.common.config.IAttributeSet;
import com.sonicsw.mf.common.config.IAttributeSetType;
import com.sonicsw.mf.common.config.ITypeCollection;
import com.sonicsw.mf.common.config.ReadOnlyException;
import com.sonicsw.mf.common.config.Reference;
import com.sonicsw.mf.common.config.query.AttributeName;
import com.sonicsw.mf.common.view.ILogicalNameSpace;


public final class AttributeSet
extends ElementNode
implements IAttributeSet, java.io.Serializable, ICanReplaceRef
{
    private static final long serialVersionUID = 0L;
    private final static int SERIALIZATION_VERSION = 3;

    public static final String RESERVED_CHARACTERS = "\\";
    public static final String ATTRIBUTE_SEPARATOR = "|";
    public static final String[] EMPTY_STRING_ARRAY = new String[0];

    final static int NO_HISTORY = 0;
    final static int DELETED_ATTRIBUTE = 1;
    final static int NEW_ATTRIBUTE = 2;
    final static int MODIFIED_ATTRIBUTE = 3;

    private TypeCollection m_typeCollection;
    private IDictionary m_dictionary;
    private String m_typeName;
    private transient HashMap m_modHistory;
    private transient HashSet m_mergeUtilHint_deletedSet;


    // This method is static so it can be used from AttributeList which passes its typeCollection context
    // to it
    static Object doTypedDelta(Object thisValue, Object thatValue, TypeCollection typeCollection)
    {
           String thisTypeName = ((AttributeSet)thisValue).getTypeName();
           String thatTypeName = ((AttributeSet)thatValue).getTypeName();

           AttributeSetType thisType = (thisTypeName == null) ? null :
                                      (AttributeSetType)typeCollection.getAttributeSetType(thisTypeName);

           AttributeSetType thatType = (thatTypeName == null) ? null :
                                      (AttributeSetType)typeCollection.getAttributeSetType(thatTypeName);

           // Now we have the types and we can create the delta between the thisValue and thatValue
           return Util.createDelta(thisValue, thisType, thatValue, thatType);
    }


    AttributeSet (String name, ElementNode parent, AttributeSetType type)
    {
        super(name, parent);
        m_modHistory = null;
        m_typeCollection =  null;
        if (type == null)
        {
            m_dictionary = new UntypedAttSet();
            m_typeName = null;
        }
        else
        {
            m_dictionary = type.createTypedSet();
            m_typeName = type.getName();
        }
    }

    AttributeSet (String name, ElementNode parent, String user, byte[] password)
    {
        super(name, parent, false);
        m_modHistory = new HashMap();
        m_typeCollection =  null;
        m_dictionary = new UntypedAttSet(user, password);
        m_typeName = null;
    }

    Object createDelta(AttributeSetType myType, AttributeSet attributeSet, AttributeSetType attributeSetType)
    {
       // Types must be identical for equality
       if (myType != null)
       {
           if (!myType.equals(attributeSetType))
        {
            return attributeSet;
        }
       }
       else
       {
           if (attributeSetType != null)
        {
            return attributeSet;
        }
       }

       // Type collection defined for the two sets must be equal
       if (!getTypeCollection().equals(attributeSet.getTypeCollection()))
    {
        return attributeSet;
    }

       HashMap modHistory = new HashMap();
       HashMap deltaValues = new HashMap();

       // Find all the attributes which were deleted or modified
       String[] myAttributes = m_dictionary.attributes();
       for (int i = 0; i < myAttributes.length; i++)
       {
           Object thisValue = getAttribute(myAttributes[i]);
           Object thatValue = attributeSet.getAttribute(myAttributes[i]);

           if (thisValue != null && thatValue == null)
           {
               modHistory.put(myAttributes[i], new Integer(AttributeSet.DELETED_ATTRIBUTE));
               continue;
           }

           if (thisValue == null && thatValue == null)
         {
            continue; // Nothing changed
        }

           if (thisValue == null && thatValue != null)
         {
            continue; // We will take care of the new attribute case later
        }

           Object delta = null;
           if ((thisValue instanceof AttributeSet) && (thatValue instanceof AttributeSet))
        {
            delta = doTypedDelta(thisValue, thatValue,  getTypeCollection());
        }
        else
        {
            delta = Util.createDelta(thisValue, null, thatValue, null);
        }

           if (delta != null)
           {
               modHistory.put(myAttributes[i], new Integer(AttributeSet.MODIFIED_ATTRIBUTE));
               deltaValues.put(myAttributes[i], delta);
           }
       }

       // Now find all the new attributes
       String[] hisAttributes = attributeSet.m_dictionary.attributes();
       for (int i = 0; i < hisAttributes.length; i++)
       {
           Object thisValue = getAttribute(hisAttributes[i]);
           Object thatValue = attributeSet.getAttribute(hisAttributes[i]);

           // Taking care of the case of a new wttribute - all other cases are already covered above
           if (thisValue == null && thatValue != null)
           {
               modHistory.put(hisAttributes[i], new Integer(AttributeSet.NEW_ATTRIBUTE));
               deltaValues.put(hisAttributes[i], thatValue);
           }
       }

       if (!modHistory.isEmpty())
    {
        return new DeltaAttributeSet(m_typeName, modHistory, deltaValues);
    }
    else
    {
        return null;
    }

    }

    public int estimateSize()
    {
        int estimate = Util.ELEMENT_NODE_SIZE + Util.estimateSize(m_typeName);
        if (m_typeCollection != null)
        {
            estimate += m_typeCollection.estimateSize();
        }
        if (m_dictionary != null)
        {
            estimate += m_dictionary.estimateSize();
        }

        return estimate;
    }

    private void writeObject(java.io.ObjectOutputStream s) throws java.io.IOException
    {
         s.writeInt(SERIALIZATION_VERSION);
         s.writeObject(m_typeCollection);
         s.writeObject(m_dictionary);
         s.writeObject(m_typeName);
    }

    private void readObject(java.io.ObjectInputStream s) throws java.io.IOException, ClassNotFoundException
    {
        int version = s.readInt();
        if (version != SERIALIZATION_VERSION)
        {
            Util.throwSerialVersionMismatch(version, SERIALIZATION_VERSION);
        }

        m_typeCollection = (TypeCollection)s.readObject();
        m_dictionary = (IDictionary)s.readObject();
        m_typeName = (String)s.readObject();
        m_modHistory = new HashMap();
    }


    void createLocalTypedSet(TypeCollection typeCollection)
    {
        if (!isTyped())
        {
            return;
        }

        m_dictionary = typeCollection.cloneInstance((TypedAttSet)m_dictionary);
    }

    // Items in an SttributeSet should not call this
    @Override
    Object getNameFromParent(Object listItem)
    {
        throw new Error();
    }


    private boolean isTyped()
    {
        return (m_dictionary instanceof TypedAttSet);
    }

    @Override
    public String getTypeName()
    {
        return m_typeName;
    }


    @Override
    public void setReadOnly(boolean readOnly)
    {
        m_readOnly = readOnly;

       if (m_typeCollection != null)
    {
        m_typeCollection.setReadOnly(readOnly);
    }

        String[] attributes = m_dictionary.attributes();
        for (int i = 0; i < attributes.length; i++)
        {
            Object value = null;
            try
            {
                value = m_dictionary.get(attributes[i]);
            } catch (AttributeSetTypeException e){}
            if (value != null && value instanceof ElementNode)
            {
                ((ElementNode)value).setReadOnly(readOnly);
            }
        }

    }

    @Override
    public void removeDeltaHistory()
    {
        m_isNew = true;
        m_modHistory = null;
    }

    @Override
    public void applyDelta(IDelta delta) throws AttributeSetTypeException
    {

        if (!(delta instanceof DeltaAttributeSet))
        {
            throw new Error("Not an attribute set delta.");
        }

        DeltaAttributeSet deltaAtt = (DeltaAttributeSet) delta;

        String[] newAttributes = deltaAtt.getNewAttributesNames();
        String[] deletedAttributes = deltaAtt.getDeletedAttributesNames();
        String[] modifiedAttributes = deltaAtt.getModifiedAttributesNames();


        // Add the new attributes
        for (int i = 0; i < newAttributes.length; i++)
        {
            String attribute = newAttributes[i];
            Object value = deltaAtt.getValue(attribute);

            // We have to create a new set object references our local type object
            if (value instanceof AttributeSet)
            {
                ((AttributeSet)value).createLocalTypedSet(m_typeCollection);
            }

            if (value instanceof ElementNode)
            {
                ((ElementNode)value).setNewParent(this);
            }

            m_dictionary.put(attribute, value);
        }

        // Delete attributes
        for (int i = 0; i < deletedAttributes.length; i++)
        {
            Object oldValue = m_dictionary.remove(deletedAttributes[i]);

            if (oldValue instanceof ElementNode)
            {
                ((ElementNode)oldValue).removeFromTree();
            }
        }

        // Modify attributes
        for (int i = 0; i < modifiedAttributes.length; i++)
        {
            String attribute = modifiedAttributes[i];
            Object newValue = deltaAtt.getValue(attribute);

            if (newValue instanceof IDelta)
            {
                Object oldValue = m_dictionary.get(attribute);
                if (oldValue == null || !(oldValue instanceof IDeltaBookKeeper))
                {
                    throw new Error("Cannot apply the delta to the old value.");
                }
                ((IDeltaBookKeeper)oldValue).applyDelta((IDelta)newValue);
                continue;
            }

            // We have to create a new set object references our local type object
            if (newValue instanceof AttributeSet)
            {
                ((AttributeSet)newValue).createLocalTypedSet(m_typeCollection);
            }

            if (newValue instanceof ElementNode)
            {
                ((ElementNode)newValue).setNewParent(this);
            }

            Object oldValue = m_dictionary.put(attribute, newValue);

            if (oldValue != null && oldValue instanceof ElementNode)
            {
                ((ElementNode)oldValue).delete();
            }
        }

    }

    @Override
    public IDelta createDelta(boolean forSubclassing)
    {
        if (isNew())
        {
            throw new Error();
        }

        HashMap deltaVals = new HashMap();
        String[] attributes = m_dictionary.attributes();

        // Populate the deltaVals (delta values)
        for (int i = 0; i < attributes.length; i++)
        {
           String attribute = attributes[i];
           Integer history = (Integer)m_modHistory.get(attribute);
           if (history == null)
           {
        	   continue;
           }

           Object actualValue = null;
           try
           {
               actualValue = m_dictionary.get(attribute);
           } catch (AttributeSetTypeException e) 
           { 
               throw new Error(e); 
           }

			switch (history.intValue()) {
				case DELETED_ATTRIBUTE:
					break;
				case NEW_ATTRIBUTE:
					if (actualValue instanceof ElementNode) {
						actualValue = ((ElementNode) actualValue)
								.cloneWithoutParent();
					}
					deltaVals.put(attribute, actualValue);
					break;
				case MODIFIED_ATTRIBUTE:
					if (actualValue instanceof IDeltaBookKeeper) {
						IDeltaBookKeeper deltaKeeper = (IDeltaBookKeeper) actualValue;
	
						// We don't support DeltaAttributeList for subclassing,
						// instead we just overwrite the whole list
						if (forSubclassing
								&& (actualValue instanceof AttributeList)) {
							deltaKeeper.removeDeltaHistory();
						}
	
						if (deltaKeeper.isNew()) {
							if (deltaKeeper instanceof ElementNode) {
								deltaKeeper = (IDeltaBookKeeper) ((ElementNode) deltaKeeper)
										.cloneWithoutParent();
							}
							deltaVals.put(attribute, deltaKeeper);
						} else {
							deltaVals.put(attribute,
									deltaKeeper.createDelta(forSubclassing));
						}
					} else {
						if (actualValue instanceof ElementNode) {
							actualValue = ((ElementNode) actualValue)
									.cloneWithoutParent();
						}
						deltaVals.put(attribute, actualValue);
					}
					break;
				default:
					throw new Error();
			}
        }

        return new DeltaAttributeSet(m_typeName, m_modHistory, deltaVals);
    }

    private TypeCollection getTypeCollection()
    {
       if (m_typeCollection == null)
    {
        m_typeCollection = new TypeCollection();
    }
       return m_typeCollection;
    }


    private static int attributeSetHistoryTable(boolean valueExists, int previuosHistory)
    {
        switch (previuosHistory)
        {
            case NO_HISTORY:          if (valueExists)
            {
                return MODIFIED_ATTRIBUTE;
            }
            else
            {
                return NEW_ATTRIBUTE;
            }

            case DELETED_ATTRIBUTE:   return MODIFIED_ATTRIBUTE;

            case NEW_ATTRIBUTE:       return NEW_ATTRIBUTE;

            case MODIFIED_ATTRIBUTE:  return MODIFIED_ATTRIBUTE;
            default: throw new Error();
        }
    }

    private static int attributeDeleteHistoryTable(boolean valueExists, int previuosHistory)
    {
        switch (previuosHistory)
        {
            case NO_HISTORY:          if (valueExists)
            {
                return DELETED_ATTRIBUTE;
            }
            else
            {
                return NO_HISTORY;
            }

            case DELETED_ATTRIBUTE:   return DELETED_ATTRIBUTE;

            case NEW_ATTRIBUTE:       return NO_HISTORY;

            case MODIFIED_ATTRIBUTE:  return DELETED_ATTRIBUTE;
            default: throw new Error();
        }
    }

    private void verifyMod(String attributeName) throws ReadOnlyException, ConfigException
    {

        if (EntityName.containsAnyChar(attributeName, RESERVED_CHARACTERS))
        {
            throw new ConfigException("Invalid name '" + attributeName + "'. " +
                                      "Attribute and type names cannot contain the characters: "
                                      + RESERVED_CHARACTERS);
        }

        if (attributeName == null || attributeName.length() == 0)
        {
            throw new ConfigException("Attribute name cannot be null or 0 length.");
        }

        if (m_readOnly)
        {
            throw new ReadOnlyException();
        }
    }

    // Mark as modified this attribute and all the nodes up the tree
    @Override
    void markModified(String attributeName, boolean oldValueExists, boolean deletion)
    {

        // If this attribute set is new then no need to keep history
        if (isNew())
        {
            return;
        }

        Integer history =  (Integer)m_modHistory.get(attributeName);
        int newHistoryEvent =  deletion ?
                                          attributeDeleteHistoryTable(oldValueExists, ((history == null) ? NO_HISTORY : history.intValue()))
                                        :
                                          attributeSetHistoryTable(oldValueExists, ((history == null) ? NO_HISTORY : history.intValue()));

        // Update the history
        if (newHistoryEvent == NO_HISTORY)
        {
            m_modHistory.remove(attributeName);
        }
        else
        {
            m_modHistory.put(attributeName, new Integer(newHistoryEvent) );
        }

        markTreeModified();
    }

    private IAttributeSetType modifyAttributeSetType(String typeName, boolean create) throws AttributeSetTypeException, ReadOnlyException, ConfigException
    {
        verifyMod(typeName);

        IAttributeSetType type = null;
        if (create)
        {
            type = getTypeCollection().createAttributeSetType(typeName);
        }
        else
        {
            type =  getTypeCollection().deleteAttributeSetType(typeName);
        }

        if (isNew())
        {
            return type;
        }

        markTreeModified();

        // Types changed - we will use the entire object rather then delta
        removeDeltaHistory();

        return type;
    }

    @Override
    void delete()
    {
        m_dictionary.delete();
    }

    @Override
    public void setObjectAttribute(String attributeName, Object value)
        throws ReadOnlyException, AttributeSetTypeException, ConfigException
    {
        setAttributeObject(attributeName, value, true);
    }

    // To bypass history recording - used for certain _MF_SYSTEM attributes
    void setAttributeObjectNoHistory(String attributeName, Object value)
        throws ReadOnlyException, AttributeSetTypeException, ConfigException
    {
        if (value == null)
        {
            throw new IllegalArgumentException("setAttributeObjectNoHistory: null value");
        }

        if (!Util.validSimpleValue(value))
        {
            throw new AttributeSetTypeException(value.getClass().getName() + " is an invalid type.");
        }

        m_dictionary.put(attributeName, value);
    }
    // To bypass history recording - used for certain _MF_SYSTEM attributes
    void renameNoHistory(String oldName, String newName)
        throws ReadOnlyException, AttributeSetTypeException, ConfigException
    {
        Object value = m_dictionary.remove(oldName);
        m_dictionary.put(newName, value);
    }


    void setAttributeObject(String attributeName, Object value, boolean checkType)
        throws ReadOnlyException, AttributeSetTypeException, ConfigException
    {
        verifyMod(attributeName);

        // Setting null is equivalent to deleting
        if (value == null)
        {
            deleteAttribute(attributeName);
            return;
        }

        if (checkType && !Util.validSimpleValue(value))
        {
            throw new AttributeSetTypeException(value.getClass().getName() + " is an invalid type.");
        }

        Object oldValue =  m_dictionary.get(attributeName);
        boolean oldValueExists = oldValue != null;

        // Updating with the same value is a NOOP
        if (oldValueExists)
        {
            if (Util.atomicAndEqual(oldValue, value))
            {
                return;
            }
        }

        // Update the value
        m_dictionary.put(attributeName, value);

        // Sets a new history event
        markModified(attributeName, oldValueExists, false);
    }

    @Override
    public IAttributeSet createAttributeSet(String attributeName, IAttributeSetType type)
        throws ReadOnlyException, AttributeSetTypeException, ConfigException
    {
        // Makes sure the type was defined in the scope of this attribute set
        if (type != null)
        {
            m_typeCollection.verifyType((AttributeSetType)type, getFullName());
        }

        // We want to create the new AttributeSet object only after we know we can successfully set the value
        setAttributeObject(attributeName, "PLACE_HOLDER", false);

        // setAttributeObject didn't throw an exception so we can create a new AttributeSet and set it
        AttributeSet set = new AttributeSet (attributeName, this, (AttributeSetType)type);
        m_dictionary.put(attributeName, set);

        return set;
    }


    @Override
    public IAttributeList createAttributeList(String attributeName) throws ReadOnlyException, AttributeSetTypeException, ConfigException
    {
        IAttributeList newList = new AttributeList(attributeName, this);
        setAttributeObject(attributeName, newList, false);
        return newList;
    }

    @Override
    public IAttributeSet createAttributeSet(String attributeName) throws ReadOnlyException, AttributeSetTypeException, ConfigException
    {
        return createAttributeSet(attributeName, null);
    }

    @Override
    public Object deleteAttribute(String attributeName) throws ReadOnlyException, ConfigException
    {
        verifyMod(attributeName);

        boolean oldValueExists = m_dictionary.get(attributeName) != null;

        // Remove the value
        Object oldValue = m_dictionary.remove(attributeName);

        // Sets a new history event
        markModified(attributeName, oldValueExists, true);

        if (oldValue instanceof ElementNode)
        {
            ((ElementNode)oldValue).removeFromTree();
        }

        return oldValue;
    }

    @Override
    public Object getAttribute(AttributeName attributeName)
    {
         int componentCount = attributeName.getComponentCount();
         if (componentCount == 0)
        {
            return null;
        }

         Object firstComponenet = attributeName.getComponent(0);

         // Integers are used only with AttributeLists
         if (firstComponenet instanceof java.lang.Integer)
        {
            return null;
        }

         Object thisLevel = getAttribute((String)firstComponenet);

         if (componentCount == 1 || thisLevel == null)
        {
            return thisLevel;
        }

         AttributeName nextLevelsname = com.sonicsw.mf.common.config.query.impl.Util.removeFirstComponent(attributeName);

         if (thisLevel instanceof AttributeSet)
        {
            return ((AttributeSet)thisLevel).getAttribute(nextLevelsname);
        }
        else if (thisLevel instanceof AttributeList)
        {
            return ((AttributeList)thisLevel).getAttribute(nextLevelsname);
        }
        else
        {
            return null;
        }

    }

    @Override
    public HashMap getAttributes()
    {
        String[] attributes = m_dictionary.attributes();
        HashMap map = new HashMap();
        for (int i = 0; i < attributes.length; i++)
        {
            Object value = null;
            try
            {
                value = m_dictionary.get(attributes[i]);
            }
            catch (AttributeSetTypeException e)
            {
                throw new Error(e);
            }
            if (value != null)
            {
                map.put(attributes[i], value);
            }
        }
        return map;
    }


    @Override
    public IAttributeSetType createAttributeSetType(String typeName) throws AttributeSetTypeException, ReadOnlyException , ConfigException
    {
         return modifyAttributeSetType(typeName, true);
    }

    @Override
    public IAttributeSetType deleteAttributeSetType(String typeName) throws AttributeSetTypeException, ReadOnlyException, ConfigException
    {
        return modifyAttributeSetType(typeName, false);
    }

    @Override
    public boolean typesEqual(ITypeCollection c)
    {
        if (!(c instanceof AttributeSet))
        {
            return false;
        }
        return getTypeCollection().typesEqual(((AttributeSet)c).getTypeCollection());
    }

    @Override
    public String[] getAllTypeNames()
    {
        return getTypeCollection().getAllTypeNames();
    }

    @Override
    public IAttributeSetType getAttributeSetType(String typeName)
    {
        return getTypeCollection().getAttributeSetType(typeName);
    }

    @Override
    public Object getAttribute(String attName)
    {
        try
        {
            return m_dictionary.get(attName);
        }
        catch (AttributeSetTypeException e) 
        {
            return null;
        }
    }



  @Override
public IAttributeMetaData getAttributeMetaData(String attName)
  {
      if (getAttribute(attName) == null)
    {
        return null;
    }

      boolean fromTamplate;

      if (isNew())
    {
        fromTamplate = false;
    }
    else if (m_modHistory != null && m_modHistory.get(attName) != null)
    {
        fromTamplate = false;
    }
    else if (underModifiedList())
    {
        fromTamplate = false;
    }
    else
    {
        fromTamplate = checkSubclassingDelta(getCompoundName().setNextComponent(attName));
    }

      return new AttributeMetaData(fromTamplate);
  }

  @Override
public String[] getDeletedAttributesInThisSubclassed()
  {
      return checkSubclassingDeletedAttributes(getCompoundName());
  }


  @Override
public void setIntegerAttribute(String attributeName, Integer value) throws ReadOnlyException, AttributeSetTypeException, ConfigException
  {
       setAttributeObject(attributeName, value, false);
  }


  @Override
public void setLongAttribute(String attributeName, Long value) throws ReadOnlyException, AttributeSetTypeException, ConfigException
  {
       setAttributeObject(attributeName, value, false);
  }

  @Override
public void setStringAttribute(String attributeName, String value) throws ReadOnlyException, AttributeSetTypeException, ConfigException
  {
       setAttributeObject(attributeName, value, false);
  }

  @Override
public void setDecimalAttribute(String attributeName, BigDecimal value) throws ReadOnlyException, AttributeSetTypeException, ConfigException
  {
       // Temporary code to catch the code that enters a junk BigDecimal object toString() will throw an exception
       try
       {
           if (value != null)
        {
            value.toString();
        }
       }
       catch (Throwable t)
       {
           t.printStackTrace();
           //throw new Error(t.toString());
       }

       setAttributeObject(attributeName, value, false);
  }

  @Override
public void setBytesAttribute(String attributeName, byte[] value) throws ReadOnlyException, AttributeSetTypeException, ConfigException
  {
       setAttributeObject(attributeName, value, false);
  }

  @Override
public void setBooleanAttribute(String attributeName, Boolean value) throws ReadOnlyException, AttributeSetTypeException, ConfigException
  {
       setAttributeObject(attributeName, value, false);
  }

  @Override
public void setDateAttribute(String attributeName, Date value) throws ReadOnlyException, AttributeSetTypeException, ConfigException
  {
       setAttributeObject(attributeName, value, false);
  }


  @Override
public void setReferenceAttribute(String attributeName, Reference value) throws ReadOnlyException, AttributeSetTypeException, ConfigException
  {
       setAttributeObject(attributeName, value, false);
  }

  @Override
public void registerAttributeChangeHandler(Object context, IAttributeChangeHandler handler)
  {
      validateRegistrationContext(context);

      ((IChangeRegistration)context).registerAttributeChangeHandler(getCompoundName(), handler);
  }

  @Override
public void unregisterAttributeChangeHandler(Object context)
  {
      validateRegistrationContext(context);

      ((IChangeRegistration)context).unregisterAttributeChangeHandler(getCompoundName());
  }

  @Override
public void registerAttributeChangeHandler(Object context, String attributeName, IAttributeChangeHandler handler)
  {
      validateRegistrationContext(context);

      if (attributeName == null)
    {
        throw new IllegalArgumentException("The attribute name must not be null.");
    }

      if (getAttribute(attributeName) == null)
    {
        throw new IllegalArgumentException("There is no attribute '" + attributeName + "' in this attribute set.");
    }

      ((IChangeRegistration)context).registerAttributeChangeHandler(getCompoundName().setNextComponent(attributeName), handler);
  }

  @Override
public void unregisterAttributeChangeHandler(Object context, String attributeName)
  {
      validateRegistrationContext(context);

      if (attributeName == null)
    {
        throw new IllegalArgumentException("The attribute name must not be null.");
    }

      ((IChangeRegistration)context).unregisterAttributeChangeHandler(getCompoundName().setNextComponent(attributeName));
  }

  private void validateRegistrationContext(Object context)
  {
      if (context == null || !(context instanceof IChangeRegistration))
    {
        throw new IllegalArgumentException("The context object is invalid.");
    }
  }

   @Override
public boolean replaceReferences(boolean thisIsSystemAttributes, IReplaceRef replaceSrvc)
   {
       HashMap map = getAttributes();
       Set keys = map.keySet();
       Iterator iter = keys.iterator();
       boolean retVal = true;
       while( iter.hasNext() )
       {
           String key = (String)iter.next();
           Object value = map.get(key);
           if (value instanceof ICanReplaceRef)
           {
               boolean ret = ((ICanReplaceRef)value).replaceReferences(false, replaceSrvc);
               if (!ret)
            {
                retVal = false;
            }
           }
           else if (value instanceof Reference)
        {
            try
            {
                Reference newRef = replaceSrvc.replace((Reference)value);
                if (newRef.getElementName().startsWith(ILogicalNameSpace.NO_STORAGE_LABEL))
                {
                    retVal = false;
                }
                m_dictionary.put(key, newRef);
            }
            catch (AttributeSetTypeException e)
            {
                // Should never happen
                e.printStackTrace();
                throw new Error(e.toString());
            }
        }
        else if (thisIsSystemAttributes && key.equals(Element.SUPER_REFERENCE))
           {
               try
               {
                   m_dictionary.put(key, replaceSrvc.replace((String)value));
               }
               catch (Exception e) // Should not happen
               {
                   e.printStackTrace();
                   throw new Error(e.toString(), e);
               }
           }
        else
        {
            continue;
        }
       }
       return retVal;

   }

   @Override
public void mergeUtilHint_markAttributeDeleted(String attName)
   {
       if (m_mergeUtilHint_deletedSet == null)
    {
        m_mergeUtilHint_deletedSet = new HashSet();
    }
       m_mergeUtilHint_deletedSet.add(attName);
   }

   @Override
public String[] mergeUtilHint_getAttributesMarkedDeleted()
   {
       if (m_mergeUtilHint_deletedSet == null)
    {
        return new String[0];
    }
    else
    {
        return (String[])m_mergeUtilHint_deletedSet.toArray(EMPTY_STRING_ARRAY);
    }
   }


}
