/*
 * Copyright (c) 2001 Sonic Software. All Rights Reserved.
 */

package com.sonicsw.mf.common.config.impl;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Objects;
import java.util.Set;

import com.sonicsw.mf.common.config.IAttributeSetType;
import com.sonicsw.mf.common.config.ReadOnlyException;


final class AttributeSetType extends HashMap implements IAttributeSetType, java.io.Serializable 
{
    private static final long serialVersionUID = 0L;
    private final static int SERIALIZATION_VERSION = 3;

    String m_name;
    private int m_numInstances;
    private int m_currentAttribute;
    private boolean m_readOnly;

    private transient Set m_set;

    AttributeSetType(String name)
    {
        super();
        m_readOnly = false;
        m_numInstances = 0;
        m_name = name;
        m_currentAttribute = 0;
        m_set = keySet();
    }

    @Override
    public boolean equals(Object o)
    {
        if (o == null)
        {
            return false;
        }

        if (!(o instanceof AttributeSetType))
        {
            return false;
        }

        AttributeSetType attributeSetType = (AttributeSetType)o;

        if (!attributeSetType.getName().equals(m_name))
        {
            return false;
        }

        String[] attributeNames = attributeSetType.getAttributeNames();
        String[] thisAttributeNames = getAttributeNames();

        // All attribute name in attributeSetType must exist in this AttributeSetType for equality
        for (int i = 0; i < attributeNames.length; i++)
        {
            if (get(attributeNames[i]) == null)
            {
                return false;
            }
        }

        // All attribute name in thisAttributeNames must exist in attributeNames for equality
        for (int i = 0; i < thisAttributeNames.length; i++)
        {
            if (attributeSetType.get(thisAttributeNames[i]) == null)
            {
                return false;
            }
        }

        return true;
    }

    int estimateSize()
    {
        return Util.estimateSize(m_name) + Util.OBJECT_SIZE + Util.estimateHashMapSize(this);
    }
    
    @Override
    public int hashCode() {
        List<Object> values = new ArrayList<Object>();
        values.add(getName());
        if(getAttributeNames() != null && getAttributeNames().length > 0) {
            values.addAll(Arrays.asList(getAttributeNames()));
        }
        return Objects.hash(values.toArray());
    }

    private void writeObject(java.io.ObjectOutputStream s) throws java.io.IOException
    {
         s.writeInt(SERIALIZATION_VERSION);
         s.writeObject(m_name);
         s.writeInt(m_numInstances);
         s.writeInt(m_currentAttribute);
         s.writeBoolean(m_readOnly);
    }
    private void readObject(java.io.ObjectInputStream s) throws java.io.IOException, ClassNotFoundException
    {
        int version = s.readInt();
        if (version != SERIALIZATION_VERSION)
        {
            Util.throwSerialVersionMismatch(version, SERIALIZATION_VERSION);
        }

        m_name = (String)s.readObject();
        m_numInstances = s.readInt();
        m_currentAttribute = s.readInt();
        m_readOnly = s.readBoolean();

        m_set = keySet();
    }


 
    void setReadOnly(boolean readOnly)
    {
        // Type that has instances must stay read only even if it is
        // in a writable element
        if (m_numInstances > 0 && !readOnly)
        {
            return;
        }

        m_readOnly = readOnly;
    }

    String getName()
    {
        return m_name;
    }

    // Creates an empty set
    TypedAttSet createTypedSet()
    {
        return createTypedSet(new Object[size()]);
    }

    // Creates a set with values
    TypedAttSet createTypedSet(Object[] values)
    {
        TypedAttSet set = new TypedAttSet(this);
        m_numInstances++;
        m_readOnly = true;
        set.setValues(values);
        return set;
    }

    void removeTypedSet(TypedAttSet set)
    {
        m_numInstances--;
    }

    boolean hasInstances()
    {
        return m_numInstances > 0;
    }

    @Override
    public void addAttributeName(String name) throws ReadOnlyException
    {
        if (m_readOnly)
        {
            throw new ReadOnlyException("New attributes cannot be added to type ",  m_name);
        }

        // Already contains this attribute
        if (containsKey(name))
        {
            return;
        }

        put(name, (new Integer(m_currentAttribute++)));
    }

    @Override
    public String[] getAttributeNames()
    {
        int count = size();
        String[] result = new String[count];
        Iterator iterator = m_set.iterator();
        for (int i = 0; i < count; i++)
        {
            result[i] = (String)iterator.next();
        }
        return result;
    }

}
