/*
 * Copyright (c) 2001 Sonic Software. All Rights Reserved.
 */

package com.sonicsw.mf.common.config.impl;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.util.Enumeration;
import java.util.jar.Attributes;
import java.util.jar.JarEntry;
import java.util.jar.JarFile;
import java.util.jar.Manifest;

import com.sonicsw.mf.common.config.IAttributeSet;
import com.sonicsw.mf.common.config.IBlob;
import com.sonicsw.mf.common.config.IChunkedBlobStreamer;
import com.sonicsw.mf.common.dirconfig.IDirElement;

public class Blob implements IBlob, java.io.Serializable
{
    private static final long serialVersionUID = 0L;
    private final static int SERIALIZATION_VERSION = 3;
    private final static String IS_LOGICAL_NAME_ATTR = "IS_LOGICAL_NAME";

    IDirElement m_element;
    byte[] m_blob;
    transient IChunkedBlobStreamer m_streamer;
    transient InputStream m_blobStream;
    boolean m_logical = false;

    private void writeObject(java.io.ObjectOutputStream s) throws java.io.IOException
    {
        s.writeInt(SERIALIZATION_VERSION);
        try
        {
            storeLogicalFlag();
        }
        catch (Exception e)
        {
        	throw new java.io.IOException("Unable to serialize logical flag for blob " + m_element.getIdentity().getName() + ": "+ e.toString());
        }
        s.writeObject(m_element);
        s.writeObject(m_blob);
        // m_streamer does not cross the wire
    }

    private void storeLogicalFlag() throws Exception
    {
    	markBlobState((Element)m_element, Boolean.valueOf(m_logical), IS_LOGICAL_NAME_ATTR);
    }

    private void readObject(java.io.ObjectInputStream s) throws java.io.IOException, ClassNotFoundException
    {
        int version = s.readInt();
        if (version != SERIALIZATION_VERSION)
        {
            throw new java.io.IOException("Serialization version mismatch. Serialized object version: " + version +
                                     " class version: " + SERIALIZATION_VERSION);
        }

        m_element = (IDirElement)s.readObject();
        m_blob = (byte[])s.readObject();
        m_logical = retrieveLogicalFlag();
    }

    private boolean retrieveLogicalFlag()
    {
    	boolean logical = false;
    	IAttributeSet topSet = m_element.getAttributes();
        IAttributeSet systemAttributes = (IAttributeSet)topSet.getAttribute(SYSTEM_ATTRIBUTES);
        if (systemAttributes != null)
        {
        	Boolean logicalObject = (Boolean)systemAttributes.getAttribute(IS_LOGICAL_NAME_ATTR);
        	if (logicalObject != null)
            {
                logical = logicalObject.booleanValue();
            }
        }
        return logical;
    }

    public Blob replaceElement(IDirElement element)
    {
        return new Blob(element, m_blob, m_streamer, m_logical);
    }

    public static boolean isExpandableFile(File localFile)
    {
        if (localFile.exists())
        {

          try
          {
              JarFile jarFile = new JarFile(localFile);
              Manifest blobManifest = jarFile.getManifest();

              if (blobManifest != null)
              {
                  Attributes attrs = blobManifest.getMainAttributes();
                  String expandValue = attrs.getValue(IBlob.EXPAND_IN_CACHE);
                  if (expandValue != null)
                  {
                      jarFile.close();
                      return expandValue.equals("true");
                  }
              }
              // if we haven't returned true, check for meta-inf/sonic.xml
              Enumeration<JarEntry> entries = jarFile.entries();
              while (entries.hasMoreElements())
              {
                  JarEntry entry = entries.nextElement();
                  if ("meta-inf/sonic.xml".equalsIgnoreCase(entry.getName()))
                  {
                      jarFile.close();
                      return true;
                  }
              }
              if (jarFile != null)
            {
                jarFile.close();
            }
          }
          catch (Throwable e)
          {}
      }
      return false;
    }

    //mrd 12/09/2004 New utility to check for incomplete blobs
    static public boolean isIncompleteBlob(IDirElement element)
    {
        IAttributeSet topSet = element.getAttributes();
        Object systemAttrs = topSet.getAttribute(SYSTEM_ATTRIBUTES);
        Integer blobState = null;
        if (systemAttrs != null)
        {
            blobState = (Integer)((IAttributeSet)systemAttrs).getAttribute(LARGE_FILE_STATE);
        }
        return ((blobState != null) && blobState.equals(INCOMPLETE));
    }

    // mrd 12/09/2004 New Utility to internally mark the blob envelope element through its SYSTEM_ATTRIBUTES,
    // without persisting the modification
    // Attribute                       Values                                 type
    // "BLOB_TRANSFER_STATE"           0 (partial), 1 (end)                   Integer
    // "LARGE_FILE_STATE"              0 (incomplete), 1 (complete)           Integer
    // "ARCHIVE_NAME"                  logical name of the archive            String
    // "SONIC_EXPAND_IN_CACHE"         Boolean.TRUE (if set)                  Boolean
    static public void markBlobState(Element element, Object blobState, String attribute) throws Exception
    {
        boolean readOnly = element.isReadOnly();
        element.setReadOnly(false);
        IAttributeSet topSet = element.getAttributes();
        Object systemAttributes = topSet.getAttribute(SYSTEM_ATTRIBUTES);
        ((AttributeSet)topSet).setReadOnly(false);
        if (systemAttributes == null)
        {
            systemAttributes = topSet.createAttributeSet(SYSTEM_ATTRIBUTES);
        }
        ((AttributeSet)systemAttributes).setReadOnly(false);
        if (blobState instanceof Integer)
        {
            ( (IAttributeSet) systemAttributes).setIntegerAttribute(attribute, (Integer)blobState);
        }
        else if (blobState instanceof String)
        {
            ( (IAttributeSet) systemAttributes).setStringAttribute(attribute, (String)blobState);
        }
        else if (blobState instanceof Boolean)
        {
            ( (IAttributeSet) systemAttributes).setBooleanAttribute(attribute, (Boolean)blobState);
        }
        element.setReadOnly(readOnly);
        ((AttributeSet)topSet).setReadOnly(readOnly);
        ((AttributeSet)systemAttributes).setReadOnly(readOnly);
    }
    
    static public Object getBlobState(Element element, String attribute)
    {
        IAttributeSet topSet = element.getAttributes();
        if (topSet == null)
        {
            return null;
        }
        IAttributeSet systemAttributes = (IAttributeSet)topSet.getAttribute(SYSTEM_ATTRIBUTES);
        if (systemAttributes == null)
        {
            return null;
        }
        return systemAttributes.getAttribute(attribute);
    }

    // mrd 12/08/2004 modified to support large files. The streamer object can get chunks of the stream as needed
    public Blob(IDirElement element, byte[] blob, IChunkedBlobStreamer streamer)
    {
        m_element = element;
        m_blob = blob;
        m_streamer = streamer;
    }

    public Blob(IDirElement element, byte[] blob)
    {
        m_element = element;
        m_blob = blob;
        m_streamer = null;
    }

    public Blob(IDirElement element, InputStream stream)
    {
        m_element = element;
        m_streamer = null;
        m_blobStream = stream;
    }

    Blob(IDirElement element, byte[] blob, IChunkedBlobStreamer streamer, boolean logical)
    {
        this(element, blob, streamer);
        m_logical = logical;
    }

    @Override
    public IDirElement getElement()
    {
        return m_element;
    }

    @Override
    public void setLogical(boolean logical)
    {
        m_logical = logical;
    }

    @Override
    public boolean isLogicalName()
    {
        return m_logical;
    }

    @Override
    public java.io.InputStream getBlobStream()
    {
        if (m_blobStream != null)
        {
            return m_blobStream;
        }
        if (m_blob == null)
        {
            return null;
        }

        return new BlobInputStream(m_blob);
    }

    @Override
    public byte[] getBlobBytes()
    {
        if (m_blob == null)
        {
            return null;
        }

        return m_blob;
    }

    class BlobInputStream extends InputStream
    {

        int m_chunkIndex = 0;
        int m_pos = 0;
        int m_count;
        byte[] m_buf;

        BlobInputStream(byte[] bytes)
        {
            super();
            m_buf = bytes;
            m_count = m_buf.length;
        }

        @Override
        public int available()
        {
            return m_count - m_pos;
        }
        @Override
        public int read()
        {
            nextChunkIfNeeded();
            if (eof())
            {
                return -1;
            }
            int r = (m_buf[m_pos++] & 0xff);
            return r;
        }

        private void nextChunkIfNeeded() throws Error
        {
            try
            {
                if (m_streamer != null)
                {
                    if (m_pos == BLOB_CHUNK_SIZE)
                    {
                        IBlob blob;
                        m_chunkIndex = m_chunkIndex + 1;
                        if (m_logical)
                        {
                          blob = m_streamer.getFSBlob(m_element.getIdentity().getName(), false,
                                                      m_chunkIndex * m_pos);
                        }
                        else
                        {
                            blob = m_streamer.getBlob(m_element.getIdentity().getName(), false,
                                                      m_chunkIndex * m_pos);
                        }
                        m_buf = blob.getBlobBytes();
                        m_pos = 0;
                        m_count = m_buf.length;
                    }
                }
            }
            catch (Exception dirE)
            {
                throw new Error(dirE.toString());
            }
        }

        /**
         * @param b
         *          the buffer into which the data is read.
         * @param off
         *          the start offset in array <code>b</code> at which the data
         *          is written.
         * @param len
         *          the maximum number of bytes to read.
         * @return the total number of bytes read into the buffer, or
         *         <code>-1</code> if there is no more data because the end of
         *         the stream has been reached.
         * @exception IOException
         *              if an I/O error occurs.
         * @exception NullPointerException
         *              if <code>b</code> is <code>null</code>.
         * @see java.io.InputStream#read()
         */
        @Override
        public int read(final byte b[], final int off, final int len) throws IOException {
            nextChunkIfNeeded();
            if (eof())
            {
                return -1;
            }
            int read = Math.min(len, available());
            System.arraycopy(m_buf, m_pos, b, off, read);
            m_pos += read;
            return read;
        }

        private boolean eof()
        {
            return (m_buf == null) || (m_pos == m_count);
        }
    }
}
