/*
 * Copyright (c) 2001 Sonic Software. All Rights Reserved.
 */

package com.sonicsw.mf.common.config.impl;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Hashtable;

import com.sonicsw.mf.common.config.IAttributeChangeHandler;
import com.sonicsw.mf.common.config.IAttributeSet;
import com.sonicsw.mf.common.config.IDeltaAttributeSet;
import com.sonicsw.mf.common.config.IDeltaElement;
import com.sonicsw.mf.common.config.query.AttributeName;

// an implementation for cacses where there is no MF container - tests and admin. clients. In this case
// IComponentContext is null (and therefore not used here)
public final class ChangeRegistration
implements IChangeRegistration
{
    private HashMap m_registrationTable;
    private Object m_lock;
    private Hashtable m_deltaTable;

    public ChangeRegistration()
    {
        m_registrationTable = new HashMap();
        m_lock = new Object();
        m_deltaTable = new Hashtable();
    }

    @Override
    public void registerAttributeChangeHandler(AttributeName name, IAttributeChangeHandler handler)
    {
         synchronized (m_lock)
         {
            String elementName = name.getElementName();

            // Sanity check
            if (elementName == null || elementName.length() == 0)
            {
                throw new IllegalArgumentException("registerHandler: No element name.");
            }

            AttributeAssociationTree elementTable = (AttributeAssociationTree)m_registrationTable.get(elementName);

            if (elementTable == null)
            {
                elementTable = new AttributeAssociationTree();
                m_registrationTable.put(elementName, elementTable);
            }

            elementTable.insertPayload(name, handler);
        }
    }

    @Override
    public void unregisterAttributeChangeHandler(AttributeName name)
    {
        synchronized (m_lock)
        {

           String elementName = name.getElementName();

            // Sanity check
            if (elementName == null || elementName.length() == 0)
            {
                throw new IllegalArgumentException("registerHandler: No element name.");
            }

            AttributeAssociationTree elementTable = (AttributeAssociationTree)m_registrationTable.get(elementName);
            if (elementTable == null)
            {
                return;
            }

            elementTable.removePayload(name);

            if (elementTable.size() == 0)
            {
                m_registrationTable.remove(elementName);
            }
         }
    }

    private void fireElementDeletedHandler(String elementName, boolean doNotify)
    {
        ArrayList notificationsList = new ArrayList();
        synchronized (m_lock)
        {
            AttributeAssociationTree elementTable = (AttributeAssociationTree)m_registrationTable.get(elementName);

            if (elementTable == null)
            {
                return;
            }

            ProcessDeltaNotifications.handleElementDeletion(elementTable, notificationsList);
        }

        // This should not be locked since the handlers might call back to register and unregister
        if (doNotify)
        {
            ProcessDeltaNotifications.doNotifications(notificationsList);
        }

        m_registrationTable.remove(elementName);
    }

    private void fireAttributeChangeHandlers(IDeltaElement delta, boolean doNotify)
    {
        ArrayList notificationsList = new ArrayList();
        synchronized (m_lock)
        {
            AttributeAssociationTree elementTable = (AttributeAssociationTree)m_registrationTable.get(delta.getIdentity().getName());

            if (elementTable == null)
            {
                return;
            }


            Object attributes = delta.getDeltaAttributes();
            if (attributes instanceof IAttributeSet)
            {
                ProcessDeltaNotifications.processDelta((IAttributeSet)attributes, elementTable, notificationsList);
            }
            else if (attributes instanceof IDeltaAttributeSet)
            {
                ProcessDeltaNotifications.processDelta((IDeltaAttributeSet)attributes, elementTable, notificationsList);
            }
        }

        // This should not be locked since the handlers might call back to register and unregister
        if (doNotify)
        {
            ProcessDeltaNotifications.doNotifications(notificationsList);
        }
    }

    // Associate the delta with the current thread
    public void setDelta(IDeltaElement delta)
    {
        m_deltaTable.put( Thread.currentThread(), delta);
    }

   // Associate the deleted element name with the currect thread
    public void setDeletedElementName(String deletedElementName)
    {
        m_deltaTable.put( Thread.currentThread(), deletedElementName);
    }


    public void fireAttributeChangeHandlers(boolean doNotify)
    {
        Object change = m_deltaTable.remove(Thread.currentThread());

        if (change == null)
        {
            throw new IllegalStateException("fireAttributeChangeHandlers can be called only for" +
                                            " IElementChange.ELEMENT_UPDATED and IElementChange.ELEMENT_DELETD configuration changes.");
        }

        if (change instanceof IDeltaElement)
        {
            fireAttributeChangeHandlers((IDeltaElement)change, doNotify);
        }
        else if (change instanceof String)
        {
            fireElementDeletedHandler((String)change, doNotify);
        }
        else // Should never happen
        {
            Error error = new Error("Wrong change type.");
            error.printStackTrace();
            throw error;
        }
       
    }

    public void adjustHandlers()
    {
        Object change = m_deltaTable.remove(Thread.currentThread());
        if (change == null)
        {
            return;
        }
        if (change instanceof IDeltaElement)
        {
            fireAttributeChangeHandlers((IDeltaElement)change, false);
        }
        else if (change instanceof String)
        {
            fireElementDeletedHandler((String)change, false);
        }
        else // Should never happen
        {
            Error error = new Error("Wrong change type.");
            error.printStackTrace();
            throw error;
        }

    }

}

