/*
 * Copyright (c) 2001 Sonic Software. All Rights Reserved.
 */
package com.sonicsw.mf.common.config.impl;

import java.io.InputStream;
import java.util.ArrayList;
import java.util.HashMap;

import com.sonicsw.mf.common.IDSTransaction;
import com.sonicsw.mf.common.config.IBasicElement;
import com.sonicsw.mf.common.config.INamingNotification;
import com.sonicsw.mf.common.dirconfig.IDeltaDirElement;
import com.sonicsw.mf.common.dirconfig.IDirElement;

public class DSTransaction implements IDSTransaction, java.io.Serializable
{
    private static final long serialVersionUID = 0L;
    private final static int SERIALIZATION_VERSION = 3;

    private ArrayList m_actions;

    public DSTransaction()
    {
        m_actions = new ArrayList();
    }

    public ArrayList getActions()
    {
        return m_actions;
    }

    public void addAction(Action action)
    {
        m_actions.add(action);
    }

    public void addAction(int index, Action action)
    {
        m_actions.add(index, action);
    }

    private void writeObject(java.io.ObjectOutputStream s) throws java.io.IOException
    {
         s.writeInt(SERIALIZATION_VERSION);
         s.writeObject(m_actions);
    }

    private void readObject(java.io.ObjectInputStream s) throws java.io.IOException, ClassNotFoundException
    {
        int version = s.readInt();
        if (version != SERIALIZATION_VERSION)
        {
            Util.throwSerialVersionMismatch(version, SERIALIZATION_VERSION);
        }

        m_actions = (ArrayList)s.readObject();
    }


    @Override
    public void addCreateElement(IDirElement element)
    {
        m_actions.add(new CreateElement(element));
    }

    public CreateElement createCreateElement(IDirElement element)
    {
        return new CreateElement(element);
    }

    @Override
    public void addCreateElements(IDirElement[] elements)
    {
        m_actions.add(new CreateElements(elements));
    }

    @Override
    public void addUpdateElement(IDeltaDirElement element)
    {
        m_actions.add(new UpdateElement(element, false));
    }

    @Override
    public void addUpdateElement(IDeltaDirElement element, boolean doNotUpdateIfEmpty)
    {
        m_actions.add(new UpdateElement(element, doNotUpdateIfEmpty));
    }

    @Override
    public void addDeleteElement(String elementName)
    {
        m_actions.add(new DeleteElement(elementName));
    }

    @Override
    public void addCreateFolder(String folderName)
    {
        m_actions.add(new CreateFolder(folderName));
    }

    @Override
    public void addCreateFolder(String folderName, boolean existingOk)
    {
        m_actions.add(new CreateFolder(folderName, existingOk));
    }

    @Override
    public void addDeleteFolder(String folderName)
    {
        m_actions.add(new DeleteFolder(folderName));
    }

    @Override
    public void addRename(String oldName, String newName)
    {
        m_actions.add(new Rename(oldName, newName));
    }

    @Override
    public void addSetAttributes(String name, HashMap attributes)
    {
        m_actions.add(new SetAttributes(name, attributes));
    }

    @Override
    public void addSubclassElement(IBasicElement delta, String newElementPath)
    {
        m_actions.add(new SubclassElement(delta, newElementPath));
    }

    @Override
    public void addCloneElement(IBasicElement deltaFromSource, String newElementPath)
    {
        m_actions.add(new CloneElement0(deltaFromSource, newElementPath, false));
    }

    @Override
    public void addCloneElement(IBasicElement deltaFromSource, String newElementPath, boolean createTemplate)
    {
        m_actions.add(new CloneElement0(deltaFromSource, newElementPath, createTemplate));
    }

    @Override
    public void addCloneElement(String sourcePath, String newElementPath)
    {
        m_actions.add(new CloneElement1(sourcePath, newElementPath, false));
    }

    @Override
    public void addCloneElement(String sourcePath, String newElementPath, boolean createTemplate)
    {
        m_actions.add(new CloneElement1(sourcePath, newElementPath, createTemplate));
    }

    @Override
    public void addAttachBlob(IBasicElement element, java.io.InputStream blobStream) throws java.io.IOException
    {
        m_actions.add(new AttachBlob(element, blobStream));
    }

    // not in the IDSTransaction interface, internal for DirectoryServiceProxy use
    // replace an AttachBlob action with an AppendBlob action
    public void addAppendBlob(AttachBlob attachAction, byte[] blob, int offset) throws java.io.IOException
    {
        m_actions.set(m_actions.indexOf(attachAction), new AppendBlob(attachAction.m_element, blob, offset));
    }

    public AttachBlob createAttachBlob(IBasicElement element, java.io.InputStream blobStream) throws java.io.IOException
    {
        return new AttachBlob(element, blobStream);
    }

    @Override
    public void addDetachBlob(IDeltaDirElement delta)
    {
        m_actions.add(new DetachBlob(delta));
    }



    public abstract class Action implements java.io.Serializable
    {
        private static final long serialVersionUID = 0L;
        public abstract INamingNotification[] getNamingNotifications();
    }

    public class CreateElement extends Action implements java.io.Serializable
    {
        private static final long serialVersionUID = 0L;
        public IDirElement m_element;
        public transient String m_newStorageName;
        CreateElement(IDirElement element)
        {
            m_element = element;
        }

        @Override
        public INamingNotification[] getNamingNotifications()
        {
            return new INamingNotification[] {new ElementCreateNotification(m_element.getIdentity().getName())};
        }
    }

    public class CreateElements extends Action  implements java.io.Serializable
    {
        private static final long serialVersionUID = 0L;
        public IDirElement[] m_elements;
        public transient String[] m_newStorageNames;
        CreateElements(IDirElement[] elements)
        {
            m_elements = elements;
        }

        @Override
        public INamingNotification[] getNamingNotifications()
        {
            INamingNotification[] notifications = new INamingNotification[m_elements.length];
            for (int i = 0; i < notifications.length; i++)
            {
                notifications[i] = new ElementCreateNotification(m_elements[i].getIdentity().getName());
            }
            return notifications;
        }

    }

    public class UpdateElement extends Action  implements java.io.Serializable
    {
        private static final long serialVersionUID = 1L;
        public IDeltaDirElement m_element;
        public boolean m_doNotUpdateIfEmpty;

        UpdateElement(IDeltaDirElement element, boolean doNotUpdateIfEmpty)
        {
            m_element = element;
            m_doNotUpdateIfEmpty = doNotUpdateIfEmpty;
        }

        @Override
        public INamingNotification[] getNamingNotifications()
        {
            return new INamingNotification[0];
        }

    }

    public class DeleteElement extends Action  implements java.io.Serializable
    {
        private static final long serialVersionUID = 0L;
        public String m_elementName;
        DeleteElement(String elementName)
        {
            m_elementName = elementName;
        }

        @Override
        public INamingNotification[] getNamingNotifications()
        {
            return new INamingNotification[] {new ElementDeleteNotification(m_elementName)};
        }

    }

    public class CreateFolder extends Action implements java.io.Serializable
    {
        private static final long serialVersionUID = 0L;
        public String m_folderName;
        public boolean m_existingOk;
        CreateFolder(String foldertName)
        {
            m_folderName = foldertName;
            m_existingOk = false;
        }

        CreateFolder(String folderName, boolean existingOk)
        {
            m_folderName = folderName;
            m_existingOk = existingOk;
        }

        @Override
        public INamingNotification[] getNamingNotifications()
        {
            return new INamingNotification[] {new FolderCreateNotification(m_folderName)};
        }

    }


    public class DeleteFolder extends Action implements java.io.Serializable
    {
        private static final long serialVersionUID = 0L;
        public String m_folderName;
        DeleteFolder(String foldertName)
        {
            m_folderName = foldertName;
        }

        @Override
        public INamingNotification[] getNamingNotifications()
        {
            return new INamingNotification[] {new FolderDeleteNotification(m_folderName)};
        }
    }

    public class Rename extends Action implements java.io.Serializable
    {
        private static final long serialVersionUID = 0L;
        public String m_oldName;
        public String m_newName;
        Rename(String oldName, String newName)
        {
            m_oldName = oldName;
            m_newName = newName;
        }

        @Override
        public INamingNotification[] getNamingNotifications()
        {
            return new INamingNotification[] {new RenameNotification(m_oldName, m_newName)};
        }

    }

    public class SubclassElement extends Action implements java.io.Serializable
    {
        private static final long serialVersionUID = 0L;

        public IBasicElement m_delta;
        public String m_newElementPath;
        public transient String m_newStorageName;

        SubclassElement(IBasicElement delta, String newElementPath)
        {
            m_delta = delta;
            m_newElementPath = newElementPath;
        }

        @Override
        public INamingNotification[] getNamingNotifications()
        {
            return new INamingNotification[] {new ElementCreateNotification(m_newElementPath)};
        }

    }

    public class CloneElement0 extends Action implements java.io.Serializable
    {
        private static final long serialVersionUID = 0L;

        public IBasicElement m_delta;
        public String m_newElementPath;
        public boolean m_createTemplate;
        public transient String m_newStorageName;

        CloneElement0(IBasicElement delta, String newElementPath, boolean createTemplate)
        {
            m_delta = delta;
            m_newElementPath = newElementPath;
            m_createTemplate = createTemplate;
        }

        @Override
        public INamingNotification[] getNamingNotifications()
        {
            return new INamingNotification[] {new ElementCreateNotification(m_newElementPath)};
        }

    }

    public class CloneElement1 extends Action implements java.io.Serializable
    {
        private static final long serialVersionUID = 0L;

        public String m_sourcePath;
        public String m_newElementPath;
        public boolean m_createTemplate;
        public transient String m_newStorageName;

        CloneElement1(String sourcePath, String newElementPath, boolean createTemplate)
        {
            m_sourcePath = sourcePath;
            m_newElementPath = newElementPath;
            m_createTemplate = createTemplate;
        }

        @Override
        public INamingNotification[] getNamingNotifications()
        {
            return new INamingNotification[] {new ElementCreateNotification(m_newElementPath)};
        }

    }

    public class SetAttributes extends Action implements java.io.Serializable
    {
        private static final long serialVersionUID = 0L;
        public String m_name;
        public HashMap m_attributes;
        SetAttributes(String name, HashMap attributes)
        {
            m_name = name;
            m_attributes = attributes;
        }

        @Override
        public INamingNotification[] getNamingNotifications()
        {
            return new INamingNotification[] {new MetaAttributesChangeNotification(m_name, m_attributes)};
        }

    }

    public class AppendBlob extends Action implements java.io.Serializable
    {
         private static final long serialVersionUID = 0L;

         public byte[] m_blob;
         public int m_src;
         public transient String m_newStorageName;
         public IBasicElement m_element;

         AppendBlob(IBasicElement element, byte[] blob, int src) throws java.io.IOException
         {
             m_element = element;
             if (blob == null)
            {
                throw new java.io.IOException("The blob stream object cannot be null.");
            }
             m_blob = blob;
             m_src = src;
         }

         @Override
        public INamingNotification[] getNamingNotifications()
         {
            if (m_element instanceof IDirElement)
            {
                return new INamingNotification[] {new ElementCreateNotification(m_element.getIdentity().getName())};
            }
            else
            {
                return new INamingNotification[0];
            }
         }
    }

    public class AttachBlob extends Action implements java.io.Serializable
    {
        private static final long serialVersionUID = 0L;

        public byte[] m_blob;
        public IBasicElement m_element;
        public transient String m_newStorageName;
        public transient InputStream m_stream;

        AttachBlob(IBasicElement element, java.io.InputStream blobStream) throws java.io.IOException
        {
            m_element = element;
            if (blobStream == null)
            {
                throw new java.io.IOException("The blob stream object cannot be null.");
            }

            m_stream = blobStream;

        }

        @Override
        public INamingNotification[] getNamingNotifications()
        {
            if (m_element instanceof IDirElement)
            {
                return new INamingNotification[] {new ElementCreateNotification(m_element.getIdentity().getName())};
            }
            else
            {
                return new INamingNotification[0];
            }
        }

    }

    public class DetachBlob extends Action implements java.io.Serializable
    {
        private static final long serialVersionUID = 0L;

        public IDeltaDirElement m_delta;

        DetachBlob(IDeltaDirElement delta)
        {
            m_delta = delta;
        }

        @Override
        public INamingNotification[] getNamingNotifications()
        {
            return new INamingNotification[0];
        }

    }




}
