

/*
 * Copyright (c) 2001 Sonic Software. All Rights Reserved.
 */

package com.sonicsw.mf.common.config.impl;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Set;
import java.util.Vector;

import com.sonicsw.mf.common.config.AttributeSetTypeException;
import com.sonicsw.mf.common.config.IAttributeList;
import com.sonicsw.mf.common.config.IAttributeSet;
import com.sonicsw.mf.common.config.IDeltaAttributeList;
import com.sonicsw.mf.common.config.IDeltaAttributeSet;
import com.sonicsw.mf.common.config.NotModifiedAttException;
import com.sonicsw.mf.common.config.Reference;
import com.sonicsw.mf.common.config.query.AttributeName;
import com.sonicsw.mf.common.view.ILogicalNameSpace;


final class DeltaAttributeSet implements IDeltaAttributeSet, IDelta, java.io.Serializable, ICanReplaceRef
{
    private static final long serialVersionUID = 0L;
    private final static int SERIALIZATION_VERSION = 3;

    private String m_typeName;
    private HashMap m_deltaValues;
    private String[] m_deletedAttributes;
    private String[] m_modifiedAttributes;
    private String[] m_newAttributes;


    DeltaAttributeSet(String typeName, HashMap modHistory, HashMap deltaValues)
    {
        // can be null
        m_typeName = typeName;

        m_deltaValues = deltaValues;

        Set set =  modHistory.keySet();

        int deletedCount = 0;
        int newCount = 0;
        int modifiedCount = 0;

        Iterator iterator = set.iterator();
        while (iterator.hasNext())
        {
            String attribute = (String)iterator.next();
            switch (((Integer)modHistory.get(attribute)).intValue())
            {
                case AttributeSet.DELETED_ATTRIBUTE: deletedCount++; break;
                case AttributeSet.NEW_ATTRIBUTE:     newCount++; break;
                case AttributeSet.MODIFIED_ATTRIBUTE: modifiedCount++; break;
            }

        }
        m_deletedAttributes = new String[deletedCount];
        m_newAttributes = new String[newCount];
        m_modifiedAttributes = new String[modifiedCount];

        deletedCount = 0;
        newCount = 0;
        modifiedCount = 0;

        iterator = set.iterator();
        while (iterator.hasNext())
        {
            String attribute = (String)iterator.next();
            switch (((Integer)modHistory.get(attribute)).intValue())
            {
                case AttributeSet.DELETED_ATTRIBUTE:  m_deletedAttributes[deletedCount++]  = attribute;
                                         break;
                case AttributeSet.NEW_ATTRIBUTE:      m_newAttributes[newCount++]  = attribute;
                                         break;
                case AttributeSet.MODIFIED_ATTRIBUTE: m_modifiedAttributes[modifiedCount++]  = attribute;
                                         break;
            }
        }
    }

    public boolean emptyDelta()
    {
        return m_deletedAttributes.length == 0 && m_newAttributes.length == 0 && m_modifiedAttributes.length == 0;
    }

    public int estimateSize()
    {
        int estimate = Util.ELEMENT_NODE_SIZE + Util.estimateSize(m_typeName);
        if (m_deltaValues != null)
        {
            estimate += Util.estimateHashMapSize(m_deltaValues);
        }
        if (m_deletedAttributes != null)
        {
            for (int i = 0; i < m_deletedAttributes.length; i++)
            {
                estimate += Util.estimateSize(m_deletedAttributes[i]);
            }
        }
        if (m_modifiedAttributes != null)
        {
            for (int i = 0; i < m_modifiedAttributes.length; i++)
            {
                estimate += Util.estimateSize(m_modifiedAttributes[i]);
            }
        }
        if (m_newAttributes != null)
        {
            for (int i = 0; i < m_newAttributes.length; i++)
            {
                estimate += Util.estimateSize(m_newAttributes[i]);
            }
        }

        return estimate;
    }

    private void writeObject(java.io.ObjectOutputStream s) throws java.io.IOException
    {
         s.writeInt(SERIALIZATION_VERSION);
         s.writeObject(m_typeName);
         s.writeObject(m_deltaValues);
         s.writeObject(m_deletedAttributes);
         s.writeObject(m_modifiedAttributes);
         s.writeObject(m_newAttributes);
    }

    private void readObject(java.io.ObjectInputStream s) throws java.io.IOException, ClassNotFoundException
    {
        int version = s.readInt();
        if (version != SERIALIZATION_VERSION)
        {
            Util.throwSerialVersionMismatch(version, SERIALIZATION_VERSION);
        }

        m_typeName = (String)s.readObject();
        m_deltaValues = (HashMap)s.readObject();
        m_deletedAttributes = (String[])s.readObject();
        m_modifiedAttributes = (String[])s.readObject();
        m_newAttributes = (String[])s.readObject();
    }

    @Override
    public boolean replaceReferences(boolean thisIsSystemAttributesDelta, IReplaceRef replaceSrvc)
    {
       Set keys = m_deltaValues.keySet();
       Iterator iter = keys.iterator();
       boolean retVal = true;
       while( iter.hasNext() )
       {
           String key = (String)iter.next();
           Object value = m_deltaValues.get(key);
           if (thisIsSystemAttributesDelta && key.equals(Element.SUPER_REFERENCE))
           {
               try
               {
                   m_deltaValues.put(key, replaceSrvc.replace((String)value));
               }
               catch (Exception e)
               {
                   e.printStackTrace();
                   throw new Error(e.toString());
               }
           }
           else if (value instanceof ICanReplaceRef)
           {
               boolean ret = ((ICanReplaceRef)value).replaceReferences(key.equals(Element.SYSTEM_ATTRIBUTES), replaceSrvc);
               if (!ret)
            {
                retVal = false;
            }
           }
           else if (value instanceof Reference)
           {
                Reference newRef = replaceSrvc.replace((Reference)value);
                if (newRef.getElementName().startsWith(ILogicalNameSpace.NO_STORAGE_LABEL))
                {
                    retVal = false;
                }
                m_deltaValues.put(key, newRef);
           }
        else
        {
            continue;
        }
       }
       return retVal;
    }

    String getTypeName()
    {
        return m_typeName;
    }

    @Override
    public String[] getDeletedAttributesNames()
    {
        return m_deletedAttributes;
    }

    @Override
    public String[] getModifiedAttributesNames()
    {
        return m_modifiedAttributes;
    }

    @Override
    public String[] getNewAttributesNames()
    {
        return m_newAttributes;
    }

    @Override
    public Object getNewValue(String attributeName) throws NotModifiedAttException
    {
        Object value = m_deltaValues.get(attributeName);

        if (value == null)
        {
            throw new NotModifiedAttException();
        }

        return value;
    }

    Object getValue(String attributeName)
    {
        return m_deltaValues.get(attributeName);
    }

    // Translate this delta to an attribute set - USED FOR DEBUGGING ONLY !!!
    // The only usage for this is to be exported to XML! THE STRUCTURE IS NOT CORRECT
    // FOR ANY OTHER USAGE.
    void toAttributeSet(IAttributeSet attributeSet) throws Exception
    {

        attributeSet.setBooleanAttribute("_MF_DELTA_OJECT_IMPL", Boolean.TRUE);
        attributeSet.setStringAttribute("TYPE_NAME", m_typeName);
        IAttributeList deletedAttributes = attributeSet.createAttributeList("DELETED_ATTRIBUTES");
        IAttributeList newAttributes = attributeSet.createAttributeList("NEW_ATTRIBUTES");
        IAttributeList modifiedAttributes = attributeSet.createAttributeList("MODIFIED_ATTRIBUTES");
        IAttributeSet deltaValues = attributeSet.createAttributeSet("DELTA_VALUES");

        for (int i = 0; i < m_deletedAttributes.length; i++)
        {
            deletedAttributes.addStringItem(m_deletedAttributes[i]);
        }

        for (int i = 0; i < m_newAttributes.length; i++)
        {
            newAttributes.addStringItem(m_newAttributes[i]);
        }

        for (int i = 0; i < m_modifiedAttributes.length; i++)
        {
            modifiedAttributes.addStringItem(m_modifiedAttributes[i]);
        }

        Iterator iterator = m_deltaValues.keySet().iterator();
        while (iterator.hasNext())
        {
           String key = (String)iterator.next();
           Object value = m_deltaValues.get(key);
           if (value instanceof IDeltaAttributeSet)
           {
               IAttributeSet convertedValue = deltaValues.createAttributeSet(key);
               ((DeltaAttributeSet)value).toAttributeSet(convertedValue);
           }
           else if (value instanceof IDeltaAttributeList)
           {
               IAttributeSet convertedValue = deltaValues.createAttributeSet(key);
               ((DeltaAttributeList)value).toAttributeSet(convertedValue);
           }
        else
        {
            ((AttributeSet)deltaValues).setAttributeObject(key, value, false);
        }

        }

    }

    // This delta represent subclassing. subModification represent a modification to the subclassed object.
    // adjustToSubclassedModification modifies this to refelect the changes made to subclassed object.
    void adjustToSubclassedModification(DeltaAttributeSet subModification)
    {

        // Enter this.m_newAttributes into a hash table for fast update
        HashMap newAttributes = new HashMap();
        for (int i = 0; i < m_newAttributes.length; i++)
        {
            newAttributes.put(m_newAttributes[i], "");
        }

        // Enter this.m_modifiedAttributes into a hash table for fast update
        HashMap modifiedAttributes = new HashMap();
        for (int i = 0; i < m_modifiedAttributes.length; i++)
        {
            modifiedAttributes.put(m_modifiedAttributes[i], "");
        }

        // Enter this.m_deletedAttributes into a hash table for fast update
        HashMap deletedAttributes = new HashMap();
        for (int i = 0; i < m_deletedAttributes.length; i++)
        {
            deletedAttributes.put(m_deletedAttributes[i], "");
        }


        // Move all subModification.m_deletedAttributes attributes to this.m_deletedAttributes list
        for (int i = 0; i < subModification.m_deletedAttributes.length; i++)
        {
            String attribute = subModification.m_deletedAttributes[i];
            newAttributes.remove(attribute);
            modifiedAttributes.remove(attribute);
            m_deltaValues.remove(attribute);
            deletedAttributes.put(attribute, "");
        }

        // Move to subModification.m_newAttributes to this.m_newAttributes
        for (int i = 0; i < subModification.m_newAttributes.length; i++)
        {
            String attribute = subModification.m_newAttributes[i];
            deletedAttributes.remove(attribute);
            newAttributes.put(attribute, "");
            m_deltaValues.put(attribute, subModification.m_deltaValues.get(attribute));
        }

        // Update modifications
        for (int i = 0; i < subModification.m_modifiedAttributes.length; i++)
        {
            String attribute = subModification.m_modifiedAttributes[i];
            if (newAttributes.get(attribute) != null) // This attribute does not exist in the super class
            {
                Object newValue = subModification.m_deltaValues.get(attribute);
                if (!(newValue instanceof DeltaAttributeSet))
                {
                    m_deltaValues.put(attribute, newValue);
                }
                else
                {
                    AttributeSet original = (AttributeSet)m_deltaValues.get(attribute);
                    try
                    {
                        original.applyDelta((DeltaAttributeSet)newValue);
                    }
                    catch (AttributeSetTypeException e)
                    {
                        throw new Error(e.toString());
                    }
                }

            }
            else if (modifiedAttributes.get(attribute) != null) // This attribute exists in the super class
            {
                Object newValue = subModification.m_deltaValues.get(attribute);
                Object oldValue = m_deltaValues.get(attribute);
                if (!(newValue instanceof DeltaAttributeSet))
                {
                    m_deltaValues.put(attribute, newValue);
                }
                else if (!(oldValue instanceof DeltaAttributeSet))
                {
                    try
                    {
                        ((AttributeSet)oldValue).applyDelta((DeltaAttributeSet)newValue);
                    }
                    catch (AttributeSetTypeException e)
                    {
                        throw new Error(e.toString());
                    }
                }
                else
                {
                    // both are deltas
                    ((DeltaAttributeSet)oldValue).adjustToSubclassedModification((DeltaAttributeSet)newValue);
                }
            }
            else // The attribute was modified for the first time at the super class
            {
                modifiedAttributes.put(attribute, "");
                m_deltaValues.put(attribute, subModification.m_deltaValues.get(attribute));
            }
        }

        // Now generate the new 'new',  'deleted' and 'modified' lists
        Iterator iterator = deletedAttributes.keySet().iterator();
        m_deletedAttributes = new String[deletedAttributes.size()];
        int j = 0;
        while (iterator.hasNext())
        {
            m_deletedAttributes[j++] = (String)iterator.next();
        }

        iterator = modifiedAttributes.keySet().iterator();
        m_modifiedAttributes= new String[modifiedAttributes.size()];
        j = 0;
        while (iterator.hasNext())
        {
            m_modifiedAttributes[j++] = (String)iterator.next();
        }

        iterator = newAttributes.keySet().iterator();
        m_newAttributes = new String[newAttributes.size()];
        j = 0;
        while (iterator.hasNext())
        {
            m_newAttributes[j++] = (String)iterator.next();
        }

    }

    // This delta represent subclassing. superModification represents a modification to the super class.
    // adjustToSuperclassModification modifies this to refelect the changes made is the super.
    // CannotAdjustToSuperModificationException is thrown if the subclassing using this delta cannot be applied anymore
    // because of the super's update
    void adjustToSuperclassModification(DeltaAttributeSet superModification) throws CannotAdjustToSuperModificationException
    {

        // Enter this.m_deletedAttributes into a hash table for fast update
        HashMap deletedAttributes = new HashMap();
        for (int i = 0; i < m_deletedAttributes.length; i++)
        {
            deletedAttributes.put(m_deletedAttributes[i], "");
        }


        // Enter this.m_modifiedAttributes into a hash table for fast update
        HashMap modifiedAttributes = new HashMap();
        for (int i = 0; i < m_modifiedAttributes.length; i++)
        {
            modifiedAttributes.put(m_modifiedAttributes[i], "");
        }


        // Delete  attributes that are no longer part of the super class
        for (int i = 0; i < superModification.m_deletedAttributes.length; i++)
        {
            String removedAttribute = superModification.m_deletedAttributes[i];
            deletedAttributes.remove(removedAttribute);
            modifiedAttributes.remove(removedAttribute);
        }

        // Now generate the new 'deleted' and 'modified' lists
        Iterator iterator = deletedAttributes.keySet().iterator();
        m_deletedAttributes = new String[deletedAttributes.size()];
        int j = 0;
        while (iterator.hasNext())
        {
            m_deletedAttributes[j++] = (String)iterator.next();
        }

        iterator = modifiedAttributes.keySet().iterator();
        m_modifiedAttributes= new String[modifiedAttributes.size()];
        j = 0;
        while (iterator.hasNext())
        {
            m_modifiedAttributes[j++] = (String)iterator.next();
        }


        // Go over superModification.m_modifiedAttributes and do the following and adjust this delta according to the
        // modifications at the super
        for (int i = 0; i < superModification.m_modifiedAttributes.length; i++)
        {
            Object thisDeltaNewValue = m_deltaValues.get(superModification.m_modifiedAttributes[i]);
            Object superDeltaNewValue = superModification.m_deltaValues.get(superModification.m_modifiedAttributes[i]);
            // If this delta overwrites the attribute or doesn't reffer to it we don't have to change anything
            if (thisDeltaNewValue == null || !(thisDeltaNewValue instanceof DeltaAttributeSet))
            {
                continue;
            }
            else if ((thisDeltaNewValue instanceof DeltaAttributeSet) && !(superDeltaNewValue instanceof DeltaAttributeSet))
            {
                // If the super delta overwrites a value this delta wants to modify then the adjustment failed
                throw new CannotAdjustToSuperModificationException(superModification.m_modifiedAttributes[i]);
            }
            else
            {
                // both thisDeltaNewValue && superDeltaNewValue are deltas
                // Sanity check
                if (!(thisDeltaNewValue instanceof DeltaAttributeSet) || !(superDeltaNewValue instanceof DeltaAttributeSet))
                {
                    throw new Error();
                }
                try
                {
                    ((DeltaAttributeSet)thisDeltaNewValue).adjustToSuperclassModification((DeltaAttributeSet)superDeltaNewValue);
                }
                catch (CannotAdjustToSuperModificationException e)
                {
                    throw new CannotAdjustToSuperModificationException(superModification.m_modifiedAttributes[i] +
                                                                       AttributeSet.ATTRIBUTE_SEPARATOR +
                                                                       e.getFailingAttribute());
                }
            }
        }

    }

    void revertToTemplate(AttributeName[] attributes)
    {
        for (int i = 0; i< attributes.length; i++)
        {
            revertToTemplate(attributes[i]);
        }
    }

    void revertToTemplate(AttributeName attribute)
    {
        m_deletedAttributes = revertInternal(m_deletedAttributes, attribute);
        m_modifiedAttributes = revertInternal(m_modifiedAttributes, attribute);
        m_newAttributes = revertInternal(m_newAttributes, attribute);
    }

    String[] revertInternal(String[] modArray, AttributeName attrName)
    {
        String[] newArray = (String[])modArray.clone();
        String firstComponent = (String)attrName.getComponent(0);
        Object value = getValue(firstComponent);
        for (int i=0; i< modArray.length; i++)
        {
            if (modArray[i].equals(firstComponent))
            {
                if ((value != null) && (value instanceof IDelta) && (attrName.getComponentCount() > 1))
                {
                    AttributeName subAttr = new AttributeName((String)attrName.getComponent(1));
                    for (int m=2; m< attrName.getComponentCount(); m++)
                    {
                        subAttr.setNextComponent((String)attrName.getComponent(m));
                    }
                    ( (DeltaAttributeSet)value).revertToTemplate(subAttr);
                    if (
                        (((DeltaAttributeSet) value).getDeletedAttributesNames().length == 0) &&
                        (((DeltaAttributeSet) value).getNewAttributesNames().length == 0) &&
                        (((DeltaAttributeSet) value).getModifiedAttributesNames().length == 0)
                        )
                    {
                        newArray = removeModification(modArray, i);
                        m_deltaValues.remove(firstComponent);
                    }
                }
                else
                {
                    newArray = removeModification(modArray, i);
                    m_deltaValues.remove(firstComponent);
                }
            }
        }
        return newArray;
    }

    String[] removeModification(String[] modArray, int removeIndex)
    {
        Vector newAttrs = new Vector(modArray.length);
        for (int j=0; j< modArray.length; j++)
        {
            if (j != removeIndex)
            {
                newAttrs.add(modArray[j]);
            }
        }
        String[] newArray = new String[newAttrs.size()];
        newAttrs.toArray(newArray);
        return newArray;
    }

    // Drill down and make sure there are no DeltaAttributeList list values
    boolean properForSubclassing()
    {
        Iterator iterator = m_deltaValues.values().iterator();
        while (iterator.hasNext())
        {
            Object value = iterator.next();
            if (value instanceof DeltaAttributeList)
            {
                return false;
            }
            else if (value instanceof DeltaAttributeSet && !((DeltaAttributeSet)value).properForSubclassing())
            {
                return false;
            }
            else
            {
                continue;
            }
        }

        return true;

    }

    boolean inSuperElement(AttributeName name)
    {
        String firstComponent = (String)name.getComponent(0);
        Object newValue = null;
        try
        {
            newValue = getNewValue(firstComponent);
        }
        catch (NotModifiedAttException e){}

        if (newValue == null)
        {
            return true;
        }

        // The last and only component is modified or overwritten
        if (name.getComponentCount() == 1)
        {
            return false;
        }

        if ((newValue instanceof AttributeSet) || (newValue instanceof AttributeList))
        {
            return false;
        }

        // newValue cannot be DeltaAttributeList since we don't generate DeltaAttributeList when subclassing
        // newValue cannot be a simple value since name has more components under newValue

        // So newValue must be a DeltaAttributeSet object
        return ((DeltaAttributeSet)newValue).inSuperElement(com.sonicsw.mf.common.config.query.impl.Util.removeFirstComponent(name));
    }


    // This must be a component of a subclassing delta. Returns the list of deleted attributes from the 'name' DeltaAttributeSet
    // object ('name' is relative to this).
    String[] getDeletedAttributes(AttributeName name)
    {
        // We want the list from this DeltaAttributeSet
        if (name.getComponentCount() == 0)
        {
            return getDeletedAttributes();
        }

        String firstComponent = (String)name.getComponent(0);
        Object newValue = null;
        try
        {
            newValue = getNewValue(firstComponent);
        }
        catch (NotModifiedAttException e){}

        if (newValue == null)
         {
            return new String[0]; // 'name' is not  in the delta at all so value is from the super
        }

        if ((newValue instanceof AttributeSet) || (newValue instanceof AttributeList))
        {
            return null;
        }

        // newValue cannot be DeltaAttributeList since we don't generate DeltaAttributeList when subclassing
        // newValue cannot be a simple value since name has more components under newValue

        // So newValue must be a DeltaAttributeSet object
        return ((DeltaAttributeSet)newValue).getDeletedAttributes(com.sonicsw.mf.common.config.query.impl.Util.removeFirstComponent(name));
    }

    private String[] getDeletedAttributes()
    {
        String[] deletedAttributes = new String[m_deletedAttributes.length];
        System.arraycopy(m_deletedAttributes, 0, deletedAttributes, 0, m_deletedAttributes.length);
        return deletedAttributes;
    }





}
