/*
 * Copyright (c) 2001 Sonic Software. All Rights Reserved.
 */

package com.sonicsw.mf.common.config.impl;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.util.HashMap;

import com.sonicsw.mf.common.config.AttributeSetTypeException;
import com.sonicsw.mf.common.config.IAttributeSet;
import com.sonicsw.mf.common.config.IDeltaElement;
import com.sonicsw.mf.common.config.IElementIdentity;
import com.sonicsw.mf.common.config.query.AttributeName;
import com.sonicsw.mf.common.dirconfig.IDeltaDirElement;

public final class DeltaElement implements IDeltaElement, IDeltaDirElement, IDelta, java.io.Serializable, ICanReplaceRef
{
  private static final long serialVersionUID = 0L;
  private final static int SERIALIZATION_VERSION = 3;

  private boolean m_isDeleted;
  private ElementIdentity m_identity;
  private Object m_attributeSet;

  DeltaElement(ElementIdentity identity, Object attributeSet)
  {
      m_isDeleted = false;
      m_identity = identity;
      m_attributeSet = attributeSet;
  }

  private void writeObject(java.io.ObjectOutputStream s) throws java.io.IOException
  {
       s.writeInt(SERIALIZATION_VERSION);
       s.writeBoolean(m_isDeleted);
       s.writeObject(m_identity);
       s.writeObject(m_attributeSet);
  }
  private void readObject(java.io.ObjectInputStream s) throws java.io.IOException, ClassNotFoundException
  {
      int version = s.readInt();
      if (version != SERIALIZATION_VERSION)
    {
        Util.throwSerialVersionMismatch(version, SERIALIZATION_VERSION);
    }

      m_isDeleted = s.readBoolean();
      m_identity = (ElementIdentity)s.readObject();
      m_attributeSet = s.readObject();

  }

  public DeltaElement createEmptyDelta(ElementIdentity identity)
  {
       return new DeltaElement(identity, new DeltaAttributeSet(null, new HashMap(), new HashMap()));
  }

  @Override
public boolean emptyDelta()
  {
      if (m_attributeSet instanceof DeltaAttributeSet)
    {
        return ((DeltaAttributeSet)m_attributeSet).emptyDelta();
    }
    else
    {
        return false;
    }
  }

  public void setIdentity(ElementIdentity identity)
  {
      m_identity = identity;
  }

  @Override
public IElementIdentity getIdentity()
  {
      return m_identity;
  }

  @Override
public  boolean isDeleted()
  {
      return m_isDeleted;
  }

  @Override
public Object getDeltaAttributes()
  {
      if (m_isDeleted)
    {
        return null;
    }

      return m_attributeSet;
  }

  @Override
public String toString()
  {
      if (m_identity != null)
    {
        return m_identity.toString();
    }
    else
    {
        return null;
    }
  }

  public int estimateSize()
  {
      DeltaAttributeSet attributes = (DeltaAttributeSet)getDeltaAttributes();
      if (attributes != null)
    {
        return attributes.estimateSize() + m_identity.estimateSize();
    }
    else
    {
        return m_identity.estimateSize();
    }
  }


    // Translate this delta to an Element - USED FOR DEBUGGING ONLY !!!
    // The only usage for this is to be exported to XML! THE STRUCTURE IS NOT CORRECT
    // FOR ANY OTHER USAGE.
    public Element toElement(String name) throws Exception
    {
        Element element = new Element(name, "DELTA ELEMENT", "1.0");
        if (m_attributeSet instanceof DeltaAttributeSet)
        {
            IAttributeSet newAttributes = element.getAttributes();
            ((DeltaAttributeSet)m_attributeSet).toAttributeSet(newAttributes);
        }
        else
        {
            element.doApplyDelta(this);
        }

        return element;
    }

    // The input for this routine is an attribute name from a subclassed element generated from this delta.
    // It returns true if the value comes from the super element (and false if the value comes from the delta).
    boolean inSuperElement(AttributeName name)
    {
        if (!(m_attributeSet instanceof DeltaAttributeSet))
        {
            return false; // The whole attribute set is over written by the delta object
        }
        else
        {
            return ((DeltaAttributeSet)m_attributeSet).inSuperElement(name);
        }
    }

    // The input for this routine is an attribute set name from a subclassed element generated from this delta.
    // It returns the list of deleted attribute names (ralative to the super) or null if the entire attribute
    // set was replaced.
    String[] getDeletedAttributes(AttributeName name)
    {
        if (!(m_attributeSet instanceof DeltaAttributeSet))
        {
            return null; // The whole attribute set is over written by the delta object
        }
        else
        {
            return ((DeltaAttributeSet)m_attributeSet).getDeletedAttributes(name);
        }
    }


    // This delta element represent subclassing. subModification represent a modification to the subclassed object.
    // adjustToSubclassedModification modifies this to refelect the changes made to subclassed object.
    public void adjustToSubclassedModification(DeltaElement subModification)
    {
        if (!(m_attributeSet instanceof DeltaAttributeSet) && !(subModification.m_attributeSet instanceof DeltaAttributeSet))
        {
            m_attributeSet = subModification.m_attributeSet;
        }
        else if (!(m_attributeSet instanceof DeltaAttributeSet))
        {
            try
            {
                ((AttributeSet)m_attributeSet).applyDelta((DeltaAttributeSet)subModification.m_attributeSet);
            }
            catch (AttributeSetTypeException e)
            {
                throw new Error(e.toString());
            }
        }
        else if (!(subModification.m_attributeSet instanceof DeltaAttributeSet))
        {
            m_attributeSet = subModification.m_attributeSet;
        }
        else
        {
            ((DeltaAttributeSet)m_attributeSet).adjustToSubclassedModification((DeltaAttributeSet)subModification.m_attributeSet);
        }
    }

    // This delta represent subclassing. superModification represents a modification to the super class.
    // adjustToSuperclassModification modifies this to refelect the changes made is the super.
    // CannotAdjustToSuperModificationException is thrown if the subclassing using this delta cannot be applied anymore
    // because of the super's update
    public void adjustToSuperclassModification(DeltaElement superModification) throws CannotAdjustToSuperModificationException
    {
        if (!(m_attributeSet instanceof DeltaAttributeSet))
        {
            return;
        }
        else if (!(superModification.m_attributeSet instanceof DeltaAttributeSet))
        {
            throw new CannotAdjustToSuperModificationException("top level element attributes");
        }
        else
        {
            ((DeltaAttributeSet)m_attributeSet).adjustToSuperclassModification(((DeltaAttributeSet)superModification.m_attributeSet));
        }
    }


    // Make sure this doesn't contain DeltaAttributeList objects
    public boolean properForSubclassing()
    {
        if (m_attributeSet instanceof DeltaAttributeSet)
        {
            return ((DeltaAttributeSet)m_attributeSet).properForSubclassing();
        }
        else
        {
            return true;
        }
    }

  public DeltaElement createClone()
  {
       return fromBytes(toBytes());
  }

  public byte[] toBytes()
  {
      try
      {
          ByteArrayOutputStream out = new ByteArrayOutputStream();
          ObjectOutputStream objectOut = new ObjectOutputStream(out);
          objectOut.writeObject(this);
          byte[] bytes = out.toByteArray();
          objectOut.close();
          return bytes;
      }
      catch (Exception e)
      {
          throw new Error("toBytes failed.");
      }
  }

  public static DeltaElement fromBytes(byte[] bytes)
  {
      try
      {
          ByteArrayInputStream in = new ByteArrayInputStream(bytes);
          ObjectInputStream objectIn = new ObjectInputStream(in);
          return (DeltaElement)objectIn.readObject();
      }
      catch (Exception e)
      {
          throw new Error("fromBytes failed.");
      }
  }

   // Replaces all references using the IReplaceRef replace service
   @Override
public boolean replaceReferences(boolean isSystemAtts, IReplaceRef replaceSrvc)
   {
       return ((ICanReplaceRef)m_attributeSet).replaceReferences(false, replaceSrvc);
   }


}
