/*
 * Copyright (c) 2001 Sonic Software. All Rights Reserved.
 */

package com.sonicsw.mf.common.config.impl;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;

import com.sonicsw.mx.util.IEmptyArray;

import com.sonicsw.mf.common.config.AttributeSetTypeException;
import com.sonicsw.mf.common.config.ConfigException;
import com.sonicsw.mf.common.config.IAttributeSet;
import com.sonicsw.mf.common.config.IBasicElement;
import com.sonicsw.mf.common.config.IBlob;
import com.sonicsw.mf.common.config.IDeltaElement;
import com.sonicsw.mf.common.config.IElement;
import com.sonicsw.mf.common.config.IElementIdentity;
import com.sonicsw.mf.common.config.IMFDirectories;
import com.sonicsw.mf.common.config.INextVersionToken;
import com.sonicsw.mf.common.config.ReadOnlyException;
import com.sonicsw.mf.common.config.Reference;
import com.sonicsw.mf.common.config.query.AttributeName;
import com.sonicsw.mf.common.dirconfig.IDirElement;


public class Element extends ElementNode implements IDirElement, ICanReplaceRef
{
  private static final long serialVersionUID = 0L;
  private final static int SERIALIZATION_VERSION = 3;
  private final static int USER_SERIALIZATION_VERSION = 100;
  private final static int EXTERNAL_USER_SERIALIZATION_VERSION = 101;

  final static String USER_NAME_ATTRIBUTE = "USER_NAME";
  final static String USER_PASSWORD_ATTRIBUTE = "PASSWORD";             //NOSONAR field change is not required.

  static final String ELEMENT_ATTRIBUTES =  AttributeSet.ATTRIBUTE_SEPARATOR;

  static final String SYSTEM_ATTRIBUTES = "_MF_SYSTEM_ATTRIBUTES";
  static final String RELEASE_VERSION = "RELEASE_VERSION";
  static final String SUBCLASS_FROM_THIS_SUPER = "SUBCLASS_FROM_THIS_SUPER";
  static final String SUPER_REFERENCE = "SUPER_REFERENCE";
  static final String IS_SUPER_ELEMENT = "IS_SUPER_ELEMENT";
  static final String IS_TEMPLATE = "IS_TEMPLATE";
  static final String SUBCLASSING_DELTA = "SUBCLASSING_DELTA";
  static final String IS_SUBCLASSED_ELEMENT = "IS_SUBCLASSED_ELEMENT";
  static final String SUBCLASSED_ELEMENTS = "SUBCLASSED_ELEMENTS";


  // User types (we don't distinguish between external user and DS managed user yet).
  private static final String MF_USER = "MF_AUTHENTICATION_USER";
  private static final String MF_EXTERNAL_USER = "MF_AUTHENTICATION_USER";


  private ElementIdentity m_identity;
  private AttributeSet m_attributes;
  private boolean m_deleted;

  private transient IElementCache m_cache;
  private transient boolean doneUpdateCalled;
  private transient boolean m_dontWriteAttributes;
  private transient IDeltaElement m_subclassingDelta; // Cahce here the subclassing delta so we don't need to deserialize every time

  private static HashMap m_directories = new HashMap();

  // Used for serialization optimization - when many elements are serialized together - the directory
  // name will be serialized only once.
  private static String getDirectoryName(String dirName)
  {
      synchronized (m_directories)
      {
          String retDir = (String)m_directories.get(dirName);
          if (retDir == null)
          {
              retDir = dirName;
              m_directories.put(dirName, dirName);
          }
          return retDir;
      }
  }

  public void dontWriteAttributes(boolean dont)
  {
      m_dontWriteAttributes = dont;
  }

  public void setDeleted()
  {
      m_deleted = true;
      m_readOnly = true;
  }

  // This is used when we update the content of a caches element from the DS and the DS version is
  // higher than the cached version
  public void setNewIdentity(ElementIdentity id)
  {
      m_identity = id;
  }

  @Override
public IDirElement getNextVersion(INextVersionToken token)
  {

       boolean internalDSToken = token instanceof InternalDSToken;

       if (m_deleted)
    {
        throw new IllegalStateException("getNextVersion cannot be called on deleted elements.");
    }

       if (token == null)
    {
        throw new IllegalStateException("A null token was passed to getNextVersion().");
    }

       ElementIdentity tokenID = null;
       if (!internalDSToken)
    {
        tokenID = ((NextVersionToken)token).getID(m_identity.getName());
    }

       if (!internalDSToken && tokenID == null)
    {
        throw new IllegalStateException("No next version token for " + m_identity);
    }


       if (internalDSToken)
       {
           if (m_identity.getCreationTimestamp() == 0)
        {
            throw new IllegalStateException("The InternalDSToken cannot be used on new elements.");
        }
           long version = m_identity.getVersion();
           Element newElement = (Element)createWritableClone();
           ((ElementIdentity)newElement.getIdentity()).setVersion(version+1);
           return (IDirElement)newElement;
       }
       else if (tokenID == null)
       {
           throw new IllegalStateException("Invalid token used by getNextVersion on " + m_identity + " token id: null");
       }
       else if (m_identity.getCreationTimestamp() != 0)
       {
           if (m_identity.getVersion() < tokenID.getVersion())
        {
            throw new IllegalStateException("Invalid token used by getNextVersion on " + m_identity + " token id: " + tokenID);
        }
        else if (!m_identity.equalVersion(tokenID))
        {
            return this;
        }
       }
       else if (tokenID.getVersion()  != 1)
       {
            throw new IllegalStateException("Invalid token used by getNextVersion on " + m_identity +
                                            " token id: " + tokenID);
       }

       if (!m_readOnly)
    {
        throw new IllegalStateException("getNextVersion should be called on read-only elements.");
    }

       Element newElement = (Element)createWritableClone();
       ElementIdentity newID = tokenID.createClone();
       newID.setVersion(m_identity.getVersion() + 1);
       newElement.setNewIdentity(newID);

       return (IDirElement)newElement;
  }

  public Element(String name, String elementTypeName, String releaseVersion, boolean createTemplate)
  {
      this(name, elementTypeName, releaseVersion);
      if (createTemplate)
    {
        setTemplate();
    }
  }

  public Element(String name, String elementTypeName, String releaseVersion, String tamplateName)
  {
      this(name, elementTypeName, releaseVersion);
      if (tamplateName != null)
    {
        addSuperToSubclassFrom(tamplateName);
    }
  }

  public Element(String name, String elementTypeName, String releaseVersion)
  {
       super("", null);
       m_identity = new ElementIdentity(name, elementTypeName, releaseVersion);
       m_attributes = new AttributeSet (ELEMENT_ATTRIBUTES, this, null);
       m_deleted = false;
       doneUpdateCalled = false;
       m_dontWriteAttributes = false;
       m_cache = null;
  }

  private Element(ElementIdentity id, boolean deleted)
  {
      super("", null);
      m_identity = id;
      m_deleted = deleted;
      m_attributes = null;
      m_readOnly = true;
  }


  @Override
public void setName(String newName) throws ReadOnlyException
  {
       if (m_readOnly)
    {
        throw new ReadOnlyException();
    }

       if (!m_isNew)
    {
        throw new IllegalStateException("Only new elements can be renamed.");
    }

       m_identity.setName(newName);

  }

  // Calculate the delta between this and element
  public DeltaElement createDelta(Element element)
  {
      if (isDeleted() || element.isDeleted())
    {
        throw new Error();
    }

      Object deltaAttributeSet = Util.createDelta(getAttributes(), null, element.getAttributes(), null);

      // If there is no delta (this and element are equivalent) we create an empty DeltaAttributeSet object
      // and use that since null is not legal for a DeltaElement
      if (deltaAttributeSet == null)
    {
        deltaAttributeSet = new DeltaAttributeSet(null, new HashMap(), new HashMap());
    }

      return new DeltaElement(m_identity, deltaAttributeSet);
  }

  public Element(IElementIdentity id)
  {
      this(id, true);
  }
  
  // This constructor allows the element to be created with isNew == false.
  // This is used by the Eclipse handler when crafting DS elements from files 
  // in the workspace. Because the handler doesn't really get elements from the
  // DS, but crafts new elements every time one is needed, we need to mimic
  // isNew == false.
  //This is necessary because the combination of isNew = true and doneUpdateCalled = false
  // makes it so that we get an exception when the SMC connects directly
  // to the DS and we browse workspace files.
  // When the SMC connects to the DS through JMS, and the element is serialized,
  // the isNew flag is set to false when the element is reconstructed by readObject,
  // and so we don't have the same problem.
  public Element(IElementIdentity id, boolean isNew)
  {
	  super("", null);
      m_identity = ((ElementIdentity)id).createClone();
      m_attributes = new AttributeSet (ELEMENT_ATTRIBUTES, this, null);
      m_deleted = false;
      doneUpdateCalled = false;
      m_dontWriteAttributes = false;
      m_cache = null;
      m_isNew = isNew;
  }

  public int estimateSize()
  {
      AttributeSet attributes = (AttributeSet)getAttributes();
      if (attributes != null)
    {
        return attributes.estimateSize() + m_identity.estimateSize();
    }
    else
    {
        return m_identity.estimateSize();
    }
  }

  public static IDirElement createDeletedElement(IElementIdentity id)
  {
      Element element = new Element(id);
      element.m_deleted = true;
      element.m_readOnly = true;
      element.doneUpdateCalled = false;
      return element;
  }

  // This method is used for connecting the element to the cache that holds
  // its content
  public void setCache(IElementCache cache)
  {
      if (!m_readOnly)
    {
        throw new Error("Only read-only elements can be cached.");
    }

      m_cache = cache;
  }

  public void cacheMe(IElementCache cache)
  {
      if (!m_readOnly)
    {
        throw new Error("Only read-only elements can be cached.");
    }

      m_cache = cache;
      cacheAttributes();
  }

  private void cacheAttributes()
  {
      if (m_cache == null || m_attributes == null)
    {
        return;
    }

      m_cache.storeAttributes(this);

      // Now must dereference the attributes to allow memory reuse
      m_attributes = null;
  }


  private void writeObject(java.io.ObjectOutputStream s) throws java.io.IOException
  {
       String type = m_identity.getType();
       if (type != null)
       {
           boolean userType = type.equals(MF_USER);
           boolean externalUserType = type.equals(MF_EXTERNAL_USER);
           boolean isEnvelope = this instanceof EnvelopeElement;
           if (!isEnvelope && (userType || externalUserType))
           {
               writeUserObject((userType ? USER_SERIALIZATION_VERSION : EXTERNAL_USER_SERIALIZATION_VERSION), s);
               return;
           }
       }

       s.writeInt(SERIALIZATION_VERSION);
       s.writeObject(m_identity);
       if (m_dontWriteAttributes)
    {
        s.writeObject(null);
    }
    else
    {
        s.writeObject(getAttributes());
    }
       s.writeBoolean(m_deleted);
  }

  private void writeUserObject(int type, java.io.ObjectOutputStream s) throws java.io.IOException
  {
       s.writeInt(type);

       /***********************/
       /* Write the identity  */
       /***********************/

       // Write tthe directory separate from the base name so that the when many users under the
       // same directory are serialized, the directory name is written only once.
       EntityName fullName = getEntity(m_identity.getName());
       s.writeObject(Element.getDirectoryName(fullName.getParent()));
       s.writeObject(fullName.getBaseName());

       s.writeLong(m_identity.getVersion());
       s.writeLong(m_identity.getCreationTimestamp());
       // s.writeObject(m_identity.getType() -   not written it's implicit from the serialization type
       s.writeObject(m_identity.getReleaseVersion());

       /**********************************************/
       /* Write the ElementNode  - the super of this */
       /**********************************************/
       // s.writeObject(m_name) - don't need to be written - always ""
       s.writeBoolean(m_readOnly);
       s.writeBoolean(m_deleted);

       /**********************************************/
       /* Write the user name and password           */
       /**********************************************/
       if (!m_deleted)
       {
           if (m_dontWriteAttributes)
           {
               s.writeBoolean(true);
               return;
           }
        else
        {
            s.writeBoolean(false);
        }

           IAttributeSet attributes = getAttributes();
           s.writeObject(attributes.getAttribute(USER_NAME_ATTRIBUTE));
           s.writeObject(attributes.getAttribute(USER_PASSWORD_ATTRIBUTE));
       }

  }

  private void readUserObject(int type, java.io.ObjectInputStream s) throws java.io.IOException, ClassNotFoundException
  {
      doneUpdateCalled = false;

      // Read the identity
      String dirName = (String)s.readObject();
      String baseName = (String)s.readObject();
      String name = dirName + IMFDirectories.MF_DIR_SEPARATOR + baseName;
      long version = s.readLong();
      long creationTimestamp= s.readLong();
      String elementType = (type == EXTERNAL_USER_SERIALIZATION_VERSION) ? MF_EXTERNAL_USER : MF_USER;
      String releaseVersion = (String)s.readObject();
      m_identity = new ElementIdentity(name, elementType, releaseVersion);
      m_identity.setVersion(version);
      m_identity.setCreationTimestamp(creationTimestamp);

      // Get the fields of the super
      m_name = "";
      m_readOnly = s.readBoolean();
      m_deleted = s.readBoolean();

      if (m_deleted)
      {
          m_attributes = new AttributeSet (ELEMENT_ATTRIBUTES, this, null);
          m_attributes.setReadOnly(m_readOnly);
          return;
      }


      boolean didntWriteAttributes = s.readBoolean();
      if (didntWriteAttributes)
    {
        m_attributes = null;
    }
    else
      {
          String user = (String)s.readObject();
          byte[] password = (byte[])s.readObject();
          m_attributes = new AttributeSet (ELEMENT_ATTRIBUTES, this, user, password);
          m_attributes.setReadOnly(m_readOnly);
      }



  }

  private void readObject(java.io.ObjectInputStream s) throws java.io.IOException, ClassNotFoundException
  {
      int version = s.readInt();
      if (version != SERIALIZATION_VERSION && version != EXTERNAL_USER_SERIALIZATION_VERSION && version != USER_SERIALIZATION_VERSION)
    {
        Util.throwSerialVersionMismatch(version, SERIALIZATION_VERSION);
    }

      if (version == EXTERNAL_USER_SERIALIZATION_VERSION || version == USER_SERIALIZATION_VERSION)
      {
          readUserObject(version, s);
          return;
      }

      m_identity = (ElementIdentity)s.readObject();
      m_attributes = (AttributeSet)s.readObject();
      m_deleted = s.readBoolean();
      doneUpdateCalled = false;
  }

  public boolean isReadOnly() { return m_readOnly;}

  @Override
public boolean isDeleted()
  {
      return m_deleted;
  }

  @Override
public boolean isDirty()
  {
      return (m_isModified || m_isNew) && !doneUpdateCalled;
  }

  @Override
public IBasicElement doneUpdate() throws ReadOnlyException
  {
     return doneUpdate(false);
  }

  @Override
public IBasicElement doneUpdateForSubclassing() throws ReadOnlyException
  {
      // Make sure this is not subclassed already
      String superRef = getSuperElementName();
      if (superRef != null)
    {
        throw new Error(m_identity.getName() + " is already subclassed from " + superRef + ".");
    }
 
      addSuperAttribute(m_identity.getName(), false);
 
      return doneUpdate(true);
  }

  public void removeSystemAttributes() throws ReadOnlyException, ConfigException
  {
      ((IAttributeSet)getAttributes()).deleteAttribute(SYSTEM_ATTRIBUTES);
  }

  @Override
public String getArchiveName()
  {
      String archiveName = null;
      IAttributeSet topSet = getAttributes();
      IAttributeSet systemAttrs = (IAttributeSet)topSet.getAttribute(IBlob.SYSTEM_ATTRIBUTES);
      if (systemAttrs != null)
    {
        archiveName = (String)systemAttrs.getAttribute(IBlob.ARCHIVE_NAME);
    }
      return archiveName;
  }

  // Return the name of the super element (if this is subclassed) or null (if not subclassed)
  @Override
public String getSuperElementName()
  {
     IAttributeSet systemAttributes = (IAttributeSet)getAttributes().getAttribute(SYSTEM_ATTRIBUTES);
     if (systemAttributes != null)
    {
        return (String)systemAttributes.getAttribute(SUPER_REFERENCE);
    }
    else
    {
        return null;
    }
  }

  // Remove the reference to the super element
  public void removeSuperElementName()
  {
     IAttributeSet systemAttributes = (IAttributeSet)m_attributes.getAttribute(SYSTEM_ATTRIBUTES);
     if (systemAttributes == null)
    {
        return;
    }
     try
     {
         systemAttributes.deleteAttribute(SUPER_REFERENCE);
     }
     catch (Exception e)
     {
         throw new Error(); // Should not happen
     }

  }


  public boolean isSuperElement()
  {
     IAttributeSet systemAttributes = (IAttributeSet)getAttributes().getAttribute(SYSTEM_ATTRIBUTES);
     if (systemAttributes != null)
     {
         Boolean isSuper = (Boolean)systemAttributes.getAttribute(IS_SUPER_ELEMENT);
         if (isSuper == null)
        {
            return false;
        }
        else
        {
            return isSuper.booleanValue();
        }
     }
    else
    {
        return false;
    }
  }

  @Override
public boolean isSubclassedElement()
  {
     IAttributeSet systemAttributes = (IAttributeSet)getAttributes().getAttribute(SYSTEM_ATTRIBUTES);
     if (systemAttributes != null)
     {
         Boolean isSub = (Boolean)systemAttributes.getAttribute(IS_SUBCLASSED_ELEMENT);
         if (isSub == null)
        {
            return false;
        }
        else
        {
            return isSub.booleanValue();
        }
     }
    else
    {
        return false;
    }
  }


  public byte[] getSubclassedDelta(String subClassedName)
  {
       IAttributeSet systemAttributes = (IAttributeSet)getAttributes().getAttribute(SYSTEM_ATTRIBUTES);
       if (systemAttributes == null)
    {
        return null;
    }
       IAttributeSet subclassedItems = (IAttributeSet)systemAttributes.getAttribute(SUBCLASSED_ELEMENTS);
       if (subclassedItems == null)
    {
        return null;
    }
       return (byte[])subclassedItems.getAttribute(subClassedName);
  }

  public void removeSubclassedDelta(String subClassedName)
  {
      try
      {
          IAttributeSet systemAttributes = (IAttributeSet)m_attributes.getAttribute(SYSTEM_ATTRIBUTES);
          if (systemAttributes == null)
        {
            return;
        }
          IAttributeSet subclassedItems = (IAttributeSet)systemAttributes.getAttribute(SUBCLASSED_ELEMENTS);
          if (subclassedItems == null)
        {
            return;
        }
          subclassedItems.deleteAttribute(subClassedName);

          // If this was the last one then this element is not a super element any more

          HashMap allAttributes =  subclassedItems.getAttributes();
          if (allAttributes.isEmpty())
          {
              systemAttributes.deleteAttribute(SUBCLASSED_ELEMENTS);
              systemAttributes.deleteAttribute(IS_SUPER_ELEMENT);
          }
      }
      catch (Exception e)
      {
          throw new Error(); // Should not happen
      }

  }


  @Override
public String[] getSubclassedList()
  {
      IAttributeSet systemAttributes = (IAttributeSet)getAttributes().getAttribute(SYSTEM_ATTRIBUTES);
       if (systemAttributes == null)
    {
        return IEmptyArray.EMPTY_STRING_ARRAY;
    }

       IAttributeSet subclassedItems = (IAttributeSet)systemAttributes.getAttribute(SUBCLASSED_ELEMENTS);
       if (subclassedItems == null)
    {
        return IEmptyArray.EMPTY_STRING_ARRAY;
    }

       HashMap allItems = subclassedItems.getAttributes();
       String[] result = new String[allItems.size()];
       Iterator iterator = allItems.keySet().iterator();
       int i = 0;
       while (iterator.hasNext())
    {
        result[i++] = (String)iterator.next();
    }
       return result;
  }

  // Create a delta element to express the adding of a subclassed element name to the super's list
  public static DeltaElement addSubToSuperDelta(String subName, Element original)
  {
      try
      {
          // FIrst, discover waht atributes the original super element has
          IAttributeSet origAttributes = original.getAttributes();
          IAttributeSet origSystemAttributes = (IAttributeSet)origAttributes.getAttribute(SYSTEM_ATTRIBUTES);
          IAttributeSet origSubclassed = null;
          if (origSystemAttributes != null)
        {
            origSubclassed = (IAttributeSet)origSystemAttributes.getAttribute(SUBCLASSED_ELEMENTS);
        }

          // The delta between base and modifiedBase will be the added subName list name
          Element base = new Element(original.getIdentity());
          IAttributeSet attributes = base.getAttributes();
          IAttributeSet systemAttributes = null;
          if (origSystemAttributes != null)
        {
            systemAttributes = attributes.createAttributeSet(SYSTEM_ATTRIBUTES);
        }
          if (origSubclassed != null)
         {
            systemAttributes.createAttributeSet(SUBCLASSED_ELEMENTS); // Create an empty SUBCLASSED_ELEMENTS list
        }
          base.doneUpdate();

          Element modifiedBase = (Element)base.createWritableClone();
          attributes = modifiedBase.getAttributes();
          systemAttributes = (IAttributeSet)attributes.getAttribute(SYSTEM_ATTRIBUTES);
          if (systemAttributes == null)
        {
            systemAttributes = attributes.createAttributeSet(SYSTEM_ATTRIBUTES);
        }
          IAttributeSet subList = (IAttributeSet)systemAttributes.getAttribute(SUBCLASSED_ELEMENTS);
          if (subList == null)
        {
            subList = systemAttributes.createAttributeSet(SUBCLASSED_ELEMENTS);
        }
          subList.setBytesAttribute(subName, new byte[0]);
          return (DeltaElement)modifiedBase.doneUpdate();

      }
      catch (Exception e)
      {
          throw new Error(); // Should not happen
      }

  }

  // Create a delta element to express the removeing of a subclassed element name from the super's list
  public static DeltaElement removeSubToSuperDelta(String[] list, String itemToRemove, IElementIdentity id)
  {
      try
      {
          // The delta between base and modifiedBase will be the removing of subName list name

          Element base = new Element(id);
          IAttributeSet attributes = base.getAttributes();
          IAttributeSet systemAttributes = attributes.createAttributeSet(SYSTEM_ATTRIBUTES);
          IAttributeSet subList = systemAttributes.createAttributeSet(SUBCLASSED_ELEMENTS);
          for (int i = 0; i < list.length; i++)
        {
            subList.setBytesAttribute(list[i], new byte[0]);
        }
          base.doneUpdate();

          Element modifiedBase = (Element)base.createWritableClone();
          attributes = modifiedBase.getAttributes();
          systemAttributes = (IAttributeSet)attributes.getAttribute(SYSTEM_ATTRIBUTES);
          subList = (IAttributeSet)systemAttributes.getAttribute(SUBCLASSED_ELEMENTS);
          subList.deleteAttribute(itemToRemove); // Delete the itemToRemove item from the list
          return (DeltaElement)modifiedBase.doneUpdate();
      }
      catch (Exception e)
      {
          throw new Error(); // Should not happen
      }

  }


  public void cleanupSuperAttributes(boolean deleteAll, boolean clearTamplate)
  {
     try
     {
         IAttributeSet systemAttributes = (IAttributeSet)m_attributes.getAttribute(SYSTEM_ATTRIBUTES);
         if (systemAttributes == null)
        {
            return;
        }

         if (clearTamplate)
        {
            systemAttributes.deleteAttribute(IS_TEMPLATE);
        }

         systemAttributes.deleteAttribute(IS_SUPER_ELEMENT);
         if (deleteAll)
         {
             systemAttributes.deleteAttribute(SUBCLASSED_ELEMENTS);
             return;
         }

         IAttributeSet subclassedItems = (IAttributeSet)systemAttributes.getAttribute(SUBCLASSED_ELEMENTS);
         if (subclassedItems == null)
        {
            return;
        }

         // Remove the binary delta objects
         Iterator iterator = subclassedItems.getAttributes().keySet().iterator();
         while (iterator.hasNext())
        {
            subclassedItems.setBytesAttribute((String)iterator.next(), new byte[0]);
        }
     }
     catch (Exception e)
     {
         throw new Error(e.toString()); // Should never happen
     }
  }

  public void addSubclassedElement(String subClassedName, byte[] serializedDelta)
  {
     try
     {
         IAttributeSet systemAttributes = (IAttributeSet)m_attributes.getAttribute(SYSTEM_ATTRIBUTES);
         if (systemAttributes == null)
        {
            systemAttributes = m_attributes.createAttributeSet(SYSTEM_ATTRIBUTES);
        }

         systemAttributes.setBooleanAttribute(IS_SUPER_ELEMENT, Boolean.TRUE);

         IAttributeSet subclassedItems = (IAttributeSet)systemAttributes.getAttribute(SUBCLASSED_ELEMENTS);
         if (subclassedItems == null)
        {
            subclassedItems = systemAttributes.createAttributeSet(SUBCLASSED_ELEMENTS);
        }

         subclassedItems.setBytesAttribute(subClassedName, serializedDelta);
     }
     catch (Exception e)
     {
         throw new Error(e.toString()); // Should never happen
     }
  }

  public void addReleaseVersion(String releaseVersion)
  {
     try
     {
         IAttributeSet systemAttributes = (IAttributeSet)m_attributes.getAttribute(SYSTEM_ATTRIBUTES);
         if (systemAttributes == null)
        {
            systemAttributes = m_attributes.createAttributeSet(SYSTEM_ATTRIBUTES);
        }

         systemAttributes.setStringAttribute(RELEASE_VERSION, releaseVersion);
     }
     catch (Exception e)
     {
         throw new Error(e.toString()); // Should never happen
     }
  }

  private void addSuperToSubclassFrom(String superName)
  {
     try
     {
         IAttributeSet systemAttributes = (IAttributeSet)m_attributes.getAttribute(SYSTEM_ATTRIBUTES);
         if (systemAttributes == null)
        {
            systemAttributes = m_attributes.createAttributeSet(SYSTEM_ATTRIBUTES);
        }

         systemAttributes.setReferenceAttribute(SUBCLASS_FROM_THIS_SUPER, new Reference(superName));
     }
     catch (Exception e)
     {
         throw new Error(e.toString()); // Should never happen
     }
  }

  // Return the name of the super element (if this is subclassed) or null (if not subclassed)
  public String getSuperToSubclassFrom()
  {
     IAttributeSet systemAttributes = (IAttributeSet)getAttributes().getAttribute(SYSTEM_ATTRIBUTES);
     if (systemAttributes != null)
     {
         Reference ref = (Reference)systemAttributes.getAttribute(SUBCLASS_FROM_THIS_SUPER);
         if (ref == null)
        {
            return null;
        }
        else
        {
            return ref.getElementName();
        }
     }
    else
    {
        return null;
    }
  }



  public final void setTemplate()
  {
     try
     {
         IAttributeSet systemAttributes = (IAttributeSet)m_attributes.getAttribute(SYSTEM_ATTRIBUTES);
         if (systemAttributes == null)
        {
            systemAttributes = m_attributes.createAttributeSet(SYSTEM_ATTRIBUTES);
        }

         systemAttributes.setBooleanAttribute(IS_TEMPLATE, Boolean.TRUE);
     }
     catch (Exception e)
     {
         throw new Error(e.toString()); // Should never happen
     }
  }

  private void replaceSubclassingReferences(IReplaceRef replaceSrvc)
  {
     try
     {
          IAttributeSet systemAttributes = (IAttributeSet)m_attributes.getAttribute(SYSTEM_ATTRIBUTES);
          if (systemAttributes == null)
        {
            return;
        }

          String superRef = (String)systemAttributes.getAttribute(SUPER_REFERENCE);
          if (superRef != null)
        {
            ((AttributeSet)systemAttributes).setAttributeObjectNoHistory(SUPER_REFERENCE,  replaceSrvc.replace(superRef));
        }
          String[] subclassedList =  getSubclassedList();

          if (subclassedList.length == 0)
        {
            return;
        }

          AttributeSet subclassedItems = (AttributeSet)systemAttributes.getAttribute(SUBCLASSED_ELEMENTS);

          for (int i = 0; i < subclassedList.length; i++)
        {
            subclassedItems.renameNoHistory(subclassedList[i], replaceSrvc.replace(subclassedList[i]));
        }
     }
     catch (Exception e)
     {
         e.printStackTrace();
         throw new Error("Subclassing inconsistency: " + e.toString()); // Should never happen
     }


  }

  private void replaceSubclassingDelta(IReplaceRef replaceSrvc)
  {
     try
     {
         IAttributeSet systemAttributes = (IAttributeSet)m_attributes.getAttribute(SYSTEM_ATTRIBUTES);
         if (systemAttributes == null)
        {
            return;
        }

         byte[] deltaBytes = (byte[])systemAttributes.getAttribute(SUBCLASSING_DELTA);

         if (deltaBytes == null)
        {
             //The application didn't ask to get this element with its subclassing delta
            return;
        }
        else
         {
             DeltaElement newDelta = DeltaElement.fromBytes(deltaBytes);
             newDelta.replaceReferences(false, replaceSrvc);
             ((AttributeSet)systemAttributes).setAttributeObjectNoHistory(SUBCLASSING_DELTA, newDelta.toBytes());
             m_subclassingDelta = null;
         }

     }
     catch (Exception e)
     {
         throw new Error(e.toString()); // Should never happen
     }

  }

  public void setSubclassingDelta(byte[] delta)
  {
     try
     {
         IAttributeSet systemAttributes = (IAttributeSet)m_attributes.getAttribute(SYSTEM_ATTRIBUTES);
         if (systemAttributes == null)
        {
            systemAttributes = m_attributes.createAttributeSet(SYSTEM_ATTRIBUTES);
        }

         ((AttributeSet)systemAttributes).setAttributeObjectNoHistory(SUBCLASSING_DELTA, delta);
         m_subclassingDelta = null;
     }
     catch (Exception e)
     {
         throw new Error(e.toString()); // Should never happen
     }
  }

  @Override
public boolean isTemplate()
  {
     IAttributeSet systemAttributes = (IAttributeSet)getAttributes().getAttribute(SYSTEM_ATTRIBUTES);
     if (systemAttributes != null)
     {
         Boolean isTemplate =  (Boolean)systemAttributes.getAttribute(IS_TEMPLATE);
         if (isTemplate == null)
        {
            return false;
        }
        else
        {
            return isTemplate.booleanValue();
        }
     }
    else
    {
        return false;
    }
  }

  @Override
public IDeltaElement getSubclassingDelta()
  {
     // Already cached
     if (m_subclassingDelta != null)
    {
        return m_subclassingDelta;
    }

     IAttributeSet systemAttributes = (IAttributeSet)getAttributes().getAttribute(SYSTEM_ATTRIBUTES);
     if (systemAttributes != null)
     {
         byte[] deltaBytes =  (byte[])systemAttributes.getAttribute(SUBCLASSING_DELTA);
         if (deltaBytes == null)
        {
            return null;
        }
        else
         {
             m_subclassingDelta = DeltaElement.fromBytes(deltaBytes);
             return m_subclassingDelta;
         }
     }
    else
    {
        return null;
    }
  }

  // Add a areference from this (subclassed) element to its super
  public void addSuperAttribute(String superClassName)
  {
      addSuperAttribute(superClassName, true);
  }

  public void addSuperAttribute(String superClassName, boolean addIsSubclassedTag)
  {
     try
     {
         IAttributeSet systemAttributes = (IAttributeSet)m_attributes.getAttribute(SYSTEM_ATTRIBUTES);
         if (systemAttributes == null)
        {
            systemAttributes = m_attributes.createAttributeSet(SYSTEM_ATTRIBUTES);
        }

         if (addIsSubclassedTag)
        {
            systemAttributes.setBooleanAttribute(IS_SUBCLASSED_ELEMENT, Boolean.TRUE);
        }

         systemAttributes.setStringAttribute(SUPER_REFERENCE, superClassName);

     }
     catch (Exception e)
     {
         throw new Error(e.toString()); // Should never happen
     }

  }

  private IBasicElement doneUpdate(boolean forSubclassing) throws ReadOnlyException
  {
     if (doneUpdateCalled)
    {
        return null;
    }

     if (m_readOnly)
    {
        throw new ReadOnlyException();
    }

      IBasicElement retElement = null;
      if (isNew())
      {
          if (forSubclassing)
        {
            throw new Error("Cannot call doneUpdateForSubclassing() on a new element.");
        }
          retElement = this;
      }
    else
    {
        retElement = (IBasicElement)createDelta(forSubclassing);
    }

      DeltaElement subclassingDelta = (DeltaElement)getSubclassingDelta();

      if (!forSubclassing && subclassingDelta != null && retElement instanceof DeltaElement)
      {
          subclassingDelta.adjustToSubclassedModification((DeltaElement)retElement);
          setSubclassingDelta(subclassingDelta.toBytes());
      }

      setReadOnly(true);
      doneUpdateCalled = true;

      return retElement;
  }



  @Override
void markModified(String attributeName, boolean oldValueExistsNotused, boolean deletionnotUsed)
  {
       if (!attributeName.equals(ELEMENT_ATTRIBUTES))
    {
        throw new Error();
    }

       markTreeModified();
  }

  // No resources to delete
  @Override
void delete(){}

  // The Element's m_attributes object should not call this
  @Override
Object getNameFromParent(Object listItem)
  {
      throw new Error();
  }

  @Override
public void setReadOnly(boolean readOnly)
  {
      m_readOnly = readOnly;

      // m_attributes can be null if the attributes are stored in m_cache (and in that case it's already marked read-only)
      if (m_attributes != null)
    {
        m_attributes.setReadOnly(readOnly);
    }
  }

  @Override
public void applyDelta(IDelta delta) throws AttributeSetTypeException
  { throw new Error ("This is not needed in Element and will not be implemented - use doApplyDelta");}

  @Override
public void removeDeltaHistory()
  {
      m_attributes.removeDeltaHistory();
      m_isNew = true;
  }

  // This should be called if the delta is not delta.isDeleted()
  // This should not be called if the element name and creation timestamp are
  // not identical between the delta and this object.
  public void doApplyDelta(IDeltaElement deltaElement) throws VersionMisMatchException
  {
      if (deltaElement.isDeleted())
    {
        throw new Error("Tried to apply a deleted delta.");
    }

      long thisVersion = m_identity.getVersion();
      long deltaVersion = deltaElement.getIdentity().getVersion();

      if (deltaVersion != thisVersion)
    {
        throw new VersionMisMatchException(deltaVersion < thisVersion);
    }

      Object newAttributes = deltaElement.getDeltaAttributes();
      if (newAttributes instanceof AttributeSet)
      {
          m_attributes = (AttributeSet)newAttributes;
          cacheAttributes();
      }
    else
    {
        try
          {
              m_attributes = (AttributeSet)getAttributes();

              // If the attributes were not in the cache and the DS was accessed then we might
              // already have the updated version of the element and we don't have to apply the delta
              if (m_identity.getVersion() == deltaVersion)
            {
                m_attributes.applyDelta((DeltaAttributeSet)newAttributes);
            }

              // Put attributes back in cache
              cacheAttributes();
          }
          catch (AttributeSetTypeException e)
          {
              throw new Error("Could not apply the delta attribute set: " + e.toString());
          }
    }

      // Increment the version number
      m_identity.setVersion(thisVersion + 1);

  }

  public void revertToTemplate(AttributeName[] attributes, String subName)
  {
      DeltaElement subClassingDelta = DeltaElement.fromBytes(getSubclassedDelta(subName));
      ((DeltaAttributeSet)subClassingDelta.getDeltaAttributes()).revertToTemplate(attributes);
      ElementIdentity id = (ElementIdentity)subClassingDelta.getIdentity();
      id.setVersion(id.getVersion() + 1);
      addSubclassedElement(subName, subClassingDelta.toBytes());
  }

  // This will be called from doneUpdate() only if it is not a new element.
  // If it's new,  doneUpdate() simply returns this element.
  // m_atributes should never be null (in cache) since createDelta() can be called only on non read-only
  // elements and only read-only elements can be cached
  @Override
public IDelta createDelta(boolean forSubclassing)
  {
      if (isNew())
    {
        throw new Error();
    }
      boolean isSubclassedElement = getSuperElementName() == null ? false : true;

      Object attributesDelta = null;
      if (m_attributes.isNew())
    {
        attributesDelta = m_attributes;
    }
    else
    {
        attributesDelta = m_attributes.createDelta(forSubclassing || isSubclassedElement);
    }

      return new DeltaElement(m_identity, attributesDelta);
  }

  @Override
public IElementIdentity getIdentity()
  {
      return m_identity;
  }

  @Override
public IAttributeSet getAttributes()
  {
      AttributeSet attributes = null;
      if (m_attributes != null || m_cache == null)
    {
        attributes = m_attributes;
    }
    else
    {
        attributes = (AttributeSet)m_cache.getAttributes(this);
    }

      return (IAttributeSet)attributes;
  }

  @Override
public IElement createWritableClone()
  {
      Element element = (Element)deepClone();
      element.setReadOnly(false);
      return element;
  }

  public Element createHeaderClone()
  {
      return new Element(m_identity.createClone(), m_deleted);
  }

  public IElement createClone()
  {
      Element element = (Element)deepClone();
      return element;
  }

  public IElement createWritableClone(String elementName, String typeName, String releaseVersion)
  {
      throw new Error("Not Implemented");
  }


  @Override
public String toString()
  {
      return "Element " + m_identity.toString();
  }

  @Override
public Object getAttribute(AttributeName attributeName)
  {
       if (isDeleted())
    {
        throw new Error("The element is deleted");
    }

       if (attributeName.getComponentCount() == 0)
    {
        return getAttributes();
    }
    else
    {
        return getAttributes().getAttribute(attributeName);
    }
  }

  @Override
boolean checkSubclassingDelta(AttributeName name)
  {
      DeltaElement delta = (DeltaElement)getSubclassingDelta();
      if (delta == null)
    {
        return false;
    }
    else
    {
        return delta.inSuperElement(name);
    }
  }

  @Override
String[] checkSubclassingDeletedAttributes(AttributeName name)
  {
      if (isDirty())
    {
        throw new Error(m_name + " is modified. Cannot call getDeletedAttributesInThisSubclassed in a modified element.");
    }

      DeltaElement delta = (DeltaElement)getSubclassingDelta();
      if (delta == null)
    {
        return new String[0];
    }
    else
    {
        return delta.getDeletedAttributes(name);
    }
  }



  // Sort the elements by parent directory. The returned HashMap contains the pairs {parentName,ArrayList} where
  // all the elements in the list are under tehe parentName directory.
  public static HashMap groupByParentDir (IDirElement[] elements)
  {
      HashMap groups = new HashMap();
      for (int i = 0; i < elements.length; i++)
      {
           EntityName eName = getEntity(elements[i].getIdentity().getName());
           String parentName = eName.getParent();

            ArrayList group = (ArrayList)groups.get(parentName);
            if (group == null)
            {
                group = new ArrayList();
                groups.put(parentName, group);
            }
            group.add(elements[i]);
      }

      return groups;

  }

  private static EntityName getEntity(String name)
  {
       EntityName entity = null;
       try{ entity = new EntityName(name); }
       catch(Exception e) { throw new  IllegalArgumentException(name + " is illegal.");}
       return entity;
  }

   // Replaces all references using the IReplaceRef replace service
   @Override
public boolean replaceReferences(boolean isSystemAtts, IReplaceRef replaceSrvc)
   {
       replaceSubclassingDelta(replaceSrvc);
       replaceSubclassingReferences(replaceSrvc);
       return m_attributes.replaceReferences(false, replaceSrvc);
   }


}
