/*
 * Copyright (c) 2001 Sonic Software. All Rights Reserved.
 */

package com.sonicsw.mf.common.config.impl;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;

import com.sonicsw.mf.common.config.query.AttributeName;


abstract class ElementNode implements IDeltaBookKeeper, java.io.Serializable
{
    private static final long serialVersionUID = 0L;
    private ElementNode m_parent;
    protected String m_name;
    protected boolean m_readOnly;

    protected transient boolean m_isNew;
    protected transient boolean m_isModified;

    // Constructs an empty new elementNode
    ElementNode(String name, ElementNode parent)
    {
        this(name, parent, true);
    }

    // Constructs an empty new or old elementNode
    ElementNode(String name, ElementNode parent, boolean isNew)
    {
        m_parent = parent;
        m_readOnly = false;
        m_isModified = false;
        m_isNew = isNew;
        m_name = name;
    }

    ElementNode deepClone()
    {
      try {
        ByteArrayOutputStream out = new ByteArrayOutputStream();
        ObjectOutputStream objectOut = new ObjectOutputStream(out);
        objectOut.writeObject(this);
        byte[] bytes = out.toByteArray();
        objectOut.close();

        ByteArrayInputStream in = new ByteArrayInputStream(bytes);
        ObjectInputStream objectIn = new ObjectInputStream(in);
        return (ElementNode) objectIn.readObject();
      }
      catch (Exception e) {
        throw new Error("deepClone failed.");
      }
    }

    protected Object cloneWithoutParent()
    {
       ElementNode parent = m_parent;
       // make a clone of the actualValue without the parent reference
       m_parent = null;
       ElementNode cloned = deepClone();
       // put the parent back in the original ElementNode
       m_parent = parent;
       return cloned;
    }


    private void writeObject(java.io.ObjectOutputStream s) throws java.io.IOException
    {
         s.writeObject(m_parent);
         s.writeObject(m_name);
         s.writeBoolean(m_readOnly);
    }
    private void readObject(java.io.ObjectInputStream s) throws java.io.IOException, ClassNotFoundException
    {
         m_parent = (ElementNode)s.readObject();
         m_name = (String)s.readObject();
         m_readOnly = s.readBoolean();

         m_isNew = false;
         m_isModified = false;
    }

    String getName()
    {
        return getTypedName().toString();
    }

    Object getTypedName()
    {
        if (m_name != null)
        {
            return m_name;
        }
        else if (m_parent != null)
        {
            return m_parent.getNameFromParent(this);
        }
        else {
            return "?"; // The name is unknown. This will only happen in a deleted node for error message  reporting.
        }
    }

    // The delta is stored at the root (an Element object) - we have to go up the tree to get the answer
    boolean checkSubclassingDelta(AttributeName name)
    {
        if (m_parent != null)
        {
            return m_parent.checkSubclassingDelta(name);
        }
        else
        {
            return false;
        }
    }

    String[] checkSubclassingDeletedAttributes(AttributeName name)
    {
        if (m_parent != null)
        {
            return m_parent.checkSubclassingDeletedAttributes(name);
        }
        else
        {
            return new String[0];
        }
    }


    // Return true if this node has a modified attribute list ancestor
    boolean underModifiedList()
    {
        if (m_parent != null)
        {
            return m_parent.underModifiedList();
        }
        else
        {
            return false;
        }
    }


    String getFullName()
    {
        String fullParentName = "";
        if (m_parent != null)
        {
            fullParentName = m_parent.getFullName();
        }
        return fullParentName + AttributeSet.ATTRIBUTE_SEPARATOR + getName();
    }

    public AttributeName getCompoundName()
    {
        if (m_parent == null)
        {
            return  new AttributeName(((Element)this).getIdentity().getName(), true);
        }

        AttributeName parentName = m_parent.getCompoundName();

        Object myName = getTypedName();
        String myNameS = (myName == null) ? null : myName.toString();
        if (myNameS != null && myNameS.equals(AttributeSet.ATTRIBUTE_SEPARATOR))
        {
            return parentName;
        }
        else if (myName instanceof Integer)
        {
            return parentName.setNextComponent(((Integer)myName).intValue());
        }
        else if (myName instanceof String)
        {
            return parentName.setNextComponent(myNameS);
        }
        else
        {
            throw new Error();
        }
    }



    void setNewParent(ElementNode newParent)
    {
        m_parent = newParent;
    }

    void removeFromTree()
    {
        delete();
        setReadOnly(true);
        m_parent = null;
    }

    void markTreeModified()
    {
        if (m_isNew)
        {
            return;
        }

        // Already marked
        if (m_isModified)
        {
            return;
        }

        m_isModified = true;

        if (m_parent != null)
        {
            m_parent.markModified(getName(), true, false);
        }
    }

    @Override
    public boolean isNew()
    {
       return m_isNew;
    }

    // Release any resources this node holds
    abstract void delete();

    // Items in a list don't have a stable name since their position in the list changes, so they must
    // get their name from the parent (AttributeList object) which looks up the position in the list
    abstract Object getNameFromParent(Object me);

    // Mark the attributeName in this node as modified
    abstract void markModified(String attributeName, boolean oldValueExists, boolean deletion);

    // Make this node and the children read-only
    abstract void setReadOnly(boolean readOnly);

}
