/*
 * Copyright (c) 2001 Sonic Software. All Rights Reserved.
 */

package com.sonicsw.mf.common.config.impl;

import java.util.ArrayList;
import java.util.StringTokenizer;

import com.sonicsw.mf.common.config.ConfigException;
import com.sonicsw.mf.common.config.IMFDirectories;


public final class EntityName
{
    private static final String MF_DIR_SEPARATOR_STRING = "" + IMFDirectories.MF_DIR_SEPARATOR;

    private final String m_entityName;
    private String m_parentName = null;
    private final String[] m_nameComponents;
    
    public EntityName(String[] components, int count)
    {
        m_nameComponents = new String[count];
        for (int i = 0; i < count; i++)
        {
            m_nameComponents[i] = components[i];
        }

        m_entityName = componentsToName(m_nameComponents);
    }

    public EntityName(String nameString) throws ConfigException
    {
        if (nameString == null || nameString.length() == 0)
        {
            throw new ConfigException("The name is invalid - cannot be null or empty.");
        }
        
        if (!nameString.startsWith(MF_DIR_SEPARATOR_STRING))
        {
            throw new ConfigException(nameString + " is invalid - it must start with '" + MF_DIR_SEPARATOR_STRING + "'");
        }

        StringTokenizer nameTokenizer = new StringTokenizer(nameString, MF_DIR_SEPARATOR_STRING);
        ArrayList<String> nameComponents = new ArrayList<String>();
        while (nameTokenizer.hasMoreTokens())
        {
            String nameComponent = nameTokenizer.nextToken();
            if (nameComponent.length() == 0)
            {
                break;
            }
            nameComponents.add(nameComponent);
        }
        
        m_nameComponents = nameComponents.toArray(new String[nameComponents.size()]);
        m_entityName = componentsToName(m_nameComponents);
    }

    public EntityName createChild(String child)
    {
        if (child.indexOf(IMFDirectories.MF_DIR_SEPARATOR) != -1)
        {
            throw new IllegalArgumentException("The child name includes the '/' character.");
        }

        String[] childComponents = new String[m_nameComponents.length + 1];
        System.arraycopy(m_nameComponents, 0, childComponents, 0, m_nameComponents.length);
        childComponents[childComponents.length - 1] = child;
        
        return new EntityName(childComponents, childComponents.length);
    }
    
    public boolean isRoot()
    {
        return m_nameComponents.length == 0;
    }

    // return normalized name
    public String getName()
    {
        return m_entityName;
    }

    public String createKidName(String kidBaseName)
    {
        if (m_nameComponents.length == 0)
        {
            return m_entityName + kidBaseName;
        }
        else
        {
            return m_entityName + IMFDirectories.MF_DIR_SEPARATOR + kidBaseName;
        }
    }

    // 0 length array is it's the root
    public String[] getNameComponents()
    {
        return (String[])m_nameComponents.clone();
    }

    public String[] getParentComponents()
    {
        // This is the root - no parent
        if (m_nameComponents.length == 0)
        {
            return null;
        }
        
        String[] parentComponents = new String[m_nameComponents.length - 1];
        if (parentComponents.length > 0)
        {
            System.arraycopy(m_nameComponents, 0, parentComponents, 0, parentComponents.length);
        }

        return parentComponents;
    }

    public EntityName getParentEntity()
    {
        if (m_nameComponents.length == 0)
        {
            throw new IllegalArgumentException("No parent.");
        }

        return new EntityName(m_nameComponents, m_nameComponents.length -1);
    }

    // Returns "" for the root
    public synchronized String getParent()
    {
        if (m_parentName != null)
        {
            return m_parentName;
        }
        
        // not previous evaluated
        switch (m_nameComponents.length)
        {
            case 0:
            {
                m_parentName = "";
                return m_parentName;
            }
            case 1:
            {
                m_parentName = MF_DIR_SEPARATOR_STRING;
                return m_parentName;
            }
            default:
            {
                m_parentName = "";
                for (int i = 0; i < m_nameComponents.length - 1; i++)
                {
                    m_parentName +=  IMFDirectories.MF_DIR_SEPARATOR + m_nameComponents[i];
                }
                return m_parentName;
            }
        }
    }

    // returns "" for the root
    public String getBaseName()
    {
        if (m_nameComponents.length == 0)
        {
            return "";
        }

        return m_nameComponents[m_nameComponents.length - 1];
    }

    public boolean isParent(EntityName child)
    {
        return parenthood(m_nameComponents, child.m_nameComponents) == 1;
    }

    private int parenthood(String[] components1, String[] components2)
    {
        int index1 = 0;

        while (index1 < components1.length && index1 < components2.length && components1[index1].equals(components2[index1]))
        {
            index1++;
        }

        // components1 is the father
        if (index1 == components1.length && index1 + 1 == components2.length)
        {
            return 1;
        }

        // components2 is the father
        if (index1 + 1 == components1.length && index1 == components2.length)
        {
            return 2;
        }

        // No parenthood
        return 0;
    }

    // Returns true if 'target' contain any of the 'chars' characters
    public static boolean containsAnyChar(String target, String chars)
    {
        for (int i = 0; i < target.length(); i++)
        {
            if ((chars.indexOf(target.charAt(i)) >= 0))
            {
                return true;
            }
        }
        return false;
    }

    private static String componentsToName(String[] components)
    {
        String name = "" + IMFDirectories.MF_DIR_SEPARATOR;
        for (int i = 0; i < components.length; i++)
        {
            name += ((i + 1 < components.length) ? components[i] + IMFDirectories.MF_DIR_SEPARATOR : components[i]);
        }

        return name;
    }
    
    @Override
    public String toString()
    {
        return getName();
    }
}

