/*
 * Copyright (c) 2001 Sonic Software. All Rights Reserved.
 */

package com.sonicsw.mf.common.config.impl;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;

import com.sonicsw.mf.common.config.IAttributeChangeHandler;
import com.sonicsw.mf.common.config.IAttributeSet;
import com.sonicsw.mf.common.config.IDeltaAttributeList;
import com.sonicsw.mf.common.config.IDeltaAttributeSet;
import com.sonicsw.mf.common.config.NotModifiedAttException;
import com.sonicsw.mf.common.config.query.AttributeName;

// Receives as input: An {attribute,handler} association tree and a delta. Call the handler of each modified
// Attribute and delete association tree nodes designated for deletion by the delta.
public final class ProcessDeltaNotifications
{
    private final static int MODIFIED = 1;

    private final static int DELETED = 2;

    private final static AttributeName ROOT_NAME = new AttributeName();

    public static void handleElementDeletion(AttributeAssociationTree tree, ArrayList list)
    {
        notifyChange(DELETED, null, (IAttributeChangeHandler)tree.getPayload(ROOT_NAME), list);
    }

    public static void processDelta(IDeltaAttributeSet delta, AttributeAssociationTree tree, ArrayList list)
    {
        notifyChange(MODIFIED, delta, (IAttributeChangeHandler)tree.getPayload(ROOT_NAME), list);

        // Notify modifed attributes
        String[] modifiedAttributes = delta.getModifiedAttributesNames();
        for (int i = 0; i < modifiedAttributes.length; i++)
        {
            Object newValue = null;
            try
            {
                newValue = delta.getNewValue(modifiedAttributes[i]);
            }
            catch (NotModifiedAttException e)
            {
                throw new IllegalArgumentException(e.toString());
            }

            Object subTree = tree.getSubtree(modifiedAttributes[i]);
            if (subTree == null)
            {
                continue;
            }
            else if (subTree instanceof AttributeAssociationTree)
            {
                if (newValue instanceof IDeltaAttributeSet)
                {
                    processDelta((IDeltaAttributeSet)newValue, (AttributeAssociationTree)subTree, list);
                }
                else if (newValue instanceof IDeltaAttributeList)
                {
                    processDelta((IDeltaAttributeList)newValue, (AttributeAssociationTree)subTree, list);
                }
                else
                {
                    // The sub tree is replaced notify the new value and remove any sub tree branches
                    IAttributeChangeHandler handler = (IAttributeChangeHandler)((AttributeAssociationTree)subTree).getPayload(ROOT_NAME);
                    notifyChange(MODIFIED, newValue, handler, list);
                    tree.removeNode(new AttributeName(modifiedAttributes[i]));
                    tree.insertPayload(new AttributeName(modifiedAttributes[i]), handler);
                }
            }
            else
            {
                // A handler
                notifyChange(MODIFIED, newValue, (IAttributeChangeHandler)subTree, list);
            }
        }

        // Notify deleted attributes
        String[] deletedAttributes = delta.getDeletedAttributesNames();
        for (int i = 0; i < deletedAttributes.length; i++)
        {
            Object subTree = tree.getSubtree(deletedAttributes[i]);
            if (subTree == null)
            {
                continue;
            }
            else if (subTree instanceof AttributeAssociationTree)
            {
                notifyChange(DELETED, null, (IAttributeChangeHandler)((AttributeAssociationTree)subTree).getPayload(ROOT_NAME), list);
            }
            else
            {
                // A handler
                notifyChange(DELETED, null, (IAttributeChangeHandler)subTree, list);
            }

            tree.removeNode(new AttributeName(deletedAttributes[i]));

        }

    }

    public static void processDelta(IDeltaAttributeList delta, AttributeAssociationTree tree, ArrayList list)
    {
        notifyChange(MODIFIED, delta, (IAttributeChangeHandler)tree.getPayload(ROOT_NAME), list);

        // Notify deleted items and then clear the tree since the ordinal number of listeners might have changed
        int[] deletedItems = delta.getDeletedItemNumbers();
        for (int i = 0; i < deletedItems.length; i++)
        {
            Object subTree = tree.getSubtree(new Integer(deletedItems[i]));
            if (subTree == null)
            {
                continue;
            }
            else if (subTree instanceof AttributeAssociationTree)
            {
                notifyChange(DELETED, null, (IAttributeChangeHandler)((AttributeAssociationTree)subTree).getPayload(ROOT_NAME), list);
            }
            else
            {
                // A handler
                notifyChange(DELETED, null, (IAttributeChangeHandler)subTree, list);
            }
        }
        if (deletedItems.length > 0)
         {
            tree.removeNode(ROOT_NAME); // Remove all the listeners from that tree
        }

        // Notify modified items
        HashMap modifiedItems = delta.getModifiedItems();
        Iterator iterator = modifiedItems.keySet().iterator();

        while (iterator.hasNext())
        {
            Integer modifiedNum = (Integer)iterator.next();
            Object newValue = modifiedItems.get(modifiedNum);

            Object subTree = tree.getSubtree(modifiedNum);
            if (subTree == null)
            {
                continue;
            }
            else if (subTree instanceof AttributeAssociationTree)
            {
                if (newValue instanceof IDeltaAttributeSet)
                {
                    processDelta((IDeltaAttributeSet)newValue, (AttributeAssociationTree)subTree, list);
                }
                else if (newValue instanceof IDeltaAttributeList)
                {
                    processDelta((IDeltaAttributeList)newValue, (AttributeAssociationTree)subTree, list);
                }
                else
                {
                    // The sub tree is replaced notify the new value to the handler clear sub branches
                    IAttributeChangeHandler handler = (IAttributeChangeHandler)((AttributeAssociationTree)subTree).getPayload(ROOT_NAME);
                    notifyChange(MODIFIED, newValue, handler, list);
                    tree.removeNode(new AttributeName(modifiedNum.intValue()));
                    tree.insertPayload(new AttributeName(modifiedNum.intValue()), handler);

                }
            }
            else
            {
                // A handler
                notifyChange(MODIFIED, newValue, (IAttributeChangeHandler)subTree, list);
            }
        }
    }

    // Note that value and handler can be null
    public static void notifyChange(int type, Object value, IAttributeChangeHandler handler, ArrayList list)
    {
        if (handler == null)
        {
            return;
        }

        list.add(new NotifyAction(type, value, handler));
    }

    // The whole attribute set is modified. Notify the new value if a handler exits and clear the tree
    public static void processDelta(IAttributeSet attributeSet, AttributeAssociationTree tree, ArrayList list)
    {
        notifyChange(MODIFIED, attributeSet, (IAttributeChangeHandler)tree.getPayload(ROOT_NAME), list);
        tree.removeNode(ROOT_NAME);
    }

    public static void doNotifications(ArrayList list)
    {
        for (int i = 0; i < list.size(); i++)
        {
            ((NotifyAction)list.get(i)).notifyChange();
        }
    }

    private static class NotifyAction
    {
        private int m_type;

        private Object m_value;

        private IAttributeChangeHandler m_handler;

        NotifyAction(int type, Object value, IAttributeChangeHandler handler)
        {
            m_type = type;
            m_value = value;
            m_handler = handler;
        }

        void notifyChange()
        {
            Thread currentThread = Thread.currentThread();
            ClassLoader originalContextClassLoader = currentThread.getContextClassLoader();
            currentThread.setContextClassLoader(m_handler.getClass().getClassLoader());

            try
            {
                if (m_type == DELETED)
                {
                    m_handler.itemDeleted();
                }
                else
                {
                    m_handler.itemModified(m_value);
                }
            }
            finally
            {
                currentThread.setContextClassLoader(originalContextClassLoader);
            }
        }
    }
}
