/*
 * Copyright (c) 2001 Sonic Software. All Rights Reserved.
 */

package com.sonicsw.mf.common.config.impl;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Objects;

import com.sonicsw.mf.common.config.AttributeSetTypeException;
import com.sonicsw.mf.common.config.IAttributeSetType;
import com.sonicsw.mf.common.config.ITypeCollection;
import com.sonicsw.mf.common.config.ReadOnlyException;

final class TypeCollection extends HashMap implements ITypeCollection, java.io.Serializable
{
    private static final long serialVersionUID = 0L;
    private final static int SERIALIZATION_VERSION = 3;

    private boolean m_readOnly;

    TypeCollection()
    {
        m_readOnly = false;
    }

    int estimateSize()
    {
        return Util.OBJECT_SIZE + Util.estimateHashMapSize(this);
    }

    @Override
    public boolean typesEqual(ITypeCollection c)
    {
        return equals((Object)c);
    }

    @Override
    public boolean equals(Object o)
    {
        if (o == null)
        {
            return false;
        }

        if (!(o instanceof TypeCollection))
        {
            return false;
        }

        TypeCollection typeCollection = (TypeCollection)o;


        // All attribute types in typeCollection must exist and be equal in this collection
        String[] typeCollectionNames =  typeCollection.getAllTypeNames();
        for (int i = 0; i < typeCollectionNames.length; i++)
        {
            AttributeSetType thisType = (AttributeSetType)getAttributeSetType(typeCollectionNames[i]);
            AttributeSetType thatType = (AttributeSetType)typeCollection.getAttributeSetType
                                            (typeCollectionNames[i]);
            if (!thatType.equals(thisType))
            {
                return false;
            }
        }

        // All attribute types in this collection  must exist and be equal in typeCollection
        String[] thisTypeCollectionNames =  getAllTypeNames();
        for (int i = 0; i < thisTypeCollectionNames.length; i++)
        {
            AttributeSetType thisType = (AttributeSetType)getAttributeSetType(thisTypeCollectionNames[i]);
            AttributeSetType thatType = (AttributeSetType)typeCollection.getAttributeSetType
                                            (thisTypeCollectionNames[i]);
            if (!thisType.equals(thatType))
            {
                return false;
            }
        }

        return true;
    }
    
    @Override
    public int hashCode() {
        return Objects.hash((Object[])getAllTypeNames());
    }

    private void writeObject(java.io.ObjectOutputStream s) throws java.io.IOException
    {
         s.writeInt(SERIALIZATION_VERSION);
         s.writeBoolean(m_readOnly);
    }
    private void readObject(java.io.ObjectInputStream s) throws java.io.IOException, ClassNotFoundException
    {
        int version = s.readInt();
        m_readOnly = s.readBoolean();
        if (version != SERIALIZATION_VERSION)
        {
            Util.throwSerialVersionMismatch(version, SERIALIZATION_VERSION);
        }
    }


    void setReadOnly(boolean readOnly)
    {
        m_readOnly = readOnly;
        Iterator iterator = values().iterator();
        while (iterator.hasNext())
        {
            ((AttributeSetType)iterator.next()).setReadOnly(readOnly);
        }
    }

    TypedAttSet cloneInstance(TypedAttSet set)
    {
         AttributeSetType type = (AttributeSetType)getAttributeSetType(set.getTypeName());
         return type.createTypedSet(set.values());
    }

    @Override
    public IAttributeSetType createAttributeSetType(String typeName) throws AttributeSetTypeException, ReadOnlyException
    {
        if (containsKey(typeName))
        {
            throw new AttributeSetTypeException(typeName + " already exists.");
        }

        if (m_readOnly)
        {
            throw new ReadOnlyException();
        }
        
        IAttributeSetType newType = new AttributeSetType(typeName);
        put(typeName, newType);
        return newType;
    }

    @Override
    public IAttributeSetType deleteAttributeSetType(String typeName) throws AttributeSetTypeException, ReadOnlyException
    {
        if (m_readOnly)
        {
            throw new ReadOnlyException();
        }

        IAttributeSetType typeObject = (IAttributeSetType)get(typeName);
        if (typeObject == null)
        {
            return null;
        }

        if (((AttributeSetType)typeObject).hasInstances())
        {
            throw new AttributeSetTypeException("There are AttributeSet instances of type '" + typeName + "' - cannot be deleted.");
        }

        remove(typeName);
 
        return typeObject;
    }

    @Override
    public IAttributeSetType getAttributeSetType(String typeName)
    {
        return (IAttributeSetType)get(typeName); 
    }

    @Override
    public String[] getAllTypeNames()
    {
        String[] result = new String[size()];
        Iterator iterator = keySet().iterator();
        int i = 0;
        while (iterator.hasNext())
        {
            result[i++] = (String)iterator.next();
        }
        return result;
    }

    void verifyType(AttributeSetType type, String typeContainer) throws AttributeSetTypeException
    {
        if (get(type.m_name) != type)
        {
            throw new AttributeSetTypeException("The IAttributeSetType object '" + get(type.m_name) +
                                                "' was not defined in " + typeContainer );
        }
    }

}
