/*
 * Copyright (c) 2001 Sonic Software. All Rights Reserved.
 */

package com.sonicsw.mf.common.config.impl;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;

import com.sonicsw.mf.common.config.Reference;


public final class Util
{
    // Upper limit estimates
    static final int OBJECT_SIZE = 16;
    static final int ARRAY_SIZE = 16;
    static final int BIGDECIMAL_SIZE_ESTIMATE = 120;
    static final int BOOLEAN_SIZE = 16;
    static final int LONG_SIZE = 16;
    static final int INTEGER_SIZE = 16;
    static final int DATE_SIZE = 24;
    static final int ELEMENT_NODE_SIZE = 64;


   static void throwSerialVersionMismatch(int objectVersion, int classVersion) throws java.io.IOException
   {
       throw new java.io.IOException("Serialization version mismatch. Serialized object version: " + objectVersion +
                                     " class version: " + classVersion);
   }

    static int estimateHashMapSize(HashMap map)
    {
        if (map == null)
        {
            return 0;
        }

        int estimate = ARRAY_SIZE;
        estimate += map.size() * 2; // Account for empty elements the HashMap contain to ensure capacity;

        int count = map.size();
        Iterator iterator = map.keySet().iterator();
        for (int i = 0; i < count; i++)
        {
            Object key = iterator.next();
            estimate += estimateSize(key) + estimateSize(map.get(key));
        }

        return estimate;
    }

   static int estimateArrayListSize(ArrayList list)
   {
        if (list  == null)
        {
            return 0;
        }

      int estimate = ARRAY_SIZE;
      estimate += list.size() * 2; // Account for empty elements the ArrayList contain to ensure capacity;
      for (int i = 0; i < list.size(); i++)
    {
        estimate += estimateSize(list.get(i));
    }
      return estimate;
   }



   public static boolean validSimpleValue(Object o)
   {
       if (o == null)
    {
        return true;
    }

       String type = o.getClass().getName();

       if (type.equals("java.lang.String"))
    {
        return true;
    }
       if (type.equals("java.lang.Integer"))
    {
        return  true;
    }
       if (type.equals("java.lang.Boolean"))
    {
        return  true;
    }
       if (type.equals("com.sonicsw.mf.common.config.Reference"))
    {
        return true;
    }
       if (type.equals("[B"))
    {
        return true;
    }
       if (type.equals("java.lang.Long"))
    {
        return true;
    }
       if (type.equals("java.math.BigDecimal"))
    {
        return true;
    }
       if (type.equals("java.util.Date"))
    {
        return true;
    }

       return false;
   }

   // Estimate an upper limit on the object's size
   static int estimateSize(Object o)
   {
       if (o == null)
    {
        return 0;
    }

       if (o instanceof String)
    {
        return ARRAY_SIZE + (((String)o).length() << 1);
    }
       if (o instanceof Integer)
    {
        return INTEGER_SIZE;
    }
       if (o instanceof Long)
    {
        return LONG_SIZE;
    }
       if (o instanceof BigDecimal)
    {
        return BIGDECIMAL_SIZE_ESTIMATE;
    }
       if (o instanceof Date)
    {
        return DATE_SIZE;
    }
       if (o instanceof Boolean)
    {
        return BOOLEAN_SIZE;
    }
       if (o instanceof byte[])
    {
        return ARRAY_SIZE + ((byte[])o).length;
    }

       if (o instanceof AttributeSet)
    {
        return ((AttributeSet)o).estimateSize();
    }
       if (o instanceof AttributeList)
    {
        return ( (AttributeList) o).estimateSize();
    }
       if (o instanceof AttributeSetType)
    {
        return ((AttributeSetType)o).estimateSize();
    }
       if (o instanceof DeltaAttributeSet)
    {
        return ((DeltaAttributeSet)o).estimateSize();
    }
       if (o instanceof DeltaAttributeList)
    {
        return ((DeltaAttributeList)o).estimateSize();
    }
       if (o instanceof Reference)
    {
        return OBJECT_SIZE + Util.estimateSize(((com.sonicsw.mf.common.config.Reference)o).getElementName());
    }

       throw new Error (" Unknown type " + o.getClass().getName());
   }

   // Creates the delta between two values. Returns null if the two values are equal.
   static Object createDelta (Object v1, Object v1Type, Object v2, Object v2Type)
   {
       String type1 = v1.getClass().getName();
       String type2 = v2.getClass().getName();

       if (!type1.equals(type2))
    {
        return v2;
    }

       if (type1.equals("java.lang.String"))
       {
           if (v1.equals(v2))
        {
            return null;
        }
        else
        {
            return v2;
        }
       }

       if (type1.equals("com.sonicsw.mf.common.config.impl.AttributeSet"))
    {
        return ((AttributeSet)v1).createDelta((AttributeSetType)v1Type,
                                                 (AttributeSet)v2,
                                                  (AttributeSetType)v2Type);
    }

       if (type1.equals("java.lang.Integer"))
    {
        if (((Integer)v1).equals(v2))
        {
            return null;
        }
        else
        {
            return v2;
        }
    }

       if (type1.equals("java.lang.Boolean"))
    {
        if (((Boolean)v1).equals(v2))
        {
            return null;
        }
        else
        {
            return v2;
        }
    }

       if (type1.equals("[B"))
       {
           byte[] b1 = (byte[])v1;
           byte[] b2 = (byte[])v2;

           if (b1.length == b2.length)
           {
               for (int i = 0; i < b1.length; i++)
               {
                   if (b1[i] != b2[i])
                {
                    return v2;
                }
               }
               return null;
           }
        else
        {
            return v2;
        }
       }

       if (type1.equals("com.sonicsw.mf.common.config.impl.AttributeList"))
    {
        return ((com.sonicsw.mf.common.config.impl.AttributeList)v1).createDelta(
                   (com.sonicsw.mf.common.config.impl.AttributeList)v2);
    }

       if (type1.equals("java.lang.Long"))
    {
        if (((Long)v1).equals(v2))
        {
            return null;
        }
        else
        {
            return v2;
        }
    }

       if (type1.equals("java.math.BigDecimal"))
    {
        if (((java.math.BigDecimal)v1).equals(v2))
        {
            return null;
        }
        else
        {
            return v2;
        }
    }

       if (type1.equals("com.sonicsw.mf.common.config.Reference"))
       {
           String name1 = ((com.sonicsw.mf.common.config.Reference)v1).getElementName();
           String name2 = ((com.sonicsw.mf.common.config.Reference)v2).getElementName();

           if (name1.equals(name2))
        {
            return null;
        }
        else
        {
            return v2;
        }
       }

       if (type1.equals("java.util.Date"))
    {
        if (((java.util.Date)v1).equals(v2))
        {
            return null;
        }
        else
        {
            return v2;
        }
    }

       throw new Error (" Unknown type " + type1);

   }

   // Returns true if both objects are atomic values and equal
   public static boolean startsWith(boolean ignoreCase, Object v1, Object v2)
   {
       if (v1 == null || v2 == null)
    {
        return false;
    }

       if (!(v1 instanceof java.lang.String) || !(v2 instanceof java.lang.String))
    {
        return false;
    }

       String source = (String)v1;
       String prefix = (String)v2;

       return source.regionMatches(ignoreCase, 0, prefix, 0, prefix.length());
   }



   // Returns true if both objects are atomic values and equal
   public static boolean atomicAndEqual(Object v1, Object v2)
   {
       String type1 = v1.getClass().getName();
       String type2 = v2.getClass().getName();

       if (!type1.equals(type2))
    {
        return false;
    }

       if (type1.equals("java.lang.String"))
       {
           if (v1.equals(v2))
        {
            return true;
        }
        else
        {
            return false;
        }
       }

       if (type1.equals("com.sonicsw.mf.common.config.impl.AttributeSet"))
    {
        return false;
    }

       if (type1.equals("java.lang.Integer"))
    {
        if (((Integer)v1).equals(v2))
        {
            return true;
        }
        else
        {
            return false;
        }
    }

       if (type1.equals("java.lang.Boolean"))
    {
        if (((Boolean)v1).equals(v2))
        {
            return true;
        }
        else
        {
            return false;
        }
    }

       if (type1.equals("[B"))
       {
           byte[] b1 = (byte[])v1;
           byte[] b2 = (byte[])v2;

           if (b1.length == b2.length)
           {
               for (int i = 0; i < b1.length; i++)
               {
                   if (b1[i] != b2[i])
                {
                    return false;
                }
               }
               return true;
           }
        else
        {
            return false;
        }
       }

       if (type1.equals("com.sonicsw.mf.common.config.impl.AttributeList"))
    {
        return false;
    }

       if (type1.equals("java.lang.Long"))
    {
        if (((Long)v1).equals(v2))
        {
            return true;
        }
        else
        {
            return false;
        }
    }

       if (type1.equals("java.math.BigDecimal"))
    {
        if (((java.math.BigDecimal)v1).equals(v2))
        {
            return true;
        }
        else
        {
            return false;
        }
    }

       if (type1.equals("com.sonicsw.mf.common.config.Reference"))
       {
           String name1 = ((com.sonicsw.mf.common.config.Reference)v1).getElementName();
           String name2 = ((com.sonicsw.mf.common.config.Reference)v2).getElementName();

           if (name1.equals(name2))
        {
            return true;
        }
        else
        {
            return false;
        }
       }

       if (type1.equals("java.util.Date"))
    {
        if (((java.util.Date)v1).equals(v2))
        {
            return true;
        }
        else
        {
            return false;
        }
    }

       throw new Error (" Unknown type " + type1);

   }

}
