/*
 * Copyright (c) 2001 Sonic Software. All Rights Reserved.
 */
package com.sonicsw.mf.common.config.query;

import java.util.HashMap;

import com.sonicsw.mf.common.config.query.impl.Util;

/**
 * Represents 'AttributeName is from template' expression. If the attribute is from a template and the boolean parameter
 * is 'true' then the expression evaluation is 'true' (and visa versa).
 */
public final class AttributeFromTemplateExpression extends BooleanExpression implements java.io.Serializable, IDualOperandExpression
{
    private static final long serialVersionUID = 0L;
    private final static int SERIALIZATION_VERSION = 3;

    private Object m_op1;
    private Object m_op2;

   /** Is the attribute from template?
    *
    * @param name the name of the attribute (could be nested). Null and empty (the root level attribute set) is invalid.
    * @param isFromTemplate if 'false' then the expression evaluation is 'true' if the attribute is NOT from a template
    *
    */
    public AttributeFromTemplateExpression(AttributeName name, boolean isFromTemplate)
    {
        this (name, new Boolean(isFromTemplate));
    }

    private AttributeFromTemplateExpression(AttributeName name, Boolean isFromTemplate)
    {
        if (name == null)
        {
            throw new NullPointerException();
        }

        if (name.getComponentCount() == 0)
        {
            throw new IllegalArgumentException("The top level attribute set cannot be specified.");
        }

        m_op1 = name;
        m_op2 = isFromTemplate;
    }

   /**
    * Returns the AttributeName object of this expression
    *
    * @return the AttributeName object of this expression
    *
    */
    @Override
    public Object getFirstOperand()
    {
        return m_op1;
    }

   /**
    * Returns the value object of this expression
    *
    * @return the value object of this expression
    *
    */
    @Override
    public Object getSecondOperand()
    {
        return m_op2;
    }

   /**
    * returns the string representation of this expression
    *
    * @return the string representation of this expression
    */
    @Override
    public String toString()
    {
        return m_op1 + " = " + m_op2;
    }

    private void writeObject(java.io.ObjectOutputStream s) throws java.io.IOException
    {
         s.writeInt(SERIALIZATION_VERSION);
         HashMap table = new HashMap(2,1);
         table.put("op1", m_op1);
         table.put("op2", m_op2);
         s.writeObject(table);
    }

    private void readObject(java.io.ObjectInputStream s) throws java.io.IOException, ClassNotFoundException
    {
         int version = s.readInt();
         if (version != SERIALIZATION_VERSION)
        {
            Util.throwSerialVersionMismatch(version, SERIALIZATION_VERSION);
        }

         HashMap table = (HashMap)s.readObject();
         m_op1 = (Object)table.get("op1");
         m_op2 = (Object)table.get("op2");
    }
}
