/*
 * Copyright (c) 2001 Sonic Software. All Rights Reserved.
 */

package com.sonicsw.mf.common.config.query;
import java.util.ArrayList;
import java.util.HashMap;

import com.sonicsw.mf.common.config.impl.AttributeSet;
import com.sonicsw.mf.common.config.query.impl.Util;

/**
 * Represents a name (possibly nested) of an attribute in a configuration element object.
 * At each level we can either have a name of an attribute or an integer that represents
 * the position of an item in an attribute list.
 */
public final class AttributeName implements java.io.Serializable
{
    private static final long serialVersionUID = 0L;
    private static final String ELEMENT_NAME_ATT_SEPARATOR =
                                AttributeSet.ATTRIBUTE_SEPARATOR + AttributeSet.ATTRIBUTE_SEPARATOR + AttributeSet.ATTRIBUTE_SEPARATOR;
    private final static int SERIALIZATION_VERSION = 4;
    private ArrayList m_components;
    private String m_elementName;

    @Override
    public boolean equals(Object obj)
    {
        if (!(obj instanceof AttributeName))
        {
            return false;
        }
        AttributeName other = (AttributeName) obj;

        if (!other.m_elementName.equals(m_elementName))
        {
            return false;
        }

        return other.m_components.equals(m_components);

    }

    @Override
    public int hashCode()
    {
        return m_elementName.hashCode() * 31 + m_components.hashCode();
    }

   /**
    * Constructs a one level attribute name in an attribute set.
    *
    * @param name the attribute name (null is illegal)
    *
    */
    public AttributeName(String name)
    {
        this();

        if (name == null)
        {
            throw new NullPointerException();
        }

        m_components.add(name);
    }
    private AttributeName(String elementName, ArrayList components)
    {
        m_components = components;
        m_elementName = elementName;
    }

    public AttributeName createClone()
    {
        return new AttributeName(m_elementName, (ArrayList)m_components.clone());
    }

   /**
    * Constructs a one level attribute name in an attribute list.
    *
    * @param name the attribute name (must be >= 0)
    *
    */
    public AttributeName(int position)
    {
        this();

        if (position < 0)
        {
            throw new IllegalArgumentException();
        }

        m_components.add(new Integer(position));
    }

   /**
    * Constructs an empty attribute name with a specific element name
    *
    * @param elementName the name of the element
    * @param hasElementName should be always 'true'. Use the 'AttributeName()' constructor if there is not element name.
    */
    public AttributeName(String elementName, boolean hasElementName)
    {
        this();
        if (hasElementName)
        {
            m_elementName = elementName;
        }
    }


   /**
    * Constructs an empty attribute name
    *
    */
    public AttributeName()
    {
        m_elementName = null;
        m_components = new ArrayList();
    }


   /**
    * Adds a component to this AttributeName
    *
    * @param component the added component (null is illegal)
    *
    * @return this AttributeName object
    */
    public AttributeName setNextComponent(String component)
    {
       if (component == null)
    {
        throw new NullPointerException();
    }

        m_components.add(component);
        return this;
    }

   /**
    * Adds an attribute list componenet to this AttributeName.
    *
    * @param positions the positions into the attribute list object (must be greater or equal to 0)
    *
    * @return this AttributeName object
    */
    public AttributeName setNextComponent(int position)
    {
       if (position < 0)
    {
        throw new IllegalArgumentException();
    }

        m_components.add(new Integer(position));
        return this;
    }

   /**
    * returns the number of components in this AttributeName.
    *
    * @return the number of components in this AttributeName
    */
    public int getComponentCount()
    {
        return m_components.size();
    }


   /**
    * returns a single component of this AttributeName.
    *
    * @param level the component level (0 is the first level)
    *
    * @return a String object (attribute name) or an Integer (the position in an attribute list).
    */
    public Object getComponent(int level)
    {
        return m_components.get(level);
    }

   /**
    * returns the string representation of this AttributeName
    *
    * @return the string representation of this AttributeName
    */
    @Override
    public String toString()
    {
        String result = (m_elementName == null) ? "" : m_elementName + ELEMENT_NAME_ATT_SEPARATOR;

        for (int i = 0; i < m_components.size(); i++)
        {
            result += m_components.get(i).toString();
            if (i + 1 < m_components.size())
            {
                result += AttributeSet.ATTRIBUTE_SEPARATOR;
            }
        }
        return result;
    }

  /**
    * returns the name of the element - can be null if the attribute is not in any specific element
    *
    * @return the name of the element (can be null)
    */
    public String getElementName()
    {
        return m_elementName;
    }

    public void setElementName(String elementName)
    {
        m_elementName = elementName;
    }

    private void writeObject(java.io.ObjectOutputStream s) throws java.io.IOException
    {
         s.writeInt(SERIALIZATION_VERSION);
         HashMap table = new HashMap(2,1);
         table.put("comp", m_components);
         table.put("element", m_elementName);
         s.writeObject(table);
    }

    private void readObject(java.io.ObjectInputStream s) throws java.io.IOException, ClassNotFoundException
    {
         int version = s.readInt();
         if (version != SERIALIZATION_VERSION)
        {
            Util.throwSerialVersionMismatch(version, SERIALIZATION_VERSION);
        }

         HashMap table = (HashMap)s.readObject();
         m_components = (ArrayList)table.get("comp");
         m_elementName = (String)table.get("element");
    }


}
