/*
 * Copyright (c) 2001 Sonic Software. All Rights Reserved.
 */

package com.sonicsw.mf.common.config.query;
import java.util.HashMap;

import com.sonicsw.mf.common.config.query.impl.Util;

/**
 * Represents a DS query with a SELECT clause, a FROM clause and a WHERE clause.
 */
public final class Query implements java.io.Serializable
{
    private static final long serialVersionUID = 0L;
    private final static int SERIALIZATION_VERSION = 3;

    private Select m_select;
    private From m_from;
    private Where m_where;
    private OrderedBy m_ordered;

   /**
    * returns the string representation the query
    *
    * @return the string representation of the query
    */
    @Override
    public String toString()
    {
        String result = "";

        if (m_select != null)
        {
            result += "SELECT " + m_select;
        }

        if (m_from != null)
        {
            result += "\nFROM " + m_from;
        }

        if (m_where != null)
        {
            result += "\nWHERE " + m_where;
        }

        return result;
    }

   /**
    * Sets a Select object in this query. If the select object is empty then only the element's identity is fetched.
    * If a Select object is not set then the whole element is fetched.
    *
    * @param select the Select object
    *
    * @return this Query object
    */
    public Query setSelect(Select select)
    {
        m_select = select;
        return this;
    }

   /**
    * Sets a From object in this query. The query must contain a non null From object.
    *
    * @param from the From object (FromDirectory or FromELementList or FromElementType).
    *
    * @return this Query object
    */
    public Query setFrom(From from)
    {
        m_from = from;
        return this;
    }

   /**
    * Sets a Where object in this query. If no Where object is set or if the Where clause is empty 
    * then all the elements of the From clause are fetched.
    *
    * @param where the Where object
    *
    * @return this Query object
    */
    public Query setWhere(Where where)
    {
        m_where = where;
        return this;
    }

  /**
    * Gets the Select object of this query.
    *
    * @return the Select object
    */
    public Select getSelect()
    {
        return m_select;
    }

  /**
    * Gets the From object of this query.
    *
    * @return the From object
    */
    public From getFrom()
    {
        return m_from;
    }

  /**
    * Gets the Where object of this query.
    *
    * @return the Where object
    */
    public Where getWhere()
    {
        return m_where;
    }
    
    public void setOrderedBy(OrderedBy o)
    {
    	m_ordered = o;
    }
    
    public OrderedBy getOrderedBy()
    {
    	return m_ordered;
    }

    private void writeObject(java.io.ObjectOutputStream s) throws java.io.IOException
    {
         s.writeInt(SERIALIZATION_VERSION);
         HashMap table = new HashMap(4,1);
         table.put("s", m_select);
         table.put("f", m_from);
         table.put("w", m_where);
         table.put("o", m_ordered);
         s.writeObject(table);
    }

    private void readObject(java.io.ObjectInputStream s) throws java.io.IOException, ClassNotFoundException
    {
         int version = s.readInt();
         if (version != SERIALIZATION_VERSION)
        {
            Util.throwSerialVersionMismatch(version, SERIALIZATION_VERSION);
        }

         HashMap table = (HashMap)s.readObject();
         m_select = (Select)table.get("s");
         m_from = (From)table.get("f");
         m_where = (Where)table.get("w");
         m_ordered = (OrderedBy)table.get("o");
    }

}
