// Copyright (c) 2001,2002,2007 Progress Software Corporation. All Rights Reserved.

package com.sonicsw.mf.common.config.query.impl;
import java.util.Iterator;

import com.sonicsw.mf.common.config.IAttributeList;
import com.sonicsw.mf.common.config.IAttributeMetaData;
import com.sonicsw.mf.common.config.IAttributeSet;
import com.sonicsw.mf.common.config.IElement;
import com.sonicsw.mf.common.config.impl.Util;
import com.sonicsw.mf.common.config.query.AttributeFromTemplateExpression;
import com.sonicsw.mf.common.config.query.AttributeName;
import com.sonicsw.mf.common.config.query.BooleanExpression;
import com.sonicsw.mf.common.config.query.ContainsExpression;
import com.sonicsw.mf.common.config.query.EqualExpression;
import com.sonicsw.mf.common.config.query.ExistsExpression;
import com.sonicsw.mf.common.config.query.IDualOperandExpression;
import com.sonicsw.mf.common.config.query.ListContainsExpression;
import com.sonicsw.mf.common.config.query.ListItemStartsWithExpression;
import com.sonicsw.mf.common.config.query.NotEqualExpression;
import com.sonicsw.mf.common.config.query.StartsWithExpression;
import com.sonicsw.mf.common.config.query.Where;

// Evaluates an element against a where clause to determine if the client wants it
public final class ElementEvaluation
{
    public static boolean evaluate(IElement element, Where where)
    {
         BooleanExpression[] booleanExpressions = where.getAndChain();

         for (int i = 0; i < booleanExpressions.length; i++)
         {
             IDualOperandExpression expression = (IDualOperandExpression)booleanExpressions[i];
             AttributeName attName = (AttributeName)expression.getFirstOperand();
             Object expressionValue = expression.getSecondOperand();
             Object elementValue = element.getAttribute(attName);

             boolean eqOperation = expression instanceof EqualExpression || expression instanceof NotEqualExpression;
             if (expression instanceof StartsWithExpression && !Util.startsWith(true, elementValue, expressionValue))
            {
                return false;
            }
            else if (expression instanceof ExistsExpression && (elementValue == null))
            {
                return false;
            }
            else if (expression instanceof ListContainsExpression && !listContains(elementValue, expressionValue))
            {
                return false;
            }
            else if (expression instanceof ListItemStartsWithExpression && !listItemStartsWith(elementValue, expressionValue))
            {
                return false;
            }
            else if (eqOperation && !evaluateEquality(expression, elementValue, expressionValue))
            {
                return false;
            }
            else if (expression instanceof AttributeFromTemplateExpression &&
                      !fromTemplate(element, attName, ((Boolean)expressionValue).booleanValue()))
            {
                return false;
            }
            else if (expression instanceof ContainsExpression && ((String)elementValue).indexOf((String)expressionValue) == -1)
            {
                return false;
            }

         }

         return true;
    }

    private static boolean fromTemplate(IElement element, AttributeName attName, boolean isFromTemplate)
    {
        // Construct the parent AttributeName of attName
        AttributeName parentName = new AttributeName();
        int numComponents = attName.getComponentCount();
        for (int i = 0; i < numComponents - 1; i++)
        {
            Object component = attName.getComponent(i);
            if (component instanceof String)
            {
                parentName.setNextComponent((String)component);
            }
            else
            {
                parentName.setNextComponent(((Integer)component).intValue());
            }
        }
        Object baseName = attName.getComponent(numComponents - 1);
        Object parent = element.getAttribute(parentName);

        IAttributeMetaData metaData = null;
        try
        {
            if (parent instanceof IAttributeSet)
            {
                metaData = ((IAttributeSet)parent).getAttributeMetaData((String)baseName);
            }
            else if  (parent instanceof IAttributeList)
            {
                metaData = ((IAttributeList)parent).getItemMetaData(((Integer)baseName).intValue());
            }
            if (metaData == null)
            {
                return false;
            }
        }
        catch (Exception e)
        {
            return false;
        }

        return isFromTemplate ? metaData.isFromTemplate() : !metaData.isFromTemplate();

    }

    private static boolean evaluateEquality(IDualOperandExpression expression, Object elementValue, Object expressionValue)
    {
        boolean eqOperation = expression instanceof EqualExpression;

        if (expressionValue == null && elementValue == null)
        {
            if (!eqOperation)
            {
                return false;
            }
        }
        else if (expressionValue == null || elementValue == null)
        {
            if (eqOperation)
            {
                return false;
            }
        }
        else
        {
            boolean isEqual = com.sonicsw.mf.common.config.impl.Util.atomicAndEqual(expressionValue, elementValue);
            if (isEqual && !eqOperation)
            {
                return false;
            }
            else if (!isEqual && eqOperation)
            {
                return false;
            }
        }

        return true;
    }

    private static boolean listContains(Object listObject, Object value)
    {
        if (value == null)
        {
            return false;
        }

        if (listObject instanceof IAttributeList) {

            if (value == null)
            {
                return false;
            }

            IAttributeList list = (IAttributeList) listObject;

            for (int i = 0; i < list.getCount(); i++) {
                if (com.sonicsw.mf.common.config.impl.Util.atomicAndEqual(value, list.getItem(i)))
                {
                    return true;
                }
            }
        }
        else if (listObject instanceof IAttributeSet) {
            IAttributeSet set = (IAttributeSet) listObject;
            
            for (Iterator i = set.getAttributes().keySet().iterator(); i.hasNext(); ) { 
                Object key = i.next();            
                if (com.sonicsw.mf.common.config.impl.Util.atomicAndEqual(value, key))
                {
                    return true;
                }
            }            
        }

        return false;
    }

    private static boolean listItemStartsWith(Object listObject, Object prefix)
    {
        if (!(listObject instanceof IAttributeList))
        {
            return false;
        }

        if (prefix == null)
        {
            return false;
        }

        IAttributeList list = (IAttributeList)listObject;

        for (int i = 0; i < list.getCount(); i++)
        {
            if (com.sonicsw.mf.common.config.impl.Util.startsWith(true, list.getItem(i), prefix))
            {
                return true;
            }
        }

        return false;
    }

}
