/*
 * Copyright (c) 2001 Sonic Software. All Rights Reserved.
 */

package com.sonicsw.mf.common.config.query.impl;
import java.util.HashMap;
import java.util.Iterator;

import com.sonicsw.mf.common.config.ConfigException;
import com.sonicsw.mf.common.config.IAttributeList;
import com.sonicsw.mf.common.config.IAttributeSet;
import com.sonicsw.mf.common.config.ReadOnlyException;
import com.sonicsw.mf.common.config.query.AttributeName;
import com.sonicsw.mf.common.config.query.Select;
import com.sonicsw.mf.common.dirconfig.IDirElement;

// Selects a subset of the elemnt's attributes
public final class ElementSelection
{

    AttributeSetNode m_tree;
 
    public ElementSelection (Select select)
    {
         m_tree = SelectionListToTree(select);
    }

    // Remove unselected attributes
    public void removeUnselectedAttributes(IDirElement element) throws ReadOnlyException, ConfigException
    {
        removeUnselectedAttributes(element.getAttributes(), m_tree);
    }

    private static void removeUnselectedAttributes(IAttributeSet attSet, AttributeSetNode selectionSet)
        throws ReadOnlyException, ConfigException
    {
         Iterator iterator = attSet.getAttributes().keySet().iterator();
         while (iterator.hasNext())
         {
              String attName = (String)iterator.next();
              Object selection = selectionSet.get(attName);
              Object attVal = attSet.getAttribute(attName);
              if (selection == null)
            {
               // Not the attribute or anything under it were selected - remove it
                attSet.deleteAttribute(attName);
            }
            else if (selection instanceof String)
            {
                // The whole attribute was selected - we can leave the whole thing
                continue;
            }
            else if (selection instanceof AttributeListNode) 
              {
                  if (!(attVal instanceof IAttributeList))
                {
                   // Erroneous selection
                    attSet.deleteAttribute(attName);
                }
                else {
                    removeUnselectedAttributes((IAttributeList)attVal, (AttributeListNode)selection); // Select recoursivelly
                }
              }
              else if (selection instanceof AttributeSetNode) 
              {
                  if (!(attVal instanceof IAttributeSet))
                {
                   // Erroneous selection
                    attSet.deleteAttribute(attName);
                }
                else {
                    removeUnselectedAttributes((IAttributeSet)attVal, (AttributeSetNode)selection); // Select recoursivelly
                }
              }
          }
    }

    private static void removeUnselectedAttributes(IAttributeList attList, AttributeListNode selectionList)
        throws ReadOnlyException, ConfigException
    {
        int listCount = attList.getCount();
        for (int i = attList.getCount() - 1; i >= 0; i--)
        {
             Object selection = selectionList.get(new Integer(i));
             Object attVal = attList.getItem(i);
             if (selection == null)
            {
              // Not the attribute or anything under it were selected - remove it
                attList.deleteAttributeItem(i);
            }
            else if (selection instanceof String)
            {
             // The whole attribute was selected - we can leave the whole thing
                continue;
            }
            else if (selection instanceof AttributeListNode)
             {
                 if (!(attVal instanceof IAttributeList))
                {
                  // Erroneous selection
                    attList.deleteAttributeItem(i);
                }
                else {
                    removeUnselectedAttributes((IAttributeList)attVal, (AttributeListNode)selection); // Select recoursivelly
                }
             }
             else if (selection instanceof AttributeSetNode)
             {
                 if (!(attVal instanceof IAttributeSet))
                {
                  // Erroneous selection
                    attList.deleteAttributeItem(i);
                }
                else {
                    removeUnselectedAttributes((IAttributeSet)attVal, (AttributeSetNode)selection); // Select recoursivelly
                }
             }
        }
        
    }

    // Insert the selection list into a tree
    private static AttributeSetNode SelectionListToTree(Select select)
    {
        AttributeName[] nameList = select.getSelectionList();
        AttributeSetNode root = new AttributeSetNode();

        for (int i = 0; i < nameList.length; i++)
        {
            AttributeName attName = nameList[i];
            if (attName.getComponentCount() == 0)
            {
                continue;
            }

            root.insert(attName.getComponent(0), Util.removeFirstComponent(attName));
        }

        return root;
    }

    private static class AttributeSetNode extends HashMap
    {
        static final String SELECTETD_ATTRIBUTE = "";
        private int m_level;

        AttributeSetNode()
        {
            this(0);
        }

        AttributeSetNode(int level)
        {
            super();
            m_level = (level);
        }

        @Override
        public String toString()
        {
            Iterator iterator = keySet().iterator();
            String result = "";
            String spaces = "";
            for (int i = 0; i < m_level; i++)
            {
                spaces += "  ";
            }
            while (iterator.hasNext())
            {
                Object attName = iterator.next();
                result += "\n"+ spaces + attName;
                Object node = get(attName);
                if (node instanceof AttributeSetNode)
                {
                    result += node.toString();
                }
            }
            return result;
        }

        void insert(Object attName, AttributeName nameLowerLevels)
        {
            // The to level of an element is never an AttributeList
            if (!correctType(attName))
            {
                return;
            }

            // The attribute selected is in this level
            if (nameLowerLevels.getComponentCount() == 0)
            {
                put(attName, SELECTETD_ATTRIBUTE);
                return;
            }

            Object insertedBefore = get(attName);
            Object nextComponent = nameLowerLevels.getComponent(0);

            if (insertedBefore == null)
            {
                insertedBefore = (nextComponent instanceof Integer) ? new AttributeListNode(m_level+1) : new AttributeSetNode(m_level+1);
                put(attName, insertedBefore);
            }

            // This attribute was already selected
            if (insertedBefore instanceof String)
            {
                return;
            }
            else if (insertedBefore instanceof AttributeSetNode)
            {
                ((AttributeSetNode)insertedBefore).insert(nextComponent, Util.removeFirstComponent(nameLowerLevels));
            }
        }

        boolean correctType(Object nameComponent)
        {
            return (nameComponent instanceof String);
        }

    }

    private static class AttributeListNode extends AttributeSetNode
    {
        AttributeListNode()
        {
            this(0);
        }

        AttributeListNode(int level)
        {
            super(level);
        }

        @Override
        boolean correctType(Object nameComponent)
        {
            return (nameComponent instanceof Integer);
        }
    }

}
