package com.sonicsw.mf.common.config.upgrade;

import java.io.FileOutputStream;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Properties;
import java.util.Set;
import java.util.Vector;

import com.sonicsw.mf.common.IDirectoryAdminService;
import com.sonicsw.mf.common.IDirectoryFileSystemService;
import com.sonicsw.mf.common.config.IElementIdentity;
import com.sonicsw.mf.common.dirconfig.DirectoryServiceException;
import com.sonicsw.mf.common.dirconfig.IDirElement;
import com.sonicsw.mf.mgmtapi.config.constants.IContainerConstants;

/**
 * <p>Title: </p>
 * <p>Description: </p>
 * <p>Copyright: Copyright (c) 2003</p>
 * <p>Company: Sonic Software </p>
 * @author Mari Davila
 * @version 1.0
 */

public class ConfigUpgrade implements IMigrationProcess
{   
     static ArrayList MQ_TYPES = new ArrayList(4);
     static ArrayList MF_TYPES = new ArrayList(8);
      
     UpgradeEnv m_upgradeEnv;
     DomainManager m_DMUpgrade = null;
     TypedConfigUpgrade m_configUpgrade = null;

     static
     {
         MQ_TYPES.add("MQ_BROKER");
         MQ_TYPES.add("MQ_BACKUPBROKER");
         MQ_TYPES.add("MQ_CLUSTER");
         MQ_TYPES.add("MQ_AUTHORIZATION_POLICY");
         MQ_TYPES.add("MQ_CERTIFICATES_STORE");
         MQ_TYPES.add("MQ_WS_PROTOCOL");
         MF_TYPES.add("MF_CONTAINER");
         MF_TYPES.add("MF_ACTIVATION_DAEMON");
         MF_TYPES.add("MF_AUTHENTICATION_DOMAIN");
         MF_TYPES.add("MF_CONTAINER_COLLECTION");
         MF_TYPES.add("MF_COMPONENT_COLLECTION");
         MF_TYPES.add("MF_COLLECTION_MONITOR");
         MF_TYPES.add("MF_AGENT_MANAGER");
         MF_TYPES.add("MF_BACKUP_AGENT_MANAGER");
         MF_TYPES.add("MF_DIRECTORY_SERVICE");
         MF_TYPES.add("MF_BACKUP_DIRECTORY_SERVICE");
         MF_TYPES.add("MF_LOGGER");
         MF_TYPES.add("MF_HOST_MANAGER");
     }
     
     // used by the SMC
     public ConfigUpgrade(IDirectoryFileSystemService dsFile, Properties env)
     {
    	 m_upgradeEnv = new UpgradeEnv(env);
		 m_upgradeEnv.setDsAdmin((IDirectoryAdminService)dsFile);
		 m_upgradeEnv.setDsFileSystem(dsFile);
		 m_configUpgrade = new TypedConfigUpgrade(m_upgradeEnv);
     }
     
     // used when creating an object to upgrade the DM
     public ConfigUpgrade(Properties env)
     {
    	 m_upgradeEnv = new UpgradeEnv(env);
     } 
     
     public ConfigUpgrade(Properties upgradeProps, HashMap containerProps)
     {
    	 m_upgradeEnv = new UpgradeEnv(upgradeProps, containerProps);
     }
     
     public ConfigUpgrade(UpgradeEnv env)
     {
    	 m_upgradeEnv = env;
     }
     
     public void setConnection(IDirectoryFileSystemService dsFile)
     {
    	 m_upgradeEnv.setDsAdmin((IDirectoryAdminService)dsFile);
		 m_upgradeEnv.setDsFileSystem(dsFile);
     }
     
     public String upgradeDM(IDirElement DMElement) throws Exception
     {
    	 m_DMUpgrade = new DomainManager();
    	 m_DMUpgrade.preConnectUpgrade(m_upgradeEnv);
    	 // create an initial connection to the DS for the domain manager
    	 // upgrade. A new connection will be made afterwards to reinitialize
    	 // the logical namespace which can be modified during the 
    	 // domain manager upgrade
    	 m_upgradeEnv.connectToNewPSEDirectoryService();
    	 m_DMUpgrade.upgrade(m_upgradeEnv);
    	 m_upgradeEnv.dsService.closeDS();
    	 // Now upgrade the container configurations and its components
    	 // we have figured out the components in FindDependencies, and could write
    	 // them out to the properties file, but this way we have the final say on the list
    	 // and don't have to depend on a list that the user could modify
    	 // in the properties file
    	 m_upgradeEnv.connectToNewPSEDirectoryService();
    	 Vector components = m_upgradeEnv.m_utils.getContainerComponents(DMElement);
    	 components.add(DMElement.getIdentity().getName());
    	 String containerName = (String)DMElement.getAttributes().getAttribute(IContainerConstants.CONTAINER_NAME_ATTR);
    	 m_upgradeEnv.putProp(CONTAINERNAMEPROP, containerName);
    	 m_configUpgrade = new TypedConfigUpgrade(m_upgradeEnv);
    	 String message = upgradeConfigs(components);
    	 m_upgradeEnv.dsService.closeDS();
    	 return message;
     }
     
     public String upgradeContainer(String containerID) throws Exception
     {
    	 IDirElement container = m_upgradeEnv.dsFileSystem.getFSElement(containerID, false);
    	 String containerName =  (String)container.getAttributes().getAttribute(IContainerConstants.CONTAINER_NAME_ATTR);
    	 m_upgradeEnv.putProp(CONTAINERNAMEPROP, containerName);
    	 m_configUpgrade = new TypedConfigUpgrade(m_upgradeEnv);
    	 Vector components = m_upgradeEnv.m_utils.getContainerComponents(container);
    	 components.add(containerID);
    	 return upgradeConfigs(components);
     }
     
     /**
      * Called by the SMC for a configuration type and version to find out if 
      * an upgrade is supported.
      * @param configType One of the known MQ configuration types which can be
      * selected from the SMC.
      * MQ_BROKER
      * MQ_BACKUPBROKER
      * MQ_CLUSTER
      * MQ_AUTHORIZATION_POLICY
      * MQ_CERTIFICATES_STORE
      * MQ_WS_PROTOCOL
      * MF_CONTAINER
      * MF_AUTHENTICATION_DOMAIN
      * MF_COMPONENT_COLLECTION
      * MF_CONTAINER_COLLECTION
      * MF_LOGGER
      * MF_ACTIVATION_DAEMON
      * MF_COLLECTION_MONITOR
      * MF_DIRECTORY_SERVICE
      * MF_AGENT_MANAGER
      * 
      * @param version Product version,  ie. 7.5, 8.0
      * @return true if ConfigUpgradeDriver can upgrade the configuration.
      * @throws Exception 
      */

     public static boolean isUpgradeSupported(String configType, String version)
     {   	
          if (UpgradeEnv.isDebugAll())
         {
             System.err.println("ConfigUpgrade.isUpgradeSupported configType == " + configType +
                         " version == " + version);
         }
         if (configType == null)
        {
            return false;
        }
         if (version == null)
        {
            return false;
        }
         // MF_LOGGER is an MF type, but its version started at 7.0
         if ((MQ_TYPES.contains(configType) || MF_TYPES.contains(configType) ||
             configType.equals("XQ_CONTAINER") || configType.equals("MF_LOGGER")))
         {
        	 Set productVersions = SUPPORTED_PRODUCT_VERSIONS.keySet();
        	 return productVersions.contains(version);
         }
         return false;
     }
     
     /**
      * This is the method used both by the SMC and internally to upgrade a 
      * configuration. The SMC calls ConfigurationDependencies to figure out the
      * dependencies of a configuration, and then calls upgradeConfigs which calls this
      * method for each of the configurations.
      * @param configName The logical name of the configuration selected to be 
      * upgraded.
      * @param dsAdminArg IDirectoryAdminService object
      * @param dsFileSystemArg IDirectoryFileSystemService object. Can be the same 
      * as dsAdminArg.
      * @param controlCode MQ control code
      * @return A string which is a message that should be displayed to the user, 
      * or null. For instance, upgrading a container with secondary connection 
      * information from 7.0 to 7.5 causes the upgrade to print an informational 
      * message.
      * @throws Exception
      */

     
     public String upgradeConfig(String configID) throws Exception
     {
    	 String message = null;
    	 IDirElement upgradeEl = null;
         if (m_upgradeEnv.isDEBUG())
        {
            System.out.println("ConfigUpgrade.upgradeConfig, config == " + configID);
        }
         try
         {
             upgradeEl = m_upgradeEnv.dsFileSystem.getFSElement(configID, true, true);
         }
         catch (DirectoryServiceException e) {} // try with _Default

         if (upgradeEl == null)
        {
            try
             {
                 // try adding _Default to it
                 upgradeEl = m_upgradeEnv.dsFileSystem.getFSElement(configID + "/_Default", true, true);
             }
             catch (DirectoryServiceException ex) {}
        }
         if (upgradeEl != null)
         {
        	 boolean upgradeCurrent = false;
        	 String elType = upgradeEl.getIdentity().getType();
        	 // in certain cases, we allow some configs to be "upgraded" a second time
        	 // to put the correct host dependent values in the configuration
        	 if (!upgradeEl.getIdentity().getReleaseVersion().equals(CURRENT_RELEASE_VERSION) ||
        			 elType.equals("MQ_BROKER") || elType.equals("MQ_BACKUPBROKER")||
        			 elType.equals("MF_CONTAINER") || elType.equals("MF_BACKUP_DIRECTORY_SERVICE"))
        	 {                 
                 if (elType.equals("MQ_BROKER") || elType.equals("MQ_BACKUPBROKER"))
                {
                    m_configUpgrade.upgradeMQ_BROKER(upgradeEl);
                }
                else if (elType.equals("MF_CONTAINER"))
                {
                    message = m_configUpgrade.upgradeMF_CONTAINER(upgradeEl);
                }
                else if (elType.equals("MF_ACTIVATION_DAEMON"))
                {
                    m_configUpgrade.upgradeMF_ACTIVATION_DAEMON(upgradeEl);
                }
                else if (elType.equals("MF_DIRECTORY_SERVICE"))
                {
                    m_configUpgrade.upgradeMF_DIRECTORY_SERVICE(upgradeEl);
                }
                else if (elType.equals("MF_BACKUP_DIRECTORY_SERVICE"))
                {
                    m_configUpgrade.upgradeMF_BACKUP_DIRECTORY_SERVICE(upgradeEl);
                }
                else if (elType.equals("MF_AGENT_MANAGER") || (elType.equals("MF_BACKUP_AGENT_MANAGER")))
                {
                    m_configUpgrade.upgradeMF_AGENT_MANAGER(upgradeEl);
                }
                else if (elType.equals("MF_COLLECTION_MONITOR"))
                {
                    m_configUpgrade.upgradeMF_COLLECTION_MONITOR(upgradeEl);
                }
                else if (elType.equals("MQ_CLUSTER"))
                {
                    m_configUpgrade.upgradeMQ_CLUSTER(upgradeEl);
                }
                else if (elType.equals("XQ_CONTAINER"))
                {
                    m_configUpgrade.upgradeXQ_CONTAINER(upgradeEl);
                }
                else if (elType.equals("MF_LOGGER"))
                {
                    m_configUpgrade.upgradeMF_LOGGER(upgradeEl);
                }
                else if (elType.equals("MF_HOST_MANAGER"))
                {
                    m_configUpgrade.upgradeMF_HOST_MANAGER(upgradeEl);
                }
                else if (MF_TYPES.contains(elType) || MQ_TYPES.contains(elType))
                {
                    m_configUpgrade.basicConfigUpgrade(upgradeEl);
                } 
        	 }// but don't upgrade unlnown configurations
         }
         return message;
     }
     
     public String upgradeConfigs(Vector configIDs) throws Exception
     {
    	 Vector weededConfigList = new Vector();
         Vector clustersList = new Vector();  
         Vector containerList = new Vector();
         boolean createdDS = false;
         
         // we can get here after a DM upgrade from the migration tool; the connection
         // has been closed and must be reopened.
         if (m_upgradeEnv.dsFileSystem == null)
         {
        	 m_upgradeEnv.connectToNewPSEDirectoryService();
        	 createdDS = true;
         }
    	 Iterator configsIt = configIDs.iterator();
         IDirElement el;
         int dsIndex = 0;
         while (configsIt.hasNext())
         {
             el = null;
             String upgradeConfigName = (String) configsIt.next();
             String foundConfigName = upgradeConfigName;
             if (m_upgradeEnv.isDEBUG())
            {
                System.out.println("Looking for element named " + upgradeConfigName);
            }
             try
             {
                 el = m_upgradeEnv.dsFileSystem.getFSElement(upgradeConfigName, false);
             }
             catch (Exception e) {}
             if (el == null)
            {
                try
                 {
                 	foundConfigName = upgradeConfigName + "/_Default";
                     if (m_upgradeEnv.isDEBUG())
                    {
                        System.out.println("Looking for element names " + upgradeConfigName + "/Default");
                    }
                     el = m_upgradeEnv.dsFileSystem.getFSElement(upgradeConfigName +
                         "/_Default", false);
                 }
                 catch (Exception e) {}
            }
             // the directory service should get upgraded before any other component
             // containers should be upgraded before the ADs that start them. The JVM_OPTIONS of the
             // ADs override the initial value of the JVM_ARGS of the container
             // Cluster configurations should go at the end, so we can collect the node names
             // from the member brokers. We collect the cluster configurations here and add them 
             // at the end later.
             if (el != null)
             {
                 if (el.getIdentity().getType().equals("MF_DIRECTORY_SERVICE"))
                 {
                     String FTrole = (String)el.getAttributes().getAttribute("FAULT_TOLERANT_ROLE");
                     if (FTrole == null || FTrole.equals("PRIMARY"))
                    {
                       //will work in 7.5 even though this attribute doesn't exist
                        weededConfigList.add(dsIndex, foundConfigName);
                    }
                    else
                    {
                        weededConfigList.add(foundConfigName);
                    }
                     dsIndex = dsIndex + 1;
                 }
                 // instances of templates are upgraded when the templates are upgraded, so they don't
                 // go in the list
                 else if (el.getSuperElementName() == null)
                 {
                     // add the Activation Daemons at the beginning of the list
                     if (el.getIdentity().getType().equals("MF_CONTAINER"))
                     {
                         weededConfigList.add(dsIndex, foundConfigName);
                         // keep a list of containers to use in haltNotifications
                         containerList.add(foundConfigName);
                     }
                     else if (el.getIdentity().getType().equals("MQ_CLUSTER"))
                    {
                        clustersList.add(foundConfigName);
                    }
                    else
                    {
                        weededConfigList.add(foundConfigName);
                    }
                 }
                 else
                     if (m_upgradeEnv.isDEBUG())
                    {
                        System.out.println("**********Did not add " + upgradeConfigName + " to weededList");
                    }
             }
             else
             {
                 if (m_upgradeEnv.isDEBUG())
                {
                    System.out.println("***********Couldn't find element " + upgradeConfigName);
                }
                 throw new Exception("ConfigUpgrade could not find configuration " + upgradeConfigName);
             }
          }

          // now put the clusters at the end of the list
          weededConfigList.addAll(clustersList);
          // Now find all the containers in the clusters,
          // so that we can stop notifications. The entire list
          // is not typically in the list of configurations to
          // upgrade.
          containerList.addAll(findClusterContainers(clustersList, weededConfigList));
          String upgradeMessage = null;
          // suspend notifications
          Iterator containersIT = containerList.iterator();
  		  while (containersIT.hasNext()) 
  		  {
  		      String ID = (String) containersIT.next();
  		      if (m_upgradeEnv.isDEBUG())
            {
                System.out.println("ConfigUpgrade.upgradeConfigs stopping notifications for " + ID);
            }
  		      m_upgradeEnv.dsFileSystem.suspendChangeNotifications(ID, null);
  		  } 
  		  Iterator configsIT = weededConfigList.iterator();
  		  // upgrade configurations
  		  while (configsIT.hasNext()) 
  		  {
  			String ID = (String) configsIT.next();
  			try
  			{
  				upgradeMessage = Utils.addToUpgradeMessage(upgradeMessage, upgradeConfig(ID));  		
  			}
  			catch (Exception e)
  			{
  				DirectoryServiceException dirE = new DirectoryServiceException("Unable to upgrade " + ID);
  				dirE.initCause(e);
  				throw dirE;
  			}
  			finally
  			{
  				if (createdDS)
                {
                    m_upgradeEnv.dsService.closeDS();
                }
  			}
  		}
    	 return upgradeMessage;
     }

      static private void generateContainerXml(String configID, String xmlFileLocation, IDirectoryAdminService dsAdmin)
      throws Exception
      {
          IDirElement containerEl = dsAdmin.getElement(configID, false);
          String xmlString = com.sonicsw.mf.common.util.Container.exportContainerBootConfiguration(containerEl, dsAdmin.getDomain());
          FileOutputStream xmlFileStream = new FileOutputStream(xmlFileLocation);
          xmlFileStream.write(xmlString.getBytes());
          xmlFileStream.close();
      }
      
      private Vector findClusterContainers(Vector clustersList, Vector knownAlready) throws Exception
      {
    	  Vector depContainers = new Vector();
    	  ConfigurationDependencies dep = new ConfigurationDependencies(m_upgradeEnv);
    	  Iterator clustersIT = clustersList.iterator();
    	  while (clustersIT.hasNext())
    	  {
    		  String clusterID = (String)clustersIT.next();
    		  if (m_upgradeEnv.isDEBUG())
            {
                System.out.println("** ConfigUpgrade.findClusterContainers called for " + clusterID);
            }
    		  Vector clusterDep = dep.findNotificationDependencies(clusterID, knownAlready);
    	      Iterator clusterDepIT = clusterDep.iterator();
    	      if (m_upgradeEnv.isDEBUG())
            {
                System.out.println("** ConfigUpgrade.findClusterContainers done for " + clusterID);
            }
    	      while (clusterDepIT.hasNext())
    	      {
    	    	  IElementIdentity depID = (IElementIdentity)clusterDepIT.next();
    	    	  String depType = depID.getType();
    	    	  if (depType.equals(IContainerConstants.DS_TYPE))
    	    	  {
    	    		  if (m_upgradeEnv.isDEBUG())
                    {
                        System.out.println("** " + depID.getName());
                    }
    	    		  depContainers.add(depID.getName());
    	    	  }
    	      }
    	      
    	  }
    	  return depContainers;
      }
}