package com.sonicsw.mf.common.config.upgrade;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Vector;

import com.sonicsw.mf.common.IDirectoryAdminService;
import com.sonicsw.mf.common.IDirectoryFileSystemService;
import com.sonicsw.mf.common.config.IAttributeSet;
import com.sonicsw.mf.common.config.IElementIdentity;
import com.sonicsw.mf.common.config.IMFDirectories;
import com.sonicsw.mf.common.config.Reference;
import com.sonicsw.mf.common.config.query.FromElementType;
import com.sonicsw.mf.common.config.query.Query;
import com.sonicsw.mf.common.dirconfig.DirectoryServiceException;
import com.sonicsw.mf.common.dirconfig.IDirElement;
import com.sonicsw.mf.jmx.client.JMSConnectorClient;
import com.sonicsw.mf.mgmtapi.config.constants.IContainerConstants;
/**
 * <p>Title: </p>
 * <p>Description: </p>
 * <p>Copyright: Copyright (c) 2003</p>
 * <p>Company: Sonic Software</p>
 * @author Mari Davila
 * @version 1.0
 */

public class ConfigurationDependencies implements IMigrationProcess
{

    static Vector allDependencies = new Vector();
    IDirectoryFileSystemService dsFileSystem;
    Utils m_utils = null;
    static JMSConnectorClient connector = null;
    static boolean wholeCluster = true;
    static boolean iwUpgrade = false;
    static boolean DEBUG = System.getProperty("DebugAll.debug", "false").equals("true");
    HashMap m_containerComponents = new HashMap();
	DirectoryService m_dsService = null;
	private boolean m_allowUpgraded = false;
    
    private String checkContainerName(String configName) throws DirectoryServiceException
    {
        String errorMessage = "ERRORWARNING: The configuration name you entered is not a container name in this domain. Please re-enter the container to be upgraded or correct the connection arguments";
    	IDirElement el = dsFileSystem.getFSElement(configName, false);
    	
        if (el == null)
        {
            return errorMessage;
        }
        String type = el.getIdentity().getType();
        if (!type.equals(IContainerConstants.DS_TYPE))
        {
            return errorMessage;
        }
        return null;    // a null return is good; it means no errors    
    }

    public void closeDS() throws Exception
    {
        m_dsService.closeDS();
    } 
    
    public IDirectoryFileSystemService getDS()
    {
    	return dsFileSystem;
    }
    
    public ConfigurationDependencies(String url, String domainName, String username, 
            String password, String mgmtNode, int rt) throws Exception
    {
    	if (url.endsWith(".xml"))
        {
            m_dsService = new DirectoryService(url, true);
        }
        else
        {
            m_dsService = new DirectoryService(url, username, password, domainName, mgmtNode, rt);
        }
        initFromConnection(m_dsService.getFileSystemService());
    }
    
    public ConfigurationDependencies(String url, String domainName, String username, 
            String password, String mgmtNode) throws Exception
    {
    	this(url, domainName, username, password, mgmtNode, REQUEST_TIMEOUT_DEFAULT);
    }
    
    public ConfigurationDependencies(UpgradeEnv env)
    {
    	this.dsFileSystem = env.dsFileSystem;
    	DEBUG = env.isDEBUG();
    	m_utils = new Utils(env);
    }

    public ConfigurationDependencies(IDirectoryFileSystemService file) throws Exception
    {
        initFromConnection(file);
    }
    
    public void setDebug(boolean debug)
    {
    	DEBUG = debug;
    }
    
    private void initFromConnection(IDirectoryFileSystemService ds) throws Exception
    {
    	dsFileSystem = ds;
		UpgradeEnv env = new UpgradeEnv(null);
		env.setDsFileSystem(ds);
        m_utils = new Utils(env);
    }
    

    /**
     * Called by the SMC and findDependencies tool
     * to find the upgrade dependencies of a selected
     * configuration. The SMC will call ConfigUpgrade.upgradeConfig
     * for each one of the configurations in the return value.
     * @param configName Configuration selected in the SMC for upgrade
     * @param knownAlready The SMC passes an empty vector here.
     * @return a Vector with the logical names of all the 
     * configurations which must be upgraded with configName
     * @throws Exception
     */

    public HashMap findConfigDependencies(String configName) throws Exception
    {
    	if (DEBUG)
        {
            System.out.println("ConfigurationDependencies.findConfigDependencies starting with configid " + configName);
        }
    	
    	IElementIdentity configID = null;
    	try
    	{
    	    configID = dsFileSystem.getFSIdentity(configName);    	   	
    	}    
	    catch (DirectoryServiceException dirE) {} 
	    if (configID == null)
        {
            try
            {
                configID = dsFileSystem.getFSIdentity(configName + IMFDirectories.MF_DIR_SEPARATOR + "_Default");
            }
            catch (DirectoryServiceException e) {}
        }
		if (configID == null)
        {
            throw new MigrationException(configName + " was not found in the Directory Service!");
        } 
    	String releaseVersion = configID.getReleaseVersion();
    	if (releaseVersion.equals(CURRENT_RELEASE_VERSION))
        {
            // this won't be true from the SMC, only from the upgrade tool
    		m_allowUpgraded = true;
        }
    	String type = configID.getType();
    	if (DEBUG)
        {
            System.out.println("ConfigurationDependencies.findConfigDependenices calling findConfigDependenciesInternal");
        }
        Vector configIDs = findConfigDependenciesInternal(configID.getName(), new Vector(), false, true);
        HashMap idsMap = new HashMap();
        if (DEBUG)
        {
            System.out.println("ConfigurationDependencies.findConfigDependencies calling addContainerComponents");
        }
        addContainerComponents(idsMap, configIDs);
        if (DEBUG)
        {
            System.out.println("ConfigurationDependencies.findConfigDependencies returning a map of " + idsMap.size() + " elements");
        }
        return idsMap;
    }
    
    // used for the upgrade after installing a workbench
    public HashMap findAllContainers(String DMName) throws Exception
    {
        // do not upgrade if there are configurations of 
        // unsupported versions
        checkAllConfigVersions();
    	HashMap ids = new HashMap();
    	Vector dependencies = new Vector();
    	iwUpgrade=true;
    	addAllContainersToDependencies(new Vector(),dependencies );
    	this.addSpecialMarker(dsFileSystem.getFSIdentity(DMName), "LOCAL");
    	addContainerComponents(ids, dependencies);
    	return ids;
    }
    
    private void addContainerComponents(HashMap dependenciesMap, Vector configIDs)
    {
    	if (DEBUG)
        {
            System.out.println("ConfigurationDependencies.addContainerComponents starting");
        }
    	Iterator idsIterator = configIDs.iterator();
        while (idsIterator.hasNext())
        {
        	IElementIdentity id = (IElementIdentity)idsIterator.next();
        	if (DEBUG)
            {
                System.out.println("ConfigurationDependencies.addContainerComponents finding components for " + id.getName());
            }
        	Vector components = (Vector)m_containerComponents.get(id);
        	if (DEBUG && components != null)
            {
                System.out.println("ConfigurationDependencies.addContainerComponents found " + components.size() + " components for " + id.getName());
            }
        	dependenciesMap.put(id, components);
        }
    	if (DEBUG)
        {
            System.out.println("ConfigurationDependencies.addContainerComponents exiting");
        }
    }
    
    Vector findNotificationDependencies(String configName, Vector knownAlready) throws Exception
    {
    	if (DEBUG)
        {
            System.out.println("ConfigurationDependencies.findNotificationDependencies " + configName);
        }
    	knownAlready.remove(configName); // so that the dependencies will be recomputed
    	Vector deps = findConfigDependenciesInternal(configName, knownAlready, true, false);
        knownAlready.add(configName);
        return deps;
    }

    void setWholeClusterUpgrade(boolean wholeClusterUpgrade)
    {
        wholeCluster = wholeClusterUpgrade;
    }

    Vector  findConfigDependenciesInternal(String configName, Vector knownAlready, boolean all, boolean onThisHost) throws Exception
    {
    	if (DEBUG)
        {
            System.out.println("ConfigurationDependencies.findConfigDependenciesInternal starting for configName " + configName);
        }
        Vector copyKnown = new Vector(knownAlready);
        Vector dependencies = new Vector();
        if (!copyKnown.contains(configName))
        {
            if (DEBUG)
            {
                System.out.println("ConfigurationDependencies.findConfigDependenciesInternal processing logical name " + configName);
            }
            IDirElement el = dsFileSystem.getFSElement(configName, false);
            Vector subDependencies = new Vector();
            // check the version of the element (through addDependency). We could get here with a 7.0 configuration
            // which shouldn't end up on the list. For instance, one could upgrade a broker, then decide to not
            // upgrade all the brokers in the cluster. The dependencies for the subsequent brokers
            // should not include the cluster, as it has been upgraded already

            // add cluster configurations after the member brokers. The SMC calls the
            // upgrade iteratively with the configurations in the same order given to
            // it by this method.
            
            copyKnown.add(el.getIdentity());
            if (!el.getIdentity().getType().equals("MQ_CLUSTER"))
            {
                addDependency(dependencies, el.getIdentity(), onThisHost);
            }

            //dependencies.add(el.getIdentity().getName());
            String elType = el.getIdentity().getType();
            if (elType.equals("MQ_BROKER") || elType.equals("MQ_BACKUPBROKER"))
            {
                subDependencies = findBrokerDependencies(el, copyKnown, all, onThisHost);
            }
            else if (elType.equals("MF_CONTAINER"))
            {
            	try
            	{
                    subDependencies = findContainerDependencies(el, copyKnown, all, onThisHost);
                    if (onThisHost)
                    {
                        addSpecialMarker(el.getIdentity(), "LOCAL");
                    }
                    
            	}
            	catch (OSrvrXSrvrException oldStuff)
            	{
            		// This container has an ESB container with XMLServiceType, 
            		// OrchestrationServiceType or BPELServiceType service instances. The MF container
            		// will not be upgraded
            		return new Vector();
            		
            	}
            }
            else if (elType.equals("MQ_CLUSTER"))
            {
                subDependencies =  findClusterDependencies(el, copyKnown, all);
            }
            else if (elType.equals("MF_DIRECTORY_SERVICE"))
            {
                // DS dependents are the same in either the complete dependencies case or the minimum dependencies case
                subDependencies = findDSDependencies(el, copyKnown, all, onThisHost);
            }
            else  if (elType.equals("MF_ACTIVATION_DAEMON"))
            {
                subDependencies = findDaemonDependencies(el, copyKnown, all, onThisHost);
            }
            else if (elType.equals("MF_AGENT_MANAGER") )
            {
                subDependencies = findAgentManagerDependencies(el, copyKnown, all, onThisHost);
            }
            else if (elType.equals("XQ_CONTAINER"))
            {
                subDependencies = findXQContainerDependencies(el, copyKnown, all, onThisHost);
            }
            else if (elType.equals("MF_COLLECTION_MONITOR") ||
                elType.equals("MF_LOGGER") || elType.equals("MF_BACKUP_DIRECTORY_SERVICE") ||
                elType.equals("MF_BACKUP_AGENT_MANAGER") ||
                elType.equals("MF_HOST_MANAGER"))
            {
                subDependencies = findComponentDependencies(el, copyKnown, all);
            }
            dependencies.addAll(subDependencies);
            copyKnown.addAll(subDependencies);
            // now add the cluster
            if (el.getIdentity().getType().equals("MQ_CLUSTER"))
            {
            	if (DEBUG)
                {
                    System.out.println("ConfigurationDependencies.findConfigDependenciesInternal adding cluster");
                }
                addDependency(dependencies, el.getIdentity(), onThisHost);
            }
            // if the element is a template, upgrade the template and all of its instances
            if (el.isTemplate())
            {
                String[] subclasses = el.getSubclassedList();
                for (int i = 0; i < subclasses.length; i++)
                {
                    String sub = subclasses[i];
                    if (!copyKnown.contains(dsFileSystem.getFSIdentity(sub)))
                    {
                        subDependencies = findConfigDependenciesInternal(sub, copyKnown, all, false);
                        dependencies.addAll(subDependencies);
                        copyKnown.addAll(subDependencies);
                    }
                }
                addSpecialMarker(el.getIdentity(), "TEMPLATE");
            }
            // if the element is an instance of a template, upgrade its template
            if (el.getSuperElementName() != null)
            {
                String templateName = el.getSuperElementName();
                if (!copyKnown.contains(dsFileSystem.getFSIdentity(templateName)))
                {
                    subDependencies = findConfigDependenciesInternal(templateName, copyKnown, all, false);
                    dependencies.addAll(subDependencies);
                    copyKnown.addAll(subDependencies);
                }
            }
        }
        if (DEBUG)
        {
            System.out.println("ConfigurationDependencies.findConfigDependenciesInternal exiting");
        }
        return dependencies;
    }

    private Vector findContainerDependencies(IDirElement container, Vector knownAlready, boolean all, boolean onThisHost) throws Exception
    {
    	if (DEBUG)
        {
            System.out.println("ConfigurationDependencies.findContainerDependencies container " + container.getIdentity().getName() + " onThisHost == " + onThisHost);
        }
        Vector dependencies = new Vector();
        Vector componentIDs = new Vector();
        Vector copyKnown = new Vector(knownAlready);
        // first add all the components in this container to the list of dependencies
        IAttributeSet components = (IAttributeSet)container.getAttributes().getAttribute("COMPONENTS");
        Iterator keyIterator = components.getAttributes().keySet().iterator();
        while (keyIterator.hasNext())
        {
            String key = (String)keyIterator.next();
            IAttributeSet componentAttrs = (IAttributeSet)components.getAttribute(key);
            Reference compRef = (Reference)componentAttrs.getAttribute("CONFIG_REF");
            String compName = compRef.getElementName();
            IDirElement component = dsFileSystem.getFSElement(compName, false);
            IElementIdentity id = component.getIdentity();
            componentIDs.add(id);

            // MF_LOGGER gets upgraded separately
            if (!copyKnown.contains(id))
            {
                Vector componentDependencies = findConfigDependenciesInternal(compName, copyKnown, all, onThisHost);
                dependencies.addAll(componentDependencies);
                copyKnown.addAll(componentDependencies);               
            }
        }
        Vector alreadySetVector = (Vector)m_containerComponents.get(container.getIdentity());
        if (alreadySetVector == null)
        {
            m_containerComponents.put(container.getIdentity(), componentIDs);
        }
        else
        {
            alreadySetVector.addAll(componentIDs);
        }

        Query allDaemonsQuery;
        IDirElement[] daemons;

        try
        {
            allDaemonsQuery = new Query();
            allDaemonsQuery.setFrom(new FromElementType("MF_ACTIVATION_DAEMON"));
            daemons = dsFileSystem.getFSElements(allDaemonsQuery, false);
        }
        catch (Exception e)
        {
            daemons = new IDirElement[0];
        }
        for (int i=0; i<daemons.length; i++)
        {
            IDirElement daemon = daemons[i];
            IAttributeSet containers = (IAttributeSet)daemon.getAttributes().getAttribute("CONTAINERS");
            HashMap containersMap = containers.getAttributes();

            Collection containersCollection = containersMap.values();
            Iterator containersIterator = containersCollection.iterator();
            while (containersIterator.hasNext())
            {
                IAttributeSet containerSet = (IAttributeSet) containersIterator.next();
                Reference containerRef = (Reference) containerSet.getAttribute("CONTAINER_REF");
                String containerName = containerRef.getElementName();
                if (containerName.equals(container.getIdentity().getName()) && !copyKnown.contains(daemon.getIdentity()))
                {
                    Vector daemonDependencies = findConfigDependenciesInternal(daemon.getIdentity().getName(), knownAlready, all, onThisHost);
                    dependencies.addAll(daemonDependencies);
                    copyKnown.addAll(daemonDependencies);
                   // dependencies.add(daemon.getIdentity().getName());
                }
            }
        }
        return dependencies;
    }

    private Vector findComponentDependencies(IDirElement monitor, Vector knownAlready, boolean all) throws Exception
    {
        Vector dependencies = new Vector();
        Vector copyKnown = new Vector(knownAlready);
        findContainerForComponent(monitor.getIdentity().getName(), copyKnown, dependencies, all);
        return dependencies;
    }
    
    private Vector findXQContainerDependencies(IDirElement esbContainer, Vector knownAlready, boolean all, boolean onThisHost) throws Exception
    {
    	// check that there are no OSrvr or XMLServer service instances in this
    	// ESB container
    	String xqContainerName = esbContainer.getIdentity().getName();
    	IDirectoryAdminService dsAdmin = (IDirectoryAdminService)dsFileSystem;
        String storageXQName = dsFileSystem.logicalToStorage(xqContainerName);			
		IDirElement xqContainer = dsAdmin.getElement(storageXQName, false);
        IAttributeSet servicesSet = (IAttributeSet)xqContainer.getAttributes().getAttribute("services");
        // throw an exception if this container has an instance of OService or XMLService
        if (servicesSet != null)
        {
            checkOSrvrXSrvr(servicesSet, dsAdmin);
        }
        Vector dependencies = findComponentDependencies(esbContainer, knownAlready, all);
        return dependencies;
    }

    private Vector findBrokerDependencies(IDirElement broker, Vector knownAlready, boolean all, boolean onThisHost) throws Exception
    {
    	if (DEBUG)
        {
            System.out.println("ConfigurationDependencies.findBrokerDependencies " + broker.getIdentity().getName());
        }
        Vector dependencies = new Vector();
        Vector copyKnown = new Vector(knownAlready);
        // if this broker is in a cluster, find out if all brokers in the cluster have to be upgraded
        IAttributeSet brokerReferences = (IAttributeSet)broker.getAttributes().getAttribute("CONFIG_ELEMENT_REFERENCES");
        Reference clusterReference = (Reference)brokerReferences.getAttribute("CLUSTER_CONFIG_ELEMENT_REF");
        if (clusterReference != null)
        {
        	if (DEBUG)
            {
                System.out.println("ConfigurationDependencies.findBrokerDependencies " + broker.getIdentity().getName() +
                		" found cluster reference ");
            }
            String clusterName = clusterReference.getElementName();
            if (!copyKnown.contains(clusterReference))
            {           	
                Vector clusterDependencies = findConfigDependenciesInternal(clusterName, copyKnown, all, false);
                dependencies.addAll(clusterDependencies);
                copyKnown.addAll(clusterDependencies);
            }
        }
             
            Reference backupBrokerRef = (Reference) brokerReferences.getAttribute("BACKUP_CONFIG_ELEMENT_REF");
            if (backupBrokerRef != null)
            { // will only be there for primary brokers
            	if (DEBUG)
                {
                    System.out.println("ConfigurationDependencies.findBrokerDependencies " + broker.getIdentity().getName() +
                    " found backup broker reference ");
                }
                String backupName = backupBrokerRef.getElementName();
                if (!copyKnown.contains(dsFileSystem.getFSIdentity(backupName)))
                {
                    Vector backupDependencies = findConfigDependenciesInternal(backupName, copyKnown, all, !onThisHost);
                    dependencies.addAll(backupDependencies);
                    copyKnown.addAll(backupDependencies);
                }
           
            }
            else
            	if (DEBUG)
                {
                    System.out.println("ConfigurationDependencies.findBrokerDependencies " + broker.getIdentity().getName() +
                    " did not find backup broker reference ");
                }
            Reference primaryBrokerRef = (Reference) brokerReferences.getAttribute("PRIMARY_CONFIG_ELEMENT_REF");
            if (primaryBrokerRef != null)
            { // will only be there for backup brokers
            	if (DEBUG)
                {
                    System.out.println("ConfigurationDependencies.findBrokerDependencies " + broker.getIdentity().getName() +
                    " found primary broker reference ");
                }
                String primaryName = primaryBrokerRef.getElementName();
                if (!copyKnown.contains(dsFileSystem.getFSIdentity(primaryName)))
                {
                    Vector primaryDependencies = findConfigDependenciesInternal(primaryName, copyKnown, all, !onThisHost);
                    dependencies.addAll(primaryDependencies);
                    copyKnown.addAll(primaryDependencies);
                }
            }
            else
            	if (DEBUG)
                {
                    System.out.println("ConfigurationDependencies.findBrokerDependencies " + broker.getIdentity().getName() +
                    " did not find primary broker reference ");
                }
        
        // If this broker is in a container, the container needs to be upgraded also
        findContainerForComponent(broker.getIdentity().getName(), copyKnown, dependencies, all);
        return dependencies;
    }
    
    private void addAllContainersToDependencies(Vector knownAlready, Vector dependencies)
        throws Exception
    {
        IDirElement[] fsContainers = null;

        Query containerTypeQuery = new Query();
        containerTypeQuery.setFrom(new FromElementType("MF_CONTAINER"));
        fsContainers = dsFileSystem.getFSElements(containerTypeQuery, false);
        if (fsContainers == null)
        {
            throw new Exception("Could not find any containers in the DS!");
        }
        int containerCount = fsContainers.length;
        for (int i=0; i< containerCount; i++)
        {
            IDirElement container = null;
            IElementIdentity containerID = null;

            container = fsContainers[i];
            containerID = container.getIdentity();
            
            if (!knownAlready.contains(containerID))
            {
                knownAlready.add(containerID);
                addDependency(dependencies, containerID, true);
            }
            Vector subDependencies = null;
            try
            {
            	subDependencies = findContainerDependencies(container, knownAlready, true, true);
            	knownAlready.addAll(subDependencies);
                dependencies.addAll(subDependencies);
            }
            catch (OSrvrXSrvrException oldStuff)
            {
            	// This container will not be upgraded
            	    dependencies.remove(containerID);
            }           
        }
    }

    // helper method to find the container of a component and its dependencies. copyKnown is the copy of the
    // known ids vector which is used internally by each findXXXDependencies. dependencies is the vector where
    // dependencies are being added. all is true if we are finding dependencies accross containers, false if we just
    // want dependencies in one container.

    private ArrayList findContainerForComponent(String componentName, Vector copyKnown, Vector dependencies, boolean all)
        throws Exception
    {
        Query allContainersQuery = new Query();
        ArrayList inContainers = new ArrayList();
        IElementIdentity[] allContainers = null;
        IDirElement[] fsContainers = null;

        Query containerTypeQuery = new Query();
        containerTypeQuery.setFrom(new FromElementType("MF_CONTAINER"));
        fsContainers = dsFileSystem.getFSElements(containerTypeQuery, false);
        if ((allContainers == null) && (fsContainers == null))
        {
            throw new Exception("Could not find any containers in the DS!");
        }
        int containerCount = ((allContainers != null) ? allContainers.length : fsContainers.length);
        for (int i=0; i< containerCount; i++)
        {
            IDirElement container = null;
            String containerName = null;

            container = fsContainers[i];
            containerName = container.getIdentity().getName();
            IAttributeSet components = (IAttributeSet)container.getAttributes().getAttribute("COMPONENTS");
            IAttributeSet componentSet = findComponent(components, componentName);
            if (DEBUG && componentSet != null && copyKnown.contains(container.getIdentity()))
            {
                System.out.println("ConfigurationDependencies.findContainerForComponent " + componentName + " not adding " + containerName + " because it is in copyKnown already");
            }
            if ((componentSet != null))
            {
            	inContainers.add(container.getIdentity());
            	if (!copyKnown.contains(container.getIdentity()))
            	{
                    Vector containerDependencies = findConfigDependenciesInternal(containerName, copyKnown, all, false);
                    dependencies.addAll(containerDependencies);
                    copyKnown.addAll(containerDependencies);
            	}
            }
        }
        return inContainers;
    }

    private IAttributeSet findComponent(IAttributeSet componentsSet, String idToFind)
    {
        HashMap setMap = componentsSet.getAttributes();
        Collection mapValues = setMap.values();
        Iterator valuesIT = mapValues.iterator();
        while (valuesIT.hasNext())
        {
            IAttributeSet componentSet = (IAttributeSet)valuesIT.next();
            Reference compReference = (Reference)componentSet.getAttribute("CONFIG_REF");
            if (compReference.getElementName().compareTo(idToFind) == 0)
            {
                return componentSet;
            }
        }
        return null;
    }

    private Vector findClusterDependencies(IDirElement cluster, Vector knownAlready, boolean all) throws Exception
    {
        Vector copyKnown = new Vector(knownAlready);
        Vector dependencies = new Vector();
        if (DEBUG)
        {
            System.out.println("Entering ConfigurationDependencies.findClusterDependencies");
        }
        // There's no dependencies if only the components in a container are being upgraded
        // return an empty vector in that case
        if (all)
        {
            // add all the brokers (and their dependencies) in this cluster to the dependencies
            Object brokers = cluster.getAttributes().getAttribute("CLUSTER_MEMBERS");

            IAttributeSet brokersSet = (IAttributeSet) brokers;
            HashMap attributes = brokersSet.getAttributes();
            Collection brokerRefs = attributes.values();
            Iterator brokerIt = brokerRefs.iterator();
            while (brokerIt.hasNext())
            {
                Reference broker = (Reference) brokerIt.next();
                String brokerName = broker.getElementName();
                if (!copyKnown.contains(dsFileSystem.getFSIdentity(brokerName)))
                {
                    Vector brokerDependencies = findConfigDependenciesInternal(brokerName,copyKnown, true, false);
                    dependencies.addAll(brokerDependencies);
                    copyKnown.addAll(brokerDependencies);
                }
            }
        }
        if (DEBUG)
        {
            System.out.println("ConfigurationDependencies.findClusterDependencies returning vector of size " + dependencies.size());
        }
        return dependencies;
    }

    private Vector findAgentManagerDependencies(IDirElement AM, Vector knownAlready, boolean all, boolean onThisHost) throws Exception
    {
        Vector copyKnown = new Vector(knownAlready);
        Vector dependencies = new Vector();

         // find the am's container and its dependencies
        findContainerForComponent(AM.getIdentity().getName(), copyKnown, dependencies, all);

        return dependencies;
    }

    private Vector findDSDependencies(IDirElement DS, Vector knownAlready, boolean all, boolean onThisHost) throws Exception
    {
        Vector copyKnown = new Vector(knownAlready);
        Vector dependencies = new Vector();

        // find the container and its dependencies
        findContainerForComponent(DS.getIdentity().getName(), copyKnown, dependencies, all);

        // Because the user could have hints specifying a non default location for the components,
        // we must use the FromElementType query from 6.0 on

        // find authentication domains, policies and collections, other DS configs in this domain (find them only once)
        IAttributeSet topSet = DS.getAttributes();
        String ftRole = (String)topSet.getAttribute("FAULT_TOLERANCE_ROLE");
        if (ftRole == null || ftRole.equals("PRIMARY"))
        {
            // 9/8/05 modified this code to test that the configuration hasn't already been added to
            // the list of dependencies. We were getting duplicate domain, colllections, etc.
            // in the case where there is nore than
            // one DS and they're not marked "PRIMARY" or "BACKUP", for instance, two DS elements with null
            // for the FT role.
            // first of all, make sure all configs are no earlier than 6.0 or 2.1. We want a warning in the
            // install if we find anything old
            checkAllConfigVersions();
            Query domainsQuery = new Query();
            domainsQuery.setFrom(new FromElementType("MF_AUTHENTICATION_DOMAIN"));
            IDirElement[] domains = dsFileSystem.getFSElements(domainsQuery, false);
            for (int i = 0; i < domains.length; i++)
            {
                if (!copyKnown.contains(domains[i].getIdentity()))
                {
                    addDependency(dependencies, domains[i].getIdentity(), false);
                    copyKnown.add(domains[i].getIdentity());
                }
            }

            Query policiesQuery = new Query();
            policiesQuery.setFrom(new FromElementType("MQ_AUTHORIZATION_POLICY"));
            IDirElement[] policies = dsFileSystem.getFSElements(policiesQuery, false);
            for (int i = 0; i < policies.length; i++)
            {
                if (!copyKnown.contains(policies[i].getIdentity()))
                {
                    addDependency(dependencies, policies[i].getIdentity(), false);
                    copyKnown.add(policies[i].getIdentity());
                }
            }
            
            Query cStoresQuery = new Query();
            cStoresQuery.setFrom(new FromElementType("MQ_CERTIFICATES_STORE"));
            IDirElement[] stores = null;
            try
            {
                stores = dsFileSystem.getFSElements(cStoresQuery, false);
            }
            catch (Exception doesntExist) {} // A 6.1 system will not have the certificates store hints or directory
            if (stores != null)
            {
                for (int i = 0; i < stores.length; i++)
                {
                    if (!copyKnown.contains(stores[i].getIdentity()))
                    {
                        addDependency(dependencies, stores[i].getIdentity(), false);
                        copyKnown.add(stores[i].getIdentity());
                    }
                }
            }
            
            Query protocolsQuery = new Query();
            protocolsQuery.setFrom(new FromElementType("MQ_WS_PROTOCOL"));
            IDirElement[] protocols = null;
            try
            {
                protocols = dsFileSystem.getFSElements(protocolsQuery, false);
            }
            catch (Exception doesntExist) {}  // directory will not exist if no elements of the type have been added
                                              // hint will not exist if we're upgrading from 6.1
            if (protocols != null)
            {
                for (int i = 0; i < protocols.length; i++)
                {
                    if (!copyKnown.contains(protocols[i].getIdentity()))
                    {
                        addDependency(dependencies, protocols[i].getIdentity(), false);
                        copyKnown.add(protocols[i].getIdentity());
                    }
                }
            }

            Query compColQuery = new Query();
            compColQuery.setFrom(new FromElementType("MF_COMPONENT_COLLECTION"));
            IDirElement[] compCols = dsFileSystem.getFSElements(compColQuery, false);
            for (int i = 0; i < compCols.length; i++)
            {
                if (!copyKnown.contains(compCols[i].getIdentity()))
                {
                    addDependency(dependencies, compCols[i].getIdentity(), false);
                    copyKnown.add(compCols[i].getIdentity());
                }
            }

            Query containerColQuery = new Query();
            containerColQuery.setFrom(new FromElementType("MF_CONTAINER_COLLECTION"));
            IDirElement[] containerCols = dsFileSystem.getFSElements(containerColQuery, false);
            for (int i = 0; i < containerCols.length; i++)
            {
                if (!copyKnown.contains(containerCols[i].getIdentity()))
                {
                    addDependency(dependencies, containerCols[i].getIdentity(), false);
                    copyKnown.add(containerCols[i].getIdentity());
                }
            }

            // if we're finding just the dependencies in one container, don't look for
            // other agent managers and DS'
           
                findTypeAndDependencies("MF_AGENT_MANAGER", dependencies, copyKnown, all, !onThisHost );
                findTypeAndDependencies("MF_DIRECTORY_SERVICE", dependencies, copyKnown, all, !onThisHost);
                findTypeAndDependencies("MF_BACKUP_DIRECTORY_SERVICE", dependencies, copyKnown, all, !onThisHost);
                findTypeAndDependencies("MF_BACKUP_AGENT_MANAGER", dependencies, copyKnown, all, !onThisHost);            
            
            if (iwUpgrade)
            {
                // all MF containers will be upgraded then
                addAllContainersToDependencies(copyKnown, dependencies);
            }
        }

        return dependencies;
    }

    private void findTypeAndDependencies(String type, Vector dependencies, Vector copyKnown, boolean all, boolean onThisHost)
        throws Exception
    {
        Query query = new Query();
        query.setFrom(new FromElementType(type));
        IDirElement[] ams = dsFileSystem.getFSElements(query, false);
        for (int i = 0; i < ams.length; i++)
        {
            String name = ams[i].getIdentity().getName();
            if (!copyKnown.contains(ams[i].getIdentity())) // it might have already been added from the container
            // dependencies
            {
                Vector elDependencies = findConfigDependenciesInternal(name, copyKnown, all,onThisHost);
                dependencies.addAll(elDependencies);
                copyKnown.addAll(elDependencies);
            }
        }
    }
    Vector findDaemonDependencies(IDirElement daemon, Vector knownAlready, boolean all, boolean onThisHost) throws Exception
    {
        // if a daemon is upgraded, its container must be upgraded
        if (DEBUG)
        {
            System.out.println("ConfigurationDependencies.findDaemonDependencies starting " + daemon.getIdentity().getName());
        }
        ArrayList startsContainers = new ArrayList();
        Vector dependencies = new Vector();
        Vector copyKnown = new Vector(knownAlready);
        ArrayList<IElementIdentity> inContainers = findContainerForComponent(daemon.getIdentity().getName(), copyKnown, dependencies, all);
        // if the container for this AD is not being added to the list of dependencies (possibly because
        // it has been upgraded already), then don't try to add the containers it starts up
        if (dependencies.isEmpty())
        {
            if (DEBUG) 
            {
                System.out.println("ConfigurationDependencies.findDaemonDependencies found no dependencies for " + daemon.getIdentity().getName());
                System.out.println("ConfigurationDependencies.findDaemonDependencies printing inContainers");
                for (int i=0; i< inContainers.size(); i++)
                {
                    System.out.print(" " + ((IElementIdentity)inContainers.get(i)).getName());
                }
                System.out.println();
                System.out.println("ConfigurationDependencies.findDaemonDependencies printing copyKnown");
                for (int i=0; i< copyKnown.size(); i++)
                {
                    System.out.print(" " + ((IElementIdentity)copyKnown.get(i)).getName());
                }
                System.out.println();
             }
            // if (any)the container did not get added to the dependencies, and it wasn't known
            // already, don't look for any more dependencies
            if (!inContainers.isEmpty())
            {
                boolean known = false;
                for (IElementIdentity id : inContainers)
                {
                    if (copyKnown.contains(id))
                    {
                        known = true;
                    }
                }
                if (!known)
                {
                    return dependencies;
                }
            }
        }
        IAttributeSet containers = (IAttributeSet)daemon.getAttributes().getAttribute("CONTAINERS");
        HashMap containersMap = containers.getAttributes();

        Collection containersCollection = containersMap.values();
        Iterator containersIterator = containersCollection.iterator();
        while (containersIterator.hasNext())
        {
            IAttributeSet containerSet = (IAttributeSet)containersIterator.next();
            Reference containerRef = (Reference)containerSet.getAttribute("CONTAINER_REF");
            String containerName = containerRef.getElementName();
            IElementIdentity childID = dsFileSystem.getFSIdentity(containerName);
            if (DEBUG)
            {
                System.out.println("ConfigurationDependencies.findDaemonDependencies adding " + childID + " to startsContainers list");
            }
            startsContainers.add(childID);
            if (!copyKnown.contains(childID))
            {
                Vector containerDependencies = findConfigDependenciesInternal(containerName, copyKnown, all, onThisHost);
                copyKnown.addAll(containerDependencies);
                dependencies.addAll(containerDependencies);
            }
        }

        // containers started by a daemon are not root containers; their AD container is.
        // mark them as such.

        Iterator markContainers = startsContainers.iterator();
        while (markContainers.hasNext())
        {
            IElementIdentity childContainer = (IElementIdentity)markContainers.next();
            if (DEBUG)
            {
                System.out.println("ConfigurationDependencies.findDaemonDependencies setting " + childContainer.getName() + " to NOTROOT");
            }
            addSpecialMarker(childContainer, "NOTROOT");
            if (onThisHost)
            {
                addSpecialMarker(childContainer, "LOCAL");
            }
        }
        return dependencies;
    }
    
    // this method is called to put markers on the container in its children list.
    // The possible markers are "NOTROOT" for containers which are definitely not root
    // containers, and LOCAL, for containers which are on the currrent host.
    private void addSpecialMarker(IElementIdentity container, String specialChildName)
    {
    	if (DEBUG)
        {
            System.out.println("ConfigurationDependencies adding " + specialChildName + " to " + container.getName());
        }
    	Vector alreadySet = (Vector)m_containerComponents.get(container);
		if (alreadySet != null)
		{
			
			if (!alreadySet.contains(specialChildName))
            {
                alreadySet.add(specialChildName);
            }
			
		}
		else
		{
		    alreadySet = new Vector();
			alreadySet.add(specialChildName);
			m_containerComponents.put(container, alreadySet);
		}
    }

    //check the version before adding a configuration to the dependencies list
    //  XQ_CONTAINERs don't keep the same release versions as MQ/MF, so we use their
    // product version
    // Do not check if m_allowUpgraded is true.
    private void addDependency(Vector dependencies, IElementIdentity id, boolean onThisHost) throws Exception
    {
    	String releaseVersion = id.getReleaseVersion();
    	int versionInt = 0;
    	try
    	{
    		versionInt = (new Integer(releaseVersion)).intValue();
    	}
    	catch (NumberFormatException e) {} // ESB artifacts have release version 5.0
    	boolean allowIt = releaseVersion.equals(CURRENT_RELEASE_VERSION) && m_allowUpgraded && onThisHost;
        boolean okToAddESBContainer = id.getType().equals("XQ_CONTAINER") && !isESBContainerUpgraded(id);
        if (allowIt || SUPPORTED_PRODUCT_VERSIONS.containsValue(releaseVersion) ||okToAddESBContainer) // XQ_CONTAINERs don't keep the same release versions as MQ/MF
        {
            if (DEBUG)
            {
                System.out.println("ConfigurationDependencies.addDependency adding " + id.getName());
            }
            dependencies.add(id);
        }
    }
    
    private boolean isESBContainerUpgraded(IElementIdentity id) throws Exception
    {
        return m_utils.getProductVersion(id.getName()).equals(CURRENT_PRODUCT_VERSION);
    }


    private void checkAllConfigVersions() throws Exception
    {
       
        // Check the versions of the existing configurations in the DS
        // start with containers so the user gets that error message first, if a
        // container is of the wrong version. 
        Query query = new Query();
        query.setFrom(new FromElementType("MF_CONTAINER"));
        IDirElement[] fsContainers = dsFileSystem.getFSElements(query, false);
        for (IDirElement container : fsContainers)
        {
            IElementIdentity id = null;
            if ((id=container.getIdentity()).getReleaseVersion().compareTo(OLDEST_UPGRADABLE_VERSION) < 0)
            {
                throw new MigrationException("Cannot perform upgrade - found configurations of unsupported versions: " + id.getName() +
                                             " of version " + id.getReleaseVersion());
            }
        }
        // check other config types. Covers the case of configurations that are not in a
        // container
        ArrayList allTypes = new ArrayList();
        allTypes.addAll(ConfigUpgrade.MF_TYPES);
        // for now, we need to remove MF_HOST_MANAGER, because it did not exist in 7.5 and 7.6
        // and we cannot query by type for it.
        // put it back as soon as the lowest supported DM version for the upgrade is 8.0. At that
        // point, if you go through this method, the first check for MF_CONTAINER will catch DMs
        // of versions lower than 8.0
        allTypes.remove("MF_HOST_MANAGER");
        allTypes.addAll(ConfigUpgrade.MQ_TYPES);
        Iterator types = allTypes.iterator();
        while (types.hasNext())
        {
            String type = (String)types.next();
            if (!type.equals("MF_CONTAINER"))
            {
                query = new Query();
                query.setFrom(new FromElementType(type));
                IDirElement[] els = dsFileSystem.getFSElements(query, false);
                for (IDirElement el : els)
                {
                    IElementIdentity id = null;
                    if ((id=el.getIdentity()).getReleaseVersion().compareTo(OLDEST_UPGRADABLE_VERSION) < 0)
                    {
                        throw new MigrationException("Cannot perform upgrade - found configurations of unsupported versions: " + id.getName() +
                                                     " of version " + id.getReleaseVersion());
                    }
                } 
            }
        }       
    }
    
    private void checkOSrvrXSrvr(IAttributeSet services, IDirectoryAdminService dsAdmin) 
        throws DirectoryServiceException, OSrvrXSrvrException
    {
    	
    	    HashMap servicesMap = services.getAttributes();
            Collection servicesMaps = servicesMap.values();
            Iterator servicesMapsIt = servicesMaps.iterator();
            while (servicesMapsIt.hasNext())
            {
                IAttributeSet serviceSet = (IAttributeSet)servicesMapsIt.next();
                String serviceName = (String)serviceSet.getAttribute("service_ref");
                IDirElement serviceEl = dsAdmin.getElement("/xqServices/" + serviceName, false);
                // cover the case where the service instance has been deleted
                if (serviceEl != null)
                {
                    String serviceType = (String)serviceEl.getAttributes().getAttribute("type_ref");
                    if (serviceType.equals("XMLServiceType") || serviceType.equals("OrchestrationServiceType") || serviceType.equals("BPELServiceType"))
                    {
                        throw new OSrvrXSrvrException();
                    }
                }
            }
    	
    }
}