package com.sonicsw.mf.common.config.upgrade;

import java.io.File;
//import java.lang.reflect.Method;
import java.io.FileInputStream;
import java.util.Hashtable;

import javax.management.Notification;
import javax.management.NotificationListener;
import javax.management.ObjectName;

import com.sonicsw.mf.comm.InvokeTimeoutException;
import com.sonicsw.mf.common.IDirectoryAdminService;
import com.sonicsw.mf.common.IDirectoryFileSystemService;
import com.sonicsw.mf.common.MFRuntimeException;
import com.sonicsw.mf.common.config.IAttributeSet;
import com.sonicsw.mf.common.dirconfig.ElementFactory;
import com.sonicsw.mf.common.dirconfig.IDirElement;
import com.sonicsw.mf.common.runtime.INotification;
import com.sonicsw.mf.framework.directory.DirectoryServiceFactory;
import com.sonicsw.mf.framework.directory.IContainerlessDS;
import com.sonicsw.mf.framework.directory.IDirectoryService;
import com.sonicsw.mf.framework.directory.IFSStorage;
import com.sonicsw.mf.jmx.client.DirectoryServiceProxy;
import com.sonicsw.mf.jmx.client.JMSConnectorAddress;
import com.sonicsw.mf.jmx.client.JMSConnectorClient;
import com.sonicsw.mf.mgmtapi.config.constants.IContainerConstants;
import com.sonicsw.mf.mgmtapi.runtime.ProxyRuntimeException;

/**
 * <p>Title: </p>
 * <p>Description: </p>
 * <p>Copyright: Copyright (c) 2004</p>
 * <p>Company: Sonic Software</p>
 * @author Mari Davila
 * @version 1.0
 */

/**
 * This is the class used internally by the BrokerInquiry, 
 * ConfigurationDependencies and ConfigUpgradeDriver classes to connect
 * to the directory service. It has the logic to figure out where the 
 * directory service storage is located, if connecting directly for 
 * a domain manager upgrade. If connecting during a domain manager 
 * upgrade, it also has logic to connect to the source or the 
 * destination directory service, depending on the upgrade step being
 * performed. During a domain manager upgrade, BrokerInquiry and 
 * ConfigurationDependencies connect to the source install DS and 
 * ConfigUpgradeDriver connects to the destination directory after the
 * SUI has copied the DS to the destination directory. In the case where
 * the directory service storage is not in the default install location, 
 * the direct connection is always made to the source location, because
 * the storage is not copied. In this case, the directory service is
 * upgraded "in place".
*/

public class DirectoryService implements IMigrationProcess
{
    IDirectoryAdminService m_dsAdmin = null;
    IDirectoryFileSystemService m_dsFileSystem = null;
    IDirectoryService m_ds = null;
    JMSConnectorClient m_connector = null;
    static boolean DEBUG = System.getProperty("DebugAll.debug", "false").equals("true");
    
    public DirectoryService(IDirElement dsElement, String absParent, String absHostDir, boolean newPSE) throws Exception
    {
    	openStorage(dsElement, absParent, absHostDir, newPSE);
    }

    public DirectoryService(String dsXmlFile) throws Exception
    {      
        this(dsXmlFile, null, null, false);
    }
    
    public DirectoryService(String dsXmlFile, boolean copyStorage) throws Exception
    {
    	if (copyStorage)
    	{
    		IDirElement dsElement = getDSElement(dsXmlFile);
    		IAttributeSet dsAttributes = dsElement.getAttributes();

            String domainName = (String) dsAttributes.getAttribute("DOMAIN_NAME");
            if (domainName == null || domainName.length() == 0)
            {
                domainName = IContainerConstants.DOMAIN_NAME_DEFAULT;
            }

            String directory = null;
            
            String dirFromDsXml = (String) dsAttributes.getAttribute("HOST_DIRECTORY");
            directory = getDomainDirectory(dirFromDsXml, null, null);
            File copyFolderFrom = new File(directory, domainName);
            if (!copyFolderFrom.exists())
            {
                // If the path in the ds.xml is a relative path, try using the 
                // parent directory of the ds.xml as "."
                if (!copyFolderFrom.isAbsolute())
                {
                    File dsXmlParentDir = (new File(dsXmlFile)).getParentFile();
                    copyFolderFrom = new File(dsXmlParentDir, copyFolderFrom.getPath());
                    if (!copyFolderFrom.exists())
                    {
                        throw new Exception("Unable to find " + copyFolderFrom.getCanonicalPath() + " to copy it before opening it. Make sure the HOST_DIRECTORY attribute is set to an absolute file path on this host.");
                    }
                }
            }
            String sonicHome = System.getProperty("sonic.home");
            File copyTo = new File(sonicHome, MIGRATIONTEMPDIR);
            File copyFolderTo = new File(copyTo, domainName);
            if (copyFolderTo.exists())
            {
                Utils.recursiveDeleteDirectory(copyFolderTo);
            }
            copyFolderTo.mkdirs();
            if (!copyFolderTo.exists())
            {
                throw new Exception("Unable to make directory " + copyFolderTo + " to copy the DS storage");
            }
            Utils.copyAll(copyFolderFrom, copyFolderTo);
            openStorage(dsElement, null, copyTo.getCanonicalPath(), false);
    	}
    }

    public DirectoryService(String dsXmlFile, String absParent, String absHostDir, boolean newPSE) throws Exception
    {
        openStorage(getDSElement(dsXmlFile), absParent, absHostDir, newPSE);
    }
    
    public DirectoryService(String url, String username, String password, String domain, String mnode, int rt) throws Exception
    {
    	if (DEBUG)
        {
            System.out.println("DirectoryService constructor for remote connection to URLs " + url);
        }
        m_connector = new JMSConnectorClient();
        Hashtable env = new Hashtable();
        env.put("ConnectionURLs", url);
        env.put("DefaultUser", username);
        if (password != null)
        {
            env.put("DefaultPassword", password);
        }
        m_connector.setRequestTimeout(rt * 1000);
        JMSConnectorAddress address = new JMSConnectorAddress(env);
        
        if (DEBUG)
        {
            System.out.println("DirectoryService constructor, mnode == " + mnode);
        }
        if (mnode != null)
        {
            address.setManagementNode(mnode);
        }
        String connectString = m_connector.connect(address, (long)40000);
        if (DEBUG)
        {
            System.out.println("DirectoryService constructor, connect result == " + connectString);
        }
        String dsName = domain + ".DIRECTORY SERVICE:ID=DIRECTORY SERVICE";
        DirectoryServiceProxy proxy = new DirectoryServiceProxy(m_connector, new ObjectName(dsName));
        m_dsFileSystem = (IDirectoryFileSystemService)proxy;
        if (DEBUG)
        {
            System.out.println("DirectoryService constructor, testing domain name with m_dsFileSystem.getDirectoryServiceVersion");
        }
        // if we have gotten this far without an exception, the host, port, user and password
        // were correct. The following tests that the domain name is correct.
        try
        {
        	 m_dsFileSystem.getDirectoryServiceVersion();
        }
        catch (ProxyRuntimeException proxyE)
        {
        	m_connector.disconnect();
        	Throwable cause = proxyE.getCause();
        	if (cause instanceof InvokeTimeoutException)
        	{
        		MigrationException migrationE = new MigrationException("The Directory Service could not be reached. Check that you have used the correct domain name.");
        		migrationE.initCause(proxyE);
        		throw migrationE;
        	}
            else
            {
                throw proxyE;
            }
        }
        m_dsAdmin = (IDirectoryAdminService) proxy;
        if (DEBUG)
        {
            System.out.println("Finished with remote connection");
        }
    }

    public DirectoryService(String url, String username, String password, String domain, String mnode) throws Exception
    {
    	this(url, username, password, domain, mnode, REQUEST_TIMEOUT_DEFAULT);
    }

    public DirectoryService(String url, String username, String password, String domain) throws Exception
    {
        this(url, username, password, domain, null);
    }

    private void openStorage(IDirElement dsElement, String absParent, String absHostDir, boolean newPSE)
        throws Exception
    {
    	String storageType = IDirectoryService.PSE_STORAGE;
    	  
    	IAttributeSet dsAttributes = dsElement.getAttributes();
        if (DEBUG)
        {
            System.out.println("upgrade.DirectoryService constructor for DS Element " + dsElement.getIdentity().getName() + " absParent = " + absParent +
                                        ", absHostDir = " + absHostDir + ", working directory == " + (new File(".")).getCanonicalPath());
        }

        String domainName = (String) dsAttributes.getAttribute("DOMAIN_NAME");
        
        if (domainName == null || domainName.length() == 0)
        {
            domainName = IContainerConstants.DOMAIN_NAME_DEFAULT;
        }

        String directory = null;
        
        String dirFromDsXml = (String) dsAttributes.getAttribute("HOST_DIRECTORY");
        directory = getDomainDirectory(dirFromDsXml, absParent, absHostDir);
        if (!newPSE && dsElement.getIdentity().getReleaseVersion().equals("101"))
        {
            storageType = IDirectoryService.FS_STORAGE;
        }

        Object tmp = dsAttributes.getAttribute("FILE_SYSTEM_STORAGE");        
        if (tmp == null)
        {
            throw new MFRuntimeException("Bad Directory Service Configuration - must contain FILE_SYSTEM_STORAGE.");
        }
        String password = (String) ( (IAttributeSet) tmp).getAttribute("PASSWORD");
        
        if (! (tmp instanceof IAttributeSet))
        {
            throw new MFRuntimeException("Bad Directory Service Configuration - FILE_SYSTEM_STORAGE must be an attribute set.");
        }
        
        // Creates the directory service object
        Hashtable directoryEnv = new Hashtable();
        directoryEnv.put(IDirectoryService.STORAGE_TYPE_ATTRIBUTE, storageType);
        if (DEBUG)
        {
            System.out.println("DirectoryService constructor connecting to storage in " + directory);
        }
        directoryEnv.put(IFSStorage.HOST_DIRECTORY_ATTRIBUTE, directory);
        if ( (password != null) && (password.length() != 0))
        {
            directoryEnv.put(IFSStorage.PASSWORD_ATTRIBUTE, password);
        }
        DirectoryServiceFactory factory = new DirectoryServiceFactory(directoryEnv);
        m_ds = factory.createDirectoryService(domainName);
        m_dsFileSystem = (IDirectoryFileSystemService) m_ds;
        m_dsAdmin = (IDirectoryAdminService) m_ds;
        if (DEBUG)
        {
            System.out.println("upgrade.DirectoryService constructor end");
        } 
    }

    public IDirectoryFileSystemService getFileSystemService()
    {
        return m_dsFileSystem;
    }

    public IDirectoryAdminService getAdminService()
    {
        return m_dsAdmin;
    }
    
    public IDirectoryService getDirectoryService()
    {
        return m_ds;
    }

    public void closeDS() throws Exception
    {
        // if there's a connector, the DS connection is by proxy
        if (m_connector != null)
        {
        	System.out.println("DirectoryService closing the connector");
            m_connector.disconnect();
            m_connector = null;
        }
        // otherwise, it's a direct connection
        else
        {
            ((IContainerlessDS)m_dsAdmin).close();
        }

       /* {
        	try
        	{
        	    Class directDSClass = Class.forName("com.sonicsw.mf.directory.IContainerlessDS");
        	    if (directDSClass != null)
        	    {
        	    	Method closeMethod = directDSClass.getMethod("close", new Class[]{});
        	    	closeMethod.invoke(m_dsAdmin, new String[]{});
        	    }
        	}
        	catch (NoClassDefFoundError notFound) {}
        }*/
    }
    
    public static String getDomainDirectory(String host_directory, String absParent, String absHostDir)
    throws Exception
    {
    	if (absHostDir != null)
        {
            return absHostDir;
        }
        else if (host_directory == null || host_directory.length() == 0)
        {
            host_directory = ".";
        }
    	File hostDirFile = new File(host_directory);
    	if (!hostDirFile.isAbsolute() && absParent != null)
        {
            return (new File(absParent, hostDirFile.getName())).getCanonicalPath();
        }
        else
        {
            return host_directory;
        }
    }
    
    static IDirElement getDSElement(String dsXml) throws Exception
    {
    	String xmlString = getXMLStringFromFile(dsXml);
        IDirElement dsConfig = ElementFactory.importElementFromXML(xmlString, null, "MF_DIRECTORY_SERVICE");
        return dsConfig;
    }
    
    static String getDSVersion(String dsXml) throws Exception
    {   	
    	String xmlString = getXMLStringFromFile(dsXml);
        IDirElement dsConfig = ElementFactory.importElementFromXML(xmlString, null, "MF_DIRECTORY_SERVICE");
        if (dsConfig == null)
        {
            dsConfig = ElementFactory.importElementFromXML(xmlString, null, "MF_BACKUP_DIRECTORY_SERVICE");
        }
        return dsConfig.getIdentity().getReleaseVersion();
    }
    
    static String getHostDirectory(String dsXml) throws Exception
    {   	
    	String xmlString = getXMLStringFromFile(dsXml);
        IDirElement dsConfig = ElementFactory.importElementFromXML(xmlString, null, "MF_DIRECTORY_SERVICE");
        if (dsConfig == null)
        {
            dsConfig = ElementFactory.importElementFromXML(xmlString, null, "MF_BACKUP_DIRECTORY_SERVICE");
        }
        IAttributeSet topSet = dsConfig.getAttributes();
        return (String)topSet.getAttribute("HOST_DIRECTORY");
    }
    
    // convenience method for the upgrade who is shutting down containers
    // It will try to shutdown the container for "howLong" ms
    public boolean shutdownContainer(String containerObjectName, long howLong) throws Exception
    {
    	if (DEBUG)
        {
            System.out.println("DirectoryService.shutdownContainer Invoking shutdown for " + containerObjectName);
        }
    	ObjectName containerObject = new ObjectName(containerObjectName);
    	m_connector.invoke(containerObject, "shutdown", new Object[]{}, new String[]{});
    	long originalTime = System.currentTimeMillis();
    	boolean shutdown = !pingContainer(containerObjectName);
    	long counter = System.currentTimeMillis() - originalTime;;
    	while (!shutdown && counter < 180000)
    	{
    		if (DEBUG)
            {
                System.out.println("DirectoryService.shutdownContainer Waiting for the container to completely shutdown");
            }
    		Thread.sleep(5000);
    		shutdown = !pingContainer(containerObjectName);
    		counter = System.currentTimeMillis() - originalTime;
    	}
    	if (!shutdown)
        {
            return false;
        }
    	// if we're shutting down the domain manager, close the connection
    	// in the case of the upgrade, we only shutdown the DM when we're
    	// connected to it.
    	if (containerObjectName.indexOf("DIRECTORY SERVICE") > -1)
        {
            closeDS();
        }
    	return true;
    }
    
    // convenience method for the upgrade who is retarting the backup DM after setting
    // backup failover read only mode
    public void restartContainer(String containerObjectName, String AMObjectName) throws Exception
    {
    	if (DEBUG)
        {
            System.out.println("Invoking restart for " + containerObjectName);
        }
    	ObjectName containerObject = new ObjectName(containerObjectName);
    	StartupListener listener = subscribeToStartup(containerObjectName, AMObjectName);
    	m_connector.invoke(containerObject, "restart", new Object[]{}, new String[]{});
    	while (!listener.isStarted())
    	{
    		if (DEBUG)
            {
                System.out.println("restartContainer waiting for notification .....");
            }
    		Thread.sleep(3000);  
    	}
    }
    
    public boolean pingContainer(String containerObjectName) throws Exception
    {
    	System.out.println("Pinging container " + containerObjectName + "...");
    	// if there's no connector to the DM, we can't tell if the container 
    	// is running
    	if (m_connector == null)
    	{
    		System.out.println("Pinging container " + containerObjectName + "...done, because the DM is not running");
    		return false;
    	}
    	String pingString = "Are you there";
    	ObjectName containerObject = new ObjectName(containerObjectName);
    	try
    	{
    	    String result = (String)m_connector.invoke(containerObject, "ping", new Object[]{pingString}, new String[]{String.class.getCanonicalName()});     	    
    	    if (result.equals(pingString))
    	    {
    	    	System.out.println("Pinging container " + containerObjectName + "...done; container is running");
    	    	return true;
    	    }
    	}
    	catch (Exception e)
    	{
    		System.out.println("Pinging container " + containerObjectName + "...done; container is not running");
    		return false;
    	}
    	System.out.println("Pinging container " + containerObjectName + "...done; container is not running");
    	return false;
    }
    
    public static IAttributeSet getDSAttributes(String dsXmlFileName)
    throws Exception
    {
        String xmlString = getXMLStringFromFile(dsXmlFileName);
        IDirElement dsConfig = ElementFactory.importElementFromXML(xmlString, null, "MF_DIRECTORY_SERVICE");
        if (dsConfig == null)
        {
            dsConfig = ElementFactory.importElementFromXML(xmlString, null, "MF_BACKUP_DIRECTORY_SERVICE");
        }

        IAttributeSet dsAttributes = dsConfig.getAttributes();
        return dsAttributes;
    }
    
    public static String getXMLStringFromFile(String xmlFile)
    throws Exception
    {
        FileInputStream fis = new FileInputStream(xmlFile);
        byte[] bytes = new byte[fis.available()];
        fis.read(bytes);
        fis.close();

        return new String(bytes);
    }
    
    public StartupListener subscribeToStartup(String containerOName, String AMObjectName) throws Exception
    {
        // get the filter for this context
        javax.management.NotificationFilterSupport filter = new javax.management.NotificationFilterSupport();
        filter.enableType("system.state.Startup");
        // subscribe to the notifications
        ObjectName name = new ObjectName(AMObjectName);
        StartupListener listener = new StartupListener(containerOName);
        m_connector.addNotificationListener(name, listener, filter, "");
        if (DEBUG)
        {
            System.out.println("subscribeToSTartup added notification listener on object " + name.getCanonicalName());
        }
        return listener;
    }
    
    private final static class StartupListener
    implements NotificationListener
    {
    	private String m_containerName;
    	private boolean m_started = false;
    	
    	public StartupListener(String containerOName)
    	{
    		if (DEBUG)
            {
                System.out.println("StartupListener will be looking for container " + containerOName + " in the notification");
            }
    		m_containerName = containerOName;
    	}
        @Override
        public void handleNotification(Notification notification, Object handback)
        {
        	if (DEBUG)
            {
                System.out.println("StartupListener.handleNotification called");
            }
            // attributes
            if (notification.getType().equals("system.state.Startup"))//getAttributes().entrySet().toArray();
            {
            	if (DEBUG)
                {
                    System.out.println("StartupListener.handleNotification got system.state.Startup");
                }
            	String containerName = (String)((INotification)notification).getAttributes().get("Container");
            	if (DEBUG)
                {
                    System.out.println("StartupListener.handleNotification containerName == " + containerName + " will compare to " + m_containerName);
                }
            	if (m_containerName.startsWith(containerName))
                {
                    m_started = true;
                }
            }
        }
        
        public boolean isStarted()
        {
        	return m_started;
        }
    }
}