package com.sonicsw.mf.common.config.upgrade;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.PrintStream;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Properties;
import java.util.Set;
import java.util.Vector;

import javax.management.JMRuntimeException;

import com.sonicsw.mf.comm.ConnectTimeoutException;
import com.sonicsw.mf.comm.InvokeTimeoutException;
import com.sonicsw.mf.common.IDirectoryFileSystemService;
import com.sonicsw.mf.common.MFSecurityException;
import com.sonicsw.mf.common.config.IAttributeSet;
import com.sonicsw.mf.common.config.IElementIdentity;
import com.sonicsw.mf.common.dirconfig.DirectoryServiceException;
import com.sonicsw.mf.common.dirconfig.IDirElement;
import com.sonicsw.mf.framework.agent.ContainerSetup;
import com.sonicsw.mf.mgmtapi.config.constants.IContainerConstants;
import com.sonicsw.mf.mgmtapi.runtime.ProxyRuntimeException;

public class FindDependencies implements IMigrationProcess
{
	
    private String m_outFile = "." + System.getProperty("file.separator")
	    + "upgrade.properties"; // default file;
    private static String UPGRADEPROPS_ERROR = "upgradeProps error";
    private FileOutputStream m_outStream = null;
    private String m_url = null;
    private String m_username = null;
    private String m_password = null;
    private String m_node = null;
    private String m_domain = DEFAULTDOMAIN;
    private static final String SONIC_HOME = System.getProperty("sonic.home");
    private boolean m_WBInstalled = false;
    private int m_requestTimeout=REQUEST_TIMEOUT_DEFAULT;
   
    private Vector m_rootIDs = null;
    private Vector m_upgradeIDs = null;
    private Vector m_onThisHost = null;
    private Vector m_templateList = null;
    private ConfigurationDependencies m_depends = null;
    private IDirectoryFileSystemService m_ds;
    private HashMap m_dependsMap = new HashMap();
    
    private void printUsage(String errorMessage)
    {
    	if (errorMessage != null)
    	{
    		System.out.println();
    	    System.out.println("  error: " + errorMessage);
    	    System.out.println();
    	}
    	System.out.println();
    	System.out.println("*****************************  upgradeProps usage ************************");
    	System.out.println();
    	System.out.println("Arguments to upgradeProps:");
    	System.out.println();
    	System.out.println("upgradeProps -url <connection url> -u <mgmt user> [-p <mgmt password>] -d <domain name> [-rt <request timeout>] <container name>");
        System.out.println();
        System.out.println("where:");
        System.out.println();
        System.out.println("<connection URL> is the url for a management connection, for instance, pcwork:2506");
        System.out.println("<mgmt user> is the user with Administrator privileges for the mgmt connection, for instance, Administrator");
        System.out.println("<mgmt password> is the password, if required, for the mgmt connection");
        System.out.println("<domain name> is the name of the Sonic domain, for instance, Domain1");
        System.out.println("<request timeout> is the maximum amount of time that upgradeProps will wait when invoking methods of the directory service. It defaults to " + REQUEST_TIMEOUT_DEFAULT);
        System.out.println("<container name> is the name of the container configuration to be upgraded, for instance, /Containers/DomainManager");
        System.out.println();
        System.out.println("<connection URL>, <mgmt user>, <domain name> and <container name> are required arguments; <mgmt password> is required only if the mgmt broker has security enabled");
        System.out.println();
        System.out.println("**************************************************************************");
        System.exit(0);
    }
	/**
	 * @param args
	 * @throws Exception 
	 */
	public static void main(String[] args) 
	{
	    FindDependencies fd = new FindDependencies();
	    fd.parseParams(args);
	}
	
	private void parseParams(String[] args)
	{
	    int argIndex = 0;
        String configName = null;
        String rt = null;
        while (argIndex < args.length) 
        {
            String arg = args[argIndex];
            if ("-url".equalsIgnoreCase(arg))
            {
                m_url = args[++argIndex];
            }
            else if ("-u".equalsIgnoreCase(arg))
            {
                m_username = args[++argIndex];
            }
            else if ("-p".equalsIgnoreCase(arg))
            {
                m_password = args[++argIndex];
            }
            else if ("-d".equalsIgnoreCase(arg))
            {
                m_domain = args[++argIndex];
            }
            else if ("-n".equalsIgnoreCase(arg))
            {
                m_node = args[++argIndex];
            }
            else if ("-f".equalsIgnoreCase(arg))
            {
                m_outFile = args[++argIndex];
            }
            else if ("-rt".equalsIgnoreCase(arg))
            {
                rt=args[++argIndex];
            }
            else if (arg.toLowerCase().startsWith("-h"))
            {
                printUsage(null);
            }
            else
            {
                configName = arg;
            }
        
            argIndex = argIndex + 1;
        }
        if (m_url == null)
        {
            printUsage("upgradeProps requires a connection url.");
        }
        if (m_username == null)
        {
            printUsage("upgradeProps requires a user name for the mgmt connection.");
        }
        if (m_domain == null)
        {
            printUsage("upgradeProps requires a domain name.");
        }
        if (configName == null)
        {
            printUsage("You must provide a container configuration name to upgradeProps.");
        }
        if (rt != null)
        {
            try
            {
                m_requestTimeout = (new Integer(rt)).intValue();
            }
            catch (NumberFormatException noNum)
            {
                printUsage("Request timeout specified is not a number: " + rt);
            }
        }
        
        try
        {
            // load sonicsw.properties to see if this is a WB install
            File propFile = new File(SONIC_HOME, "sonicsw.properties");
            if (propFile.exists())
            {
                FileInputStream propsIn = new FileInputStream(propFile);
                Properties installProps = new Properties();
                installProps.load(propsIn);
                propsIn.close();
                String wbInstalled = installProps.getProperty("sonic.wb.installed");
                m_WBInstalled = wbInstalled != null && wbInstalled.equals("true");
            }

            Vector configIDs = findDependencies(m_url, m_domain, m_username, m_password, m_node,configName);
            writeProperties(configIDs, configName);
            //testPropFile();
        }
        catch (MigrationException migE)
        {
            Utils.printError(UPGRADEPROPS_ERROR, null, migE.getMessage());
        }
        catch (ProxyRuntimeException proxyE)
        {
            Throwable cause = proxyE.getCause();
            if (cause != null && (cause instanceof InvokeTimeoutException))
            {
                Utils.printError(UPGRADEPROPS_ERROR, null, "upgradeProps had a problem communicating with the domain manager." + 
                        " You might want to increase the request timeout parameter -rt <requestTimeout>. The default is " +  REQUEST_TIMEOUT_DEFAULT + " seconds.");
            }
            else
            {
                Utils.printError(UPGRADEPROPS_ERROR, proxyE, null);
            }
        }
        catch (JMRuntimeException jmsE)
        {
            Throwable cause = jmsE.getCause();
            if (cause != null && (cause instanceof ConnectTimeoutException))
            {
                Utils.printError(UPGRADEPROPS_ERROR, null, "upgradeProps had a problem connecting to the domain manager. Check that your connection information is correct, and make sure that your domain manager is running.");
            }
            else if (cause != null && (cause instanceof MFSecurityException))
            {
                Utils.printError(UPGRADEPROPS_ERROR, null, "The connection attempt threw a security exception. Check that the specified username and password are correct.");
            }
            else
            {
                Utils.printError(UPGRADEPROPS_ERROR, jmsE, null);
            }
        }
        catch (Exception e)
        {
            Utils.printError(UPGRADEPROPS_ERROR, e, null);
        }
        finally
        {
            try
            {
                closeConnection();
            }
            catch (Exception e)
            {
                // well, we tried to recover but couldn't
                Utils.printError(UPGRADEPROPS_ERROR, e, "Could not close the connection to the domain");
            }
            System.exit(0);
        }
	}
	
	private void testPropFile() throws IOException
	{
		Properties props = new Properties();
		FileInputStream in = new FileInputStream(m_outFile);
		props.load(in);
		in.close();
		FileOutputStream testout = new FileOutputStream("testprops.out");
		PrintStream testStream = new PrintStream(testout);
		props.list(testStream);
		testout.close();
	}
	
	public Vector findDependencies(String url, String domainName, String username, 
            String password, String mgmtNode, String containerID) throws Exception
    {
		System.out.println("upgradeProps connecting to the Directory Service...");
		m_depends = new ConfigurationDependencies(url, domainName, username, password, mgmtNode, m_requestTimeout);
		System.out.println("upgradeProps connecting to the Directory Service...done");
		m_ds = m_depends.getDS();
		if (url.toLowerCase().endsWith("xml")) // if a direct connection was used
		{
			IAttributeSet xmlSet = DirectoryService.getDSAttributes(url);
			if (m_domain == null)
            {
                m_domain = (String)xmlSet.getAttribute("DOMAIN_NAME");
            }
		}
		
		System.out.println("upgradeProps finding upgrade dependencies...");
		m_dependsMap = internalFindDependencies(containerID);
		System.out.println("upgradeProps finding upgrade dependencies...done");
		Set keySet = m_dependsMap.keySet();
		return new Vector(keySet);
    }
	
	private HashMap internalFindDependencies(String containerID) throws Exception
	{
		if (m_WBInstalled)
        {
            return m_depends.findAllContainers(containerID);
        }
        else
        {
            return m_depends.findConfigDependencies(containerID);
        }
	}
	
	private void closeConnection() throws Exception
	{
		if (m_depends != null)
        {
            m_depends.closeDS();
        }
		if (m_url != null && m_url.toLowerCase().endsWith("xml"))
		{
			File tempStorage = new File(SONIC_HOME, MIGRATIONTEMPDIR);
			Utils.recursiveDeleteDirectory(tempStorage);
		}
			
	}
	
	private void writeProperties(Vector configIDs, String configID) throws IOException, DirectoryServiceException
	{
		try 
		{
			m_outStream = new FileOutputStream(m_outFile);
		} 
		catch (FileNotFoundException notFound) 
		{
			notFound.printStackTrace();
			System.exit(0);
		} // bizarre!!!!!
		System.out.println("upgradeProps writing out properties file " + ((new File(m_outFile)).getCanonicalPath()) + "...");
		m_rootIDs = new Vector();
		m_upgradeIDs = new Vector();
		m_onThisHost = new Vector();
		m_templateList = new Vector();
		//writeIntro();
		// write out the debug flag first!!!
		writeDebugFlag();
		writeConnectionProperties();
		writeControlCodes();
		writeUpgradeList(configIDs, configID);
		writeContainerProperties();
		m_outStream.close();
		System.out.println("upgradeProps writing out properties file " + ((new File(m_outFile)).getCanonicalPath()) + "...done");
	}
	
	private void writeIntro() throws IOException
	{
		m_outStream.write(("!*****************************************************************").getBytes());
		m_outStream.write(NEWLINEBYTES);
		m_outStream.write(NEWLINEBYTES);
		m_outStream.write(("!NOTE: Use Unix slashes '/' or double backslashes \"\\\\\" in").getBytes());
		m_outStream.write(NEWLINEBYTES);
		m_outStream.write(("!in path values on Windows. For instance, c:/Sonic/MQ" + CURRENT_PRODUCT_VERSION+ "/ds.xml").getBytes());
		m_outStream.write(NEWLINEBYTES);
		m_outStream.write(("!and not c:\\Sonic\\MQ" + CURRENT_PRODUCT_VERSION+ "\\ds.xml").getBytes());
		m_outStream.write(NEWLINEBYTES);
		m_outStream.write(NEWLINEBYTES);
		m_outStream.write(("!*****************************************************************").getBytes());
	}
	
	private void writeDebugFlag() throws IOException
	{
		m_outStream.write(NEWLINEBYTES);
		m_outStream.write(("!***************  debug flag - set to true to enable debugging output ").getBytes());
	    m_outStream.write(NEWLINEBYTES);
	    m_outStream.write(NEWLINEBYTES);
	    m_outStream.write((DEBUGPROP + "=false").getBytes());
	    m_outStream.write(NEWLINEBYTES);
	}
	
	private void writeConnectionProperties() throws IOException
	{
		String urlConverted = m_url;
		boolean xmlURL = urlConverted.toLowerCase().endsWith("xml");
		if (xmlURL)
        {
            urlConverted = Utils.replaceBackSlashes(urlConverted);
        }
		m_outStream.write(NEWLINEBYTES);
		//m_outStream.write(("!***************  ds boot file for existing domain manager must be specified for a domain manager upgrade ").getBytes());
	   // m_outStream.write(NEWLINEBYTES);
	    //m_outStream.write(NEWLINEBYTES);
	    //m_outStream.write((DSBOOTFILEPROP + "=").getBytes());
	    //if (xmlURL)
	    	//m_outStream.write(urlConverted.getBytes());
	    //m_outStream.write(NEWLINEBYTES);
	    
		m_outStream.write(NEWLINEBYTES);
		m_outStream.write(("!***************  Tool connection properties").getBytes());
		m_outStream.write(NEWLINEBYTES);
		m_outStream.write(NEWLINEBYTES);
		m_outStream.write((URLPROP + "=" + urlConverted).getBytes());
		m_outStream.write(NEWLINEBYTES);
		m_outStream.write((USERNAMEPROP + "=" + (m_username == null ? DEFAULTUSER : m_username)).getBytes());
		m_outStream.write(NEWLINEBYTES);
		m_outStream.write((PASSWORDPROP + "=" + (m_password == null ? "" : m_password)).getBytes());
		m_outStream.write(NEWLINEBYTES);
		m_outStream.write((DOMAINPROP + "=" + (m_domain == null  ? DEFAULTDOMAIN : m_domain)).getBytes());
		m_outStream.write(NEWLINEBYTES);
		m_outStream.write((NODEPROP + "=" + (m_node == null ? "" : m_node)).getBytes());
		m_outStream.write(NEWLINEBYTES);
		m_outStream.write((RTPROP + "=" + m_requestTimeout).getBytes());
		m_outStream.write(NEWLINEBYTES);
		m_outStream.write((CONNPROPSPROP + "=").getBytes());
		m_outStream.write(NEWLINEBYTES);
		
	}
	/*private static String BROKERCNPROP = "broker.controlNumber";
    private static String LOGGERCNPROP = "event.monitor.controlNumber";
    private static String ESBCNPROP = "esb.controlNumber";
    private static String RDBMSCNPROP = "rdbms.controlNumber";
    private static String WBCNPROP = "workbench.controlNumber";
    private static String ROOTLISTPROP = "root.list";
	*/
	private void writeControlCodes() throws IOException
	{
		// if we're on the domain manager host, write out the control codes
		// we know about
		File  mqHome = new File(SONIC_HOME + FILESEPARATOR + "MQ" + CURRENT_PRODUCT_VERSION);
		String mqControlCode = null;
		if (mqHome.exists())
		{
			File mqPropsFile = new File(mqHome, "product.properties");
			if (mqPropsFile.exists())
			{
			    FileInputStream mqPropsStream = new FileInputStream(mqPropsFile);
			    Properties mqProps = new Properties();
			    mqProps.load(mqPropsStream);
			    mqPropsStream.close();
			    mqControlCode = mqProps.getProperty(BROKERCNPROP);
			}
		}
		File  esbHome = new File(SONIC_HOME + FILESEPARATOR + "ESB" + CURRENT_PRODUCT_VERSION);
		String esbControlCode = null;
		if (esbHome.exists())
		{
			File esbPropsFile = new File(esbHome, "product.properties");
			if (esbPropsFile.exists())
			{
			    FileInputStream esbPropsStream = new FileInputStream(esbPropsFile);
			    Properties esbProps = new Properties();
			    esbProps.load(esbPropsStream);
			    esbPropsStream.close();
			    esbControlCode = esbProps.getProperty(ESBCNPROP);
			}
		}
		
		m_outStream.write(NEWLINEBYTES);
		m_outStream.write(("!***************    Control Numbers ").getBytes());
		m_outStream.write(NEWLINEBYTES);
		m_outStream.write(NEWLINEBYTES);
		m_outStream.write((BROKERCNPROP + "=" + (mqControlCode == null ? "" : mqControlCode)).getBytes());
		m_outStream.write(NEWLINEBYTES);
		m_outStream.write((ESBCNPROP + "=" + (esbControlCode == null ? "" : esbControlCode)).getBytes() );
		m_outStream.write(NEWLINEBYTES);
		
	}
	
	private void writeUpgradeList(Vector configIDs, String configID) throws IOException
	{
		// get the containerIDs and figure out what's root and what's on this host
		if (!configIDs.isEmpty()) 
	    {
			Iterator idsIterator = configIDs.iterator();
			while (idsIterator.hasNext()) 
			{
				IElementIdentity id = (IElementIdentity) idsIterator.next();
				if (id.getType().equals(IContainerConstants.DS_TYPE))
				{
					boolean isRoot = true;
					boolean onThisHost = false;
					Vector components = (Vector)m_dependsMap.get(id);
					if (components != null)
					{
					    // always include the original container in the list of
					    // root containers
						if (!id.getName().equals(configID) && (components.contains("NOTROOT") || !components.contains("LOCAL")))
                        {
                            isRoot = false;
                        }
						if (components.contains("LOCAL"))
                        {
                            onThisHost = true;
                        }
				        if (isRoot)
                        {
                            m_rootIDs.add(id);
                        }
                        else
                        {
                            m_upgradeIDs.add(id);
                        }
				        if (onThisHost)
                        {
                            m_onThisHost.add(id);
                        }
					}
				}
				else
					// is it a template?
					// container templates end up in the upgrade.list, so
					// only non-container templates end up in m_templateList
				{
					if (isATemplate(id))
                    {
                        m_templateList.add(id);
                    }
				}
			}
	    }
		m_outStream.write(NEWLINEBYTES);
		m_outStream.write(("!***************  Root container configurations").getBytes());
	    m_outStream.write(NEWLINEBYTES);
	    m_outStream.write(NEWLINEBYTES);
	    m_outStream.write((ROOTLISTPROP + "=").getBytes());
	    
		Iterator idsIterator = m_rootIDs.iterator();
		while (idsIterator.hasNext()) 
		{
			IElementIdentity containerId = (IElementIdentity) idsIterator.next();
			m_outStream.write(containerId.getName().getBytes());
			if (idsIterator.hasNext())
            {
                m_outStream.write(",".getBytes());
            }
		}
		m_outStream.write(NEWLINEBYTES);		
		
		m_outStream.write(NEWLINEBYTES);
		m_outStream.write(("!***************  Configurations to be upgraded ").getBytes());
		m_outStream.write(NEWLINEBYTES);
		m_outStream.write(NEWLINEBYTES);		
		m_outStream.write((UPGRADELISTPROP + "=").getBytes());
		idsIterator = m_upgradeIDs.iterator();
		while (idsIterator.hasNext()) 
		{
			IElementIdentity containerId = (IElementIdentity) idsIterator.next();
			m_outStream.write(containerId.getName().getBytes());	
			if (idsIterator.hasNext())
            {
                m_outStream.write(",".getBytes());
            }	
		}
		
		m_outStream.write(NEWLINEBYTES);
		
		m_outStream.write(NEWLINEBYTES);
		m_outStream.write(("!***************  Template list ").getBytes());
		m_outStream.write(NEWLINEBYTES);
		m_outStream.write(NEWLINEBYTES);		
		m_outStream.write((TEMPLATELISTPROP + "=").getBytes());
		idsIterator = m_templateList.iterator();
		while (idsIterator.hasNext()) 
		{
			IElementIdentity elId = (IElementIdentity) idsIterator.next();
			m_outStream.write(elId.getName().getBytes());	
			if (idsIterator.hasNext())
            {
                m_outStream.write(",".getBytes());
            }	
		}
		
		m_outStream.write(NEWLINEBYTES);
	}
	
	// assumes the container list has been computed

    private void writeContainerProperties() throws IOException, DirectoryServiceException
    {
        Vector allConfigs = new Vector();
        allConfigs.addAll(m_rootIDs);
        allConfigs.addAll(m_upgradeIDs);
        Iterator idsIterator = allConfigs.iterator();
        while (idsIterator.hasNext())
        {
            IElementIdentity containerId = (IElementIdentity)idsIterator.next();
            String containerName = containerId.getNameComponents()[containerId.getNameComponents().length - 1];
            // Spaces in the container name need to be preceded by a backslash
            // in the property file so Properties.load(stream) will make the space
            // part of the property name.
            String containerNameEscaped = containerName.replaceAll(" ", "\\\\ ");
            m_outStream.write(NEWLINEBYTES);
            m_outStream.write(("!***************  Properties for container " + containerId.getName()).getBytes());
            m_outStream.write(NEWLINEBYTES);
            writeSlashWarning();
            boolean hasPrimaryDS = hasPrimaryDS(containerId) != null;
            // If we're calling find dependencies for a container that has already
            // been upgraded, we're migrating the container (like backup DMs and brokers).
            // upgradeProps is not called for versions between 8.0 and the current version
            // because containers have already been migrated.
            if (containerId.getReleaseVersion().compareTo(FIRST_CIPGRADE_RELEASE_VERSION) < 0 || 
            containerId.getReleaseVersion().equals(CURRENT_RELEASE_VERSION) || hasPrimaryDS)
            {
                m_outStream.write(NEWLINEBYTES);
                m_outStream.write((containerNameEscaped + "." + WDPROP + "=").getBytes());
                if (hasPrimaryDS && containerId.getReleaseVersion().compareTo(FIRST_CIPGRADE_RELEASE_VERSION) >= 0) // it is the DM
                {
                    // write the default WD. The user will have to change this if, for instance
                    // their WD through SDM is something else. We only allow this for 8.0 --> 8.5
                    // upgrades; for previous upgrades to 8.*, we migrate to the default location
                    // get rid of escape characters
                    String m_sonicHomeEscaped = SONIC_HOME.replaceAll("\\\\", "/");
                    // escape spaces
                    m_sonicHomeEscaped = m_sonicHomeEscaped.replaceAll(" ", "\\\\ ");
                    m_outStream.write((m_sonicHomeEscaped + "/Containers/" + m_domain + "." + containerName).getBytes());
                }
                m_outStream.write(NEWLINEBYTES);

                if (containerId.getReleaseVersion().compareTo(FIRST_CIPGRADE_RELEASE_VERSION) < 0 || 
                containerId.getReleaseVersion().equals(CURRENT_RELEASE_VERSION))
                {
                    String containerLogDir = m_domain + "." + containerName + ".log";

                    m_outStream.write(NEWLINEBYTES);
                    m_outStream.write((containerNameEscaped + "." + BOOTFILEPWDPROP + "=").getBytes());
                    m_outStream.write(NEWLINEBYTES);
                    m_outStream.write((containerNameEscaped + "." + CENTRALURLPROP + "=").getBytes());
                    m_outStream.write(NEWLINEBYTES);
                    m_outStream.write((containerNameEscaped + "." + CENTRALUSERPROP + "=").getBytes());
                    m_outStream.write(NEWLINEBYTES);
                    m_outStream.write((containerNameEscaped + "." + CENTRALPWPROP + "=").getBytes());
                    m_outStream.write(NEWLINEBYTES);
                    m_outStream.write((containerNameEscaped + "." + CENTRALRTPROP + "=").getBytes());
                    m_outStream.write(NEWLINEBYTES);
                    m_outStream.write((containerNameEscaped + "." + CENTRALCTPROP + "=").getBytes());
                    m_outStream.write(NEWLINEBYTES);
                    m_outStream.write((containerNameEscaped + "." + CENTRALSCPROP + "=").getBytes());
                    m_outStream.write(NEWLINEBYTES);
                    m_outStream.write((containerNameEscaped + "." + CENTRALLBPROP + "=").getBytes());
                    m_outStream.write(NEWLINEBYTES);
                    m_outStream.write((containerNameEscaped + "." + CENTRALCONNPROPSPROP + "=").getBytes());
                    m_outStream.write(NEWLINEBYTES);
                    File containersDir = ContainerSetup.findSonicContainersDir(new File(SONIC_HOME));
                    File thisContainerDir = ContainerSetup.findContainerDir(containersDir, m_domain, containerName);
                    String containerDirAbsolute = CONTAINERSDEFAULT;
                    // Here we are calculating the absolute value of the container
                    // WD, even though we're using relative paths everywhere.
                    // This code can be removed when we're sure.
                    if (m_onThisHost.contains(containerId))
                    {
                        containerDirAbsolute = thisContainerDir.getCanonicalPath();
                        containerDirAbsolute = Utils.replaceBackSlashes(containerDirAbsolute);
                    }
                    // String newContainerWD = CONTAINERSDEFAULT;
                    // Container cache
                    m_outStream.write(NEWLINEBYTES);
                    m_outStream.write((containerNameEscaped + "." + CACHEPROP + "=./container.cache").getBytes());
                    m_outStream.write(NEWLINEBYTES);
                    // container log
                    m_outStream.write(NEWLINEBYTES);
                    m_outStream.write((containerNameEscaped + "." + CONTLOGPROP + "=./" + containerLogDir).getBytes());
                    m_outStream.write(NEWLINEBYTES);
                    // host directory for the DS component
                    if (hasDS(containerId) != null)
                    {
                        m_outStream.write(NEWLINEBYTES);
                        m_outStream.write((containerNameEscaped + "." + HOSTDIRPROP + "=.").getBytes());
                        m_outStream.write(NEWLINEBYTES);
                    }

                    IDirElement brokerEl = null;
                    if ((brokerEl = hasBroker(containerId)) != null)
                    {
                        // write a comment about db.action=LEAVE
                        m_outStream.write(NEWLINEBYTES);
                        m_outStream.write(("!If you use db.action=LEAVE, the mqstore.db.connect and recovery.log.file properties will be ignored").getBytes());
                        m_outStream.write(NEWLINEBYTES);
                        // do not write DB action if the
                        // container is not on this host
                        boolean local = m_onThisHost.contains(containerId);
                        if (!local)
                        {
                            m_outStream.write("!".getBytes());
                        }
                        m_outStream.write((containerNameEscaped + "." + DBACTIONPROP + "=COPY").getBytes());
                        m_outStream.write(NEWLINEBYTES);
                        IAttributeSet brokerSet = brokerEl.getAttributes();
                        IAttributeSet dbParams = (IAttributeSet)brokerSet.getAttribute("BROKER_DATABASE_PARAMETERS");
                        IAttributeSet logParams = (IAttributeSet)brokerSet.getAttribute("BROKER_RECOVERY_LOG_PARAMETERS");
                        IAttributeSet sslParams = (IAttributeSet)brokerSet.getAttribute("BROKER_SSL_PARAMETERS");
                        // MQSTORE_DB_CONNECT
                        String mqStoreValue = (String)dbParams.getAttribute("MQSTORE_DB_CONNECT");
                        if (mqStoreValue == null)
                        {
                            mqStoreValue = "./SonicMQStore";
                        }
                        String DBName;
                        int slashPos, backslashPos;
                        slashPos = mqStoreValue.lastIndexOf("/");
                        backslashPos = mqStoreValue.lastIndexOf("\\");
                        if (slashPos > backslashPos)
                        {
                            DBName = mqStoreValue.substring(slashPos + 1);
                        }
                        else
                        {
                            DBName = mqStoreValue.substring(backslashPos + 1);
                        }
                        // String newDBConnect = containerDirAbsolute + "/" + DBName;
                        // revert to always having a relative path. Especially useful
                        // for configurations that are reused on different hosts
                        String newDBConnect = "./" + DBName;
                        m_outStream.write(NEWLINEBYTES);
                        if (!local)
                        {
                            m_outStream.write("!".getBytes());
                        }
                        m_outStream.write((containerNameEscaped + "." + MQSTOREPROP + "=" + newDBConnect).getBytes());
                        m_outStream.write(NEWLINEBYTES);

                        // RECOVERY_LOG_FILE
                        String logLocation = (String)logParams.getAttribute("RECOVERY_LOG_PATH");
                        if (logLocation == null)
                        {
                            logLocation = "./log";
                        }
                        String logDir;
                        slashPos = logLocation.lastIndexOf("/");
                        backslashPos = logLocation.lastIndexOf("\\");
                        if (slashPos > backslashPos)
                        {
                            logDir = logLocation.substring(slashPos + 1);
                        }
                        else
                        {
                            logDir = logLocation.substring(backslashPos + 1);
                        }
                        // String newLogLocation = containerDirAbsolute + "/" + logDir;
                        // revert to always having a relative path. Especially useful
                        // for configurations that are reused on different hosts
                        String newLogLocation = "./" + logDir;
                        m_outStream.write(NEWLINEBYTES);
                        if (!local)
                        {
                            m_outStream.write("!".getBytes());
                        }
                        m_outStream.write((containerNameEscaped + "." + LOGFILEPROP + "=" + newLogLocation).getBytes());
                        m_outStream.write(NEWLINEBYTES);
                        // SSL_CERTIFICATES_DIR
                        // we keep the same value if there was one
                        String sslCertDir = (String)sslParams.getAttribute("SSL_CA_CERTIFICATES_DIR");
                        if (sslCertDir != null && sslCertDir.length() != 0)
                        {
                            if (!local)
                            {
                                m_outStream.write("!".getBytes());
                            }
                            m_outStream.write((containerNameEscaped + "." + SSLCERTDIRPROP + "=" + Utils.replaceBackSlashes(sslCertDir)).getBytes());
                            m_outStream.write(NEWLINEBYTES);
                        }
                        // SSL_CERTIFICATE_CHAIN
                        String sslCertChain = (String)sslParams.getAttribute("SSL_CERTIFICATE_CHAIN");
                        if (sslCertChain != null && sslCertChain.length() != 0)
                        {
                            if (!local)
                            {
                                m_outStream.write("!".getBytes());
                            }
                            m_outStream.write((containerNameEscaped + "." + SSLCERTCHAINPROP + "=" + Utils.replaceBackSlashes(sslCertChain)).getBytes());
                            m_outStream.write(NEWLINEBYTES);
                        }
                    }
                    m_outStream.write((containerNameEscaped + "." + WSERVICEPROP + "=").getBytes());
                    m_outStream.write(NEWLINEBYTES);

                    // *********************** COMPONENTS list
                    /*
                     * Vector components = (Vector)m_dependsMap.get(containerId.getName()); if (components != null) {
                     * m_outStream.write((containerName + "." + COMPONENTSPROP + "=").getBytes()); Iterator compIT =
                     * components.iterator(); while (compIT.hasNext()) { IElementIdentity compId =
                     * (IElementIdentity)compIT.next(); m_outStream.write(compId.getName().getBytes()); if
                     * (compIT.hasNext()) m_outStream.write(",".getBytes()); } m_outStream.write(NEWLINEBYTES); }
                     */
                }
            }
        }
    }
	
	private IDirElement hasBroker(IElementIdentity containerID) throws DirectoryServiceException
	{
		Vector components = (Vector)m_dependsMap.get(containerID);
		Iterator compIterator = components.iterator();
		while (compIterator.hasNext())
		{
			Object entry = compIterator.next();
			if (entry instanceof IElementIdentity)
			{
			    IElementIdentity compID = (IElementIdentity)entry;
			    if (compID.getType().equals("MQ_BROKER") || compID.getType().equals("MQ_BACKUPBROKER"))
                {
                    return m_ds.getFSElement(compID.getName(), false);
                }
			}
		}
		return null;
	}
	
	private IDirElement hasDS(IElementIdentity containerID) throws DirectoryServiceException
	{
		Vector components = (Vector)m_dependsMap.get(containerID);
		Iterator compIterator = components.iterator();
		while (compIterator.hasNext())
		{
			Object entry = compIterator.next();
			if (entry instanceof IElementIdentity)
			{
			    IElementIdentity compID = (IElementIdentity)entry;
			    if (compID.getType().equals("MF_DIRECTORY_SERVICE") || compID.getType().equals("MF_BACKUP_DIRECTORY_SERVICE"))
                {
                    return m_ds.getFSElement(compID.getName(), false);
                }
			}
		}
		return null;
	}
	
    private IDirElement hasPrimaryDS(IElementIdentity containerID) throws DirectoryServiceException
    {
        Vector components = (Vector)m_dependsMap.get(containerID);
        Iterator compIterator = components.iterator();
        while (compIterator.hasNext())
        {
            Object entry = compIterator.next();
            if (entry instanceof IElementIdentity)
            {
                IElementIdentity compID = (IElementIdentity)entry;
                if (compID.getType().equals("MF_DIRECTORY_SERVICE"))
                {
                    return m_ds.getFSElement(compID.getName(), false);
                }
            }
        }
        return null;
    }
	
	private void writeSlashWarning() throws IOException
	{
		m_outStream.write(NEWLINEBYTES);
		m_outStream.write(("!NOTE: Use Unix slashes '/' in all path values")
						.getBytes());
		m_outStream.write(NEWLINEBYTES);
	}
	
	private boolean isATemplate(IElementIdentity id)
	{
		Vector components = (Vector)m_dependsMap.get(id);
		if (components != null)
		{
			return (components.contains("TEMPLATE"));
		}
		return false;			    
	}

}
