package com.sonicsw.mf.common.config.upgrade;

import com.sonicsw.mf.common.config.IMFDirectories;

/**
 * Interface which defines a versioned domain manager upgrade.
 * A domain manager upgrade will consist of implementations
 * of this interface for each supported domain version.
 * Implementations of this interface must register an instance
 * of the class with the UpgradeRegistry so it can be used
 * by the upgrade driver classes.
 * 
 * @see com.sonicsw.mf.common.config.upgrade.ConfigUpgradeDriver
 * @see com.sonicsw.mf.common.config.upgrade.DomainManager
 * @see com.sonicsw.mf.common.config.upgrade.UpgradeEnv
 * @author davila
 * 
 */

public interface IVersionedDomainManagerUpgrade 
{
	
	
	public static final String HIERARCHICAL_TYPES_PATH = IMFDirectories.MF_DIR_SEPARATOR + "mx" + IMFDirectories.MF_DIR_SEPARATOR + "hierarchicalTypes" + IMFDirectories.MF_DIR_SEPARATOR;
	
	/**
	 * Called by the upgrade process before it establishes the connection
	 * to the domain that is to be used during the upgrade. Upgrade actions that
	 * are performed without a connection to the DS (like removing files) or actions
	 * which are required before a typical connection to the pre-upgrade DS can be 
	 * established without errors by the current code base should happen in this 
	 * method. If this method requires a connection to the DS, it should be noted
	 * that the connections are always performed with the current code base; if a
	 * pre-upgrade DS is structurally different from what the code base expects,
	 * part of the development effort during the release will have to include
	 * modifying the directory service code so that it can be used by this method
	 * to connect to a pre-upgrade directory service successfully.
	 * @param env The environment for the upgrade, which includes all the arguments
	 * passed to the upgrade from the SUI. In the case of this method, the 
	 * environment does not contain a connection to the DS - env.getDS will return
	 * null.
	 * @throws Exception
	 */
	public void preConnectUpgrade(UpgradeEnv env) throws Exception;
	
	/**
	 * Called by the upgrade process to perform domain upgrade related actions.
	 * Domain upgrade actions are typically concerned with upgrade of system
	 * configurations stored outside of the logical namespace and not visible 
	 * to users through tools like the SMC and the config API, as opposed to
	 * application configurations like Brokers, Clusters and Management 
	 * Framework configurations. 
	 * @param env The environment for the upgrade, through which operations on
	 * the DS can be performed. 
	 * @return A String which can be displayed by the SUI as an informational 
	 * message for the user. It can be null. This message should 
	 * aggregate the messages from the result of calling upgrade for the
	 * earlier version of the domain manager.
	 * @throws Exception
	 */
	public String upgrade(UpgradeEnv env) throws Exception;
	
	
	/**
	 * Return the followup version of the domain after it's been upgraded
	 * by this object. For instance, if this object upgrades domains of 
	 * version "101", this method would return "102"
	 * @return
	 */
	public String getUpgradedReleaseVersion();
	
	/**
	 * Return the product version that this object can handle
	 * @return
	 */
	public String getProductVersion();

}
