package com.sonicsw.mf.common.config.upgrade;

import java.io.*;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Properties;
import java.util.StringTokenizer;
import java.util.Vector;

import javax.management.JMRuntimeException;

import com.sonicsw.mf.comm.ConnectTimeoutException;
import com.sonicsw.mf.comm.InvokeTimeoutException;
import com.sonicsw.mf.common.IDirectoryAdminService;
import com.sonicsw.mf.common.IDirectoryFileSystemService;
import com.sonicsw.mf.common.MFSecurityException;
import com.sonicsw.mf.common.config.IAttributeSet;
import com.sonicsw.mf.common.config.IElementIdentity;
import com.sonicsw.mf.common.config.IMFDirectories;
import com.sonicsw.mf.common.config.Reference;
import com.sonicsw.mf.common.config.query.From;
import com.sonicsw.mf.common.config.query.FromElementType;
import com.sonicsw.mf.common.config.query.Query;
import com.sonicsw.mf.common.dirconfig.DirectoryServiceException;
import com.sonicsw.mf.common.dirconfig.IDeltaDirElement;
import com.sonicsw.mf.common.dirconfig.IDirElement;
import com.sonicsw.mf.common.runtime.IRemoteExecResult;
import com.sonicsw.mf.common.runtime.impl.ExecUtility;
import com.sonicsw.mf.framework.IContainer;
import com.sonicsw.mf.framework.agent.ContainerSetup;
import com.sonicsw.mf.framework.agent.ci.ILauncherContainerDriver;
import com.sonicsw.mf.framework.agent.ci.LauncherDriver;
import com.sonicsw.mf.jmx.client.DirectoryServiceProxy;
import com.sonicsw.mf.mgmtapi.config.constants.IContainerConstants;
import com.sonicsw.mf.mgmtapi.config.constants.IDirectoryServiceConstants;
import com.sonicsw.mf.mgmtapi.runtime.IAgentManagerProxy;
import com.sonicsw.mf.mgmtapi.runtime.IAgentProxy;
import com.sonicsw.mf.mgmtapi.runtime.ProxyRuntimeException;

public final class Migrate implements IMigrationProcess
{
	private DirectoryService m_ds;
	private Properties m_migrateProps = new Properties();
	private Properties m_origProps = new Properties();
	private HashMap<String, HashMap<String, String>> m_containerProps = new HashMap<String, HashMap<String,String>>();
	private IDirectoryFileSystemService m_fileSystemService;
	private String m_brokerCN;
	private Vector m_rootList = new Vector();
	private Vector m_upgradeList = new Vector();
	private Vector m_templateList = new Vector();
	private boolean m_isESBInstalled = false;
	private boolean m_isRDBMSInstalled = false;
	private String m_sonicHome = System.getProperty("sonic.home");
	private String m_archives = (new File(m_sonicHome, "Archives")).getCanonicalPath();
	private File m_esbOnDisk = new File(m_archives, "ESB/" + CURRENT_PRODUCT_VERSION + "/ESBcontainer.car");
	private File m_rdbmsOnDisk = new File(m_sonicHome, "DBService" + CURRENT_PRODUCT_VERSION + "/lib");
	private File m_wbOnDisk = new File(m_sonicHome, "Workbench" + CURRENT_PRODUCT_VERSION);
    private boolean m_isLoggerInstalled = false;
    private boolean m_debug = false;
    private IDirElement m_DMElement = null;
    private ConfigUpgrade m_configUpgrade = null;
    private File m_seedPropFile = new File("seedimports.properties");
    private File m_containersDir = null;
    private boolean m_restartDM = false;
    // m_restartContainers is of the form:
    // containerID key
    // String value <methodForStartup>,<logDirectory>
    // <methodForStartup> could also include ERROR, which means something went
    // wrong when the container was shutdown and we need to tell the user.
    //<logDirectory> is required because the LauncherContainerDriver will be looking
    // in the container log to figure out if the container has been started or not
    private HashMap<String,String> m_restartContainers = new HashMap<String,String>();
    private boolean m_isWBUpgrade = System.getProperty("wb.upgrade", "false").equalsIgnoreCase("true");
    private int m_requestTimeout = REQUEST_TIMEOUT_DEFAULT;
    private static String UPGRADE_ERROR = "upgrade error";
    private static String UPGRADE_MESSAGE = "upgrade message";
    private String m_wbLocation = null;
    private static final Properties SONICSW_PROPS = new Properties();
    private static String m_OS = System.getProperty("os.name").toLowerCase();
    
    private static void printUsage(String errorMessage)
    {
    	if (errorMessage != null)
    	{
    		System.out.println();
    	    System.out.println("  error: " + errorMessage);
    	    System.out.println();
    	}
    	System.out.println();
    	System.out.println("*****************************  upgrade usage ****************************");
    	System.out.println();
    	System.out.println("Arguments to upgrade:");
    	System.out.println();
    	System.out.println("upgrade <properties file> [<old workbench location>]");
    	System.out.println("or");
    	System.out.println("upgrade -h|-help");
        System.out.println();
        System.out.println("where:");
        System.out.println();
        System.out.println("<properties file> is the file generated by calling the upgradeProps script. This argument is required.");
        System.out.println("<old workbench location> In a workbench upgrade, this is the location of the existing workbench. Optional, needed only if the workspace is to be migrated from the old installation to the new one, and both the old and new workbench contain Eclipse that Sonic installed.");
        System.out.println();
        System.out.println("upgrade -h|help will print this text.");
        System.out.println();
        System.out.println("**************************************************************************");
        System.exit(0);
    }
    
	/**
	 * @param args
	 */
	public static void main(String[] args) 
	{
		if (args.length < 1)
        {
            printUsage("You must provide a properties file to upgrade.");
        }
		if (args.length > 2)
        {
            printUsage(null);
        }
		String propFileName = args[0];
		// unix will pass empty strings when arguments are not give. Windows
		// passes nulls
		if (propFileName.length() == 0)
        {
            printUsage("You must provide a properties file to upgrade.");
        }
		if (propFileName.toLowerCase().startsWith("-h"))
        {
            printUsage(null);
        }
		String wbLocation = null;
		if (args.length == 2 && args[1].length() > 0)
        {
		    wbLocation = args[1];
        }
		File propFile = new File(propFileName);
		if (!propFile.exists())
		{
			Utils.printError(UPGRADE_ERROR, null, "File " + propFile + " does not exist");
			System.exit(-1);
		}
		if (!propFile.isFile())
		{
			Utils.printError(UPGRADE_ERROR, null, "File " + propFile + " is not a file");
			System.exit(-1);
		}
		if (wbLocation != null)
		{
			File wbFile = new File(wbLocation);
			if (!wbFile.exists())
			{
				Utils.printError(UPGRADE_ERROR, null, "Directory " + wbLocation + " does not exist");
				System.exit(-1);
			}
			if (!wbFile.isDirectory())
			{
				Utils.printError(UPGRADE_ERROR, null, wbLocation + " should be a directory, but it's a file");
				System.exit(-1);
			}
			// make sure there is an eclipse directory in the src workbench. The WB
			// upgrade will throw an exception if it isn't
			File srcEclipse = new File(wbFile, "eclipse");
			if (!srcEclipse.exists())
			{
				Utils.printError(UPGRADE_ERROR, null, wbLocation + " does not contain an eclipse installation. Modify or remove the workbench location argument and try again.");
			    System.exit(-1);
			}
		}
		Migrate migrate = null;
		String upgradeMessage = null;
		try
		{
		    migrate = new Migrate(propFile, wbLocation);
		    migrate.debug("Value of sonic.home = " + migrate.m_sonicHome);
		    // migrate will get called to restart root containers on the 
		    // domain manager host after the DM migration and seeding
		    // if so, restart containers and exit
		    if (migrate.m_restartContainers.size() > 0)
		    {
		    	migrate.debug("Before restartRootContainers");
		    	migrate.restartRootContainers();
		    	migrate.debug("After restartRootContainers");
		    }
		    else
		    {
		       System.out.println("upgrade connecting to the domain...");
		       migrate.connect(null);
		       System.out.println("upgrade connecting to the domain...done");
		       migrate.debug("About to readProps");
		       migrate.readProps();		 
		       migrate.debug("About to checkErrors");
		       migrate.checkErrors();
		       migrate.debug("About to doMigration");
		       upgradeMessage = migrate.doMigration();
		       migrate.debug("done");
		    }
		    migrate.debug("After if then else");
		    
		}
		catch (MigrationException migE)
		{
			if (migrate != null)
            {
                migrate.debug("In MigrationException");
            }
			// create a formatted error for the migration utility
			Utils.printError(UPGRADE_ERROR, null, migE.getMessage() + 
					System.getProperty("line.separator") + "Upgrade will not start");
		}
		catch (ProxyRuntimeException proxyE)
		{
			Throwable cause = proxyE.getCause();
			if (cause != null && (cause instanceof InvokeTimeoutException))
			{
				Utils.printError(UPGRADE_ERROR, null, "The upgrade had a problem communicating with the domain manager. Revert to the state prior to this upgrade and try again." + 
						" You might want to increase the request timeout through the property request.timeout in the properties file.");
			}
            else
            {
                Utils.printError(UPGRADE_ERROR, proxyE, null);
            }
				
		}
		catch (JMRuntimeException jmsE)
		{
			Throwable cause = jmsE.getCause();
			if (cause != null && (cause instanceof ConnectTimeoutException))
			{
				Utils.printError(UPGRADE_ERROR, null, "The upgrade had a problem connecting to the domain manager. Check that your connection information is correct, and that the domain manager container is running");
			}
			else if (cause != null && (cause instanceof MFSecurityException))
            {
                Utils.printError(UPGRADE_ERROR, null, "The connection attempt threw a security exception. Check that the specified username and password are correct.");
            }
            else
            {
                Utils.printError(UPGRADE_ERROR, jmsE, null);
            }
		}
		catch (Exception e)
		{
			Utils.printError(UPGRADE_ERROR, e, null);
		}
		finally
		{
			if (upgradeMessage != null && upgradeMessage.length() > 0)
            {
                Utils.printError(UPGRADE_MESSAGE, null, upgradeMessage);
            }
			if (migrate != null)
            {
                try {migrate.disconnect();} catch (Exception ex) {ex.printStackTrace();}
            }
		}
		System.exit(0);
	}
	
	private void debug(String printThis)
	{
		if (m_debug)
        {
            System.out.println(printThis);
        }
	}
	
	Migrate(File propFile, String wbLocation) throws Exception
	{
	    this.m_wbLocation = wbLocation;
		FileInputStream propStream = new FileInputStream(propFile);
		m_origProps.load(propStream); 
		// get rid of the empty string values
		Enumeration keys = m_origProps.keys();
		while (keys.hasMoreElements())
		{
			String key = (String)keys.nextElement();
			String value = (String)m_origProps.get(key);
			if (value.length() != 0)
			{
				m_migrateProps.put(key, value);
			}
		}
		propStream.close();
		String url = m_migrateProps.getProperty(URLPROP);
		String domainName = m_migrateProps.getProperty(DOMAINPROP);
		if (domainName == null && url != null && url.toLowerCase().endsWith("xml"))
        {
            m_migrateProps.put(DOMAINPROP, DEFAULTDOMAIN);
        }
		String debugBOOL = m_migrateProps.getProperty(DEBUGPROP);
		if (debugBOOL != null)
        {
            m_debug = (new Boolean(m_migrateProps.getProperty(DEBUGPROP))).booleanValue();
        }
		if (m_debug)
         {
            System.setProperty("DebugAll.debug", "true");  // to support legacy upgrade code
        }
		String rt = m_migrateProps.getProperty(RTPROP);
        if (rt != null)
        {
        	try
        	{
        		m_requestTimeout = (new Integer(rt)).intValue();
        	}
        	catch (NumberFormatException noNum)
        	{
        		throw new MigrationException("Request timeout specified is not a number: " + rt);
        	}
        }
		if (m_migrateProps.getProperty(RESTARTCONTAINERSPROP) != null)
        {
            parseRestartContainers();
        }
	}
	
	private void saveProps() throws IOException
	{
		FileOutputStream outStream = new FileOutputStream(m_seedPropFile);
		Properties seedingProps = new Properties();
		seedingProps.put(DOMAINPROP, m_migrateProps.getProperty(DOMAINPROP));
		if (m_DMElement != null)
		{
			HashMap<String, String> dmProps = getContainerProps(getContainerName(m_DMElement.getIdentity().getName()));
			String origRelVersion = dmProps.get("RELEASE_VERSION");
			boolean dsMoved = origRelVersion != null && willMigrate(origRelVersion);
			// what has been all throughout the upgrade DSHOSTPROP is 
			// the location of the DS; whereas the property is used as the working
			// area of the DM container by install ant scripts
		    String dsHostPropValue = dsMoved ? m_migrateProps.getProperty(NEWWDPROP) : m_migrateProps.getProperty(DSHOSTPROP);
		    seedingProps.put(DSHOSTPROP, dsHostPropValue); 
		    String username = m_migrateProps.getProperty(USERNAMEPROP);
		    if (username != null)
            {
                seedingProps.put(USERNAMEPROP, m_migrateProps.getProperty(USERNAMEPROP));
            }
		    seedingProps.put(URLPROP, dsHostPropValue + "/ds.xml");
		    // save the WD of the DM to find the launchcontainer script. If this is saved, it will be used
		    // as the directory where we can find the launch script
		    String wd = dmProps.get(WDPROP);
		    if (origRelVersion != null && origRelVersion.compareTo("104") >= 0 && wd != null)
            {
                seedingProps.put(WDPROP, wd);
            }
		// Write default connection properties for ESB seed files
		// get the URL, username and password of the DM mgmt container
		// assume these haven't changed and can be fetched from the static copy
		// we have
		    IAttributeSet connectionArgs = (IAttributeSet)m_DMElement.getAttributes().getAttribute(IContainerConstants.CONNECTION_ATTR);
		    String conUrl = (String)connectionArgs.getAttribute(IContainerConstants.CONNECTIONURLS_ATTR);
		    if (conUrl == null)
            {
                conUrl = IContainerConstants.CONNECTIONURLS_DEFAULT;
            }
		    String conUser = (String)connectionArgs.getAttribute(IContainerConstants.DEFAULTUSER_ATTR);
		    String conPwd = (String)connectionArgs.getAttribute(IContainerConstants.DEFAULTPASSWORD_ATTR);
		    seedingProps.put(CONTAINERURLPROP, conUrl);
		    if (conUser != null && conUser.length() > 0)
            {
                seedingProps.put(CONTAINERUSERPROP, conUser);
            }
		    if (conPwd != null && conPwd.length() > 0)
            {
                seedingProps.put(CONTAINERPWDPROP, conPwd);
            }
		    if (m_migrateProps.getProperty(ESBCNPROP) != null)
            {
                seedingProps.put(ESBCNFORSEEDERPROP, m_migrateProps.getProperty(ESBCNPROP));
            }
		    // save broker.path for the ESB samples target. Chop off the _Default
			if (m_isWBUpgrade)
			{
	            String brokerPath = m_migrateProps.getProperty(BROKERPATHPROP);
	            brokerPath = brokerPath.substring(0, brokerPath.lastIndexOf(IMFDirectories.MF_DIR_SEPARATOR));
				seedingProps.put(BROKERPATHPROP, brokerPath);
			}
			// save the src wb location for the workspace migration. All error checking has happened
			// before we get here.
			if (m_wbLocation != null)
             {
                seedingProps.put(SAVEDWBLOCATIONPROP, m_wbLocation);
				   //outStream.write((SAVEDWBLOCATIONPROP + "=" + m_wbLocation.replaceAll("\\\\", "/")).getBytes());	
            }
		}
		// write the list of containerIDs which must be restarted after importing
		// and seeding of files is done
		addRestartContainersList(seedingProps);
		seedingProps.put(DEBUGPROP, m_migrateProps.getProperty(DEBUGPROP));
		seedingProps.store(outStream, "Properties which will drive the seeing of product jars, cars and sample configurations, and the restart of MF containers");
		outStream.close();
	}
	
	// keep some of the properties used by multiple methods as instance variables
	private void readProps() throws Exception
	{
		m_brokerCN = m_migrateProps.getProperty(BROKERCNPROP);
		m_containersDir = ContainerSetup.findSonicContainersDir(new File(m_sonicHome));
		String rootlist = m_migrateProps.getProperty(ROOTLISTPROP);
		debug("readProps rootlist == " + rootlist);
		if (rootlist != null)
		{
		    StringTokenizer rootTokens = new StringTokenizer(rootlist, ",", false);
		    // parse the root.list and the upgrade.list and save them
		    while (rootTokens.hasMoreTokens())
		    {
			    String containerID = rootTokens.nextToken();
			    debug("readProps adding " + containerID + " to m_rootList");
			    m_rootList.add(containerID);						
		    }
		}
		String upgradelist = m_migrateProps.getProperty(UPGRADELISTPROP);
		if (upgradelist != null)
		{
			debug("readProps reading upgradeList ");
		    StringTokenizer upgradeTokens = new StringTokenizer(upgradelist, ",", false);
		    while (upgradeTokens.hasMoreTokens())
		    {
			    String containerID = upgradeTokens.nextToken();
			    debug("readProps read from upgradeList " + containerID);
			    m_upgradeList.add(containerID);						
		    }
		}
		String templateList = m_migrateProps.getProperty(TEMPLATELISTPROP);
		if (templateList != null)
		{
			debug("readProps reading templateList ");
		    StringTokenizer templateTokens = new StringTokenizer(templateList, ",", false);
		    while (templateTokens.hasMoreTokens())
		    {
			    String elID = templateTokens.nextToken();
			    debug("readProps read from tempalteList " + elID);
			    m_templateList.add(elID);						
		    }
		}
		// find out what's installed
		// You need the "in the DS check" for remote containers
		// you need the on disk check on the domain manager, where
		// the files have not been loaded up to the DS yet.
		// ESB
		debug("finding out what's installed");
		debug("testing esb");
		if (m_fileSystemService.getFSElement("/Archives/ESB/" + CURRENT_PRODUCT_VERSION + "/ESBcontainer.car", false) != null)
        {
            m_isESBInstalled = true;
        }
        else if (m_esbOnDisk.exists())
        {
            m_isESBInstalled = true;
        }
		debug("testing dbservice");
		if (m_fileSystemService.getFSElement("/System/DBService/" + CURRENT_PRODUCT_VERSION + "/lib/esb_rdbms.jar", false) != null)
        {
            m_isRDBMSInstalled = true;
        }
        else if (m_rdbmsOnDisk.exists())
        {
            m_isRDBMSInstalled = true;
        }
		// if the user specified -wb <old workbench location> and there's no workbench installed,
		// throw an error.
		if (m_wbLocation != null && !m_wbOnDisk.exists())
        {
            throw new MigrationException("This installation does not contain a Workbench yet an old workbench directory was specified. Try again without the -wb argument");
        }
		readContainerProps();
	}
	
	private void connect(String url) throws Exception
	{
		if (url == null)
        {
            url = m_migrateProps.getProperty(URLPROP);
        }
		String username = m_migrateProps.getProperty(USERNAMEPROP);
		String password = m_migrateProps.getProperty(PASSWORDPROP);
		String node = m_migrateProps.getProperty(NODEPROP);
		String domain = m_migrateProps.getProperty(DOMAINPROP);
		String conProps = m_migrateProps.getProperty(CONNPROPSPROP);
		debug("connection.system.properties == " + conProps);
		if (conProps != null)
        {
            setConnectionProps(conProps);
        }
		if (url.toLowerCase().endsWith(".xml"))
        {
            m_ds = new DirectoryService(url, true);
        }
        else
        {
            m_ds = new DirectoryService(url, username, password, domain, node, m_requestTimeout);
        }
		m_fileSystemService = m_ds.getFileSystemService();
	}
	
	private void disconnect() throws Exception
	{
		if (m_ds != null)
		{
			m_ds.closeDS(); 
			m_ds = null;
			m_fileSystemService = null;
		}
	}
	
	private void setConnectionProps(String propsValue) throws Exception
	{
		if (propsValue == null)
        {
            debug("setConnectionProps, propsValue == null");
        }
        else if (propsValue.length() == 0)
        {
            debug("setConnectionProps, propsValue length == 0");
        }
		StringTokenizer tokenizer = new StringTokenizer(propsValue, ",=", false);
		// check that we have an even number of tokens
		if (tokenizer.hasMoreElements() && !((tokenizer.countTokens() % 2) == 0))
        {
            throw new MigrationException("Incorrect connection system property values detected; odd number of tokens in string " + propsValue);
        }
		while (tokenizer.hasMoreTokens())
		{
			String prop = tokenizer.nextToken();
			String value = tokenizer.nextToken();
			debug("setConnectionProps prop " + prop + " value " + value);
			System.setProperty(prop, value);
		}
	}
	
	/*
	 * Error conditions:
    * The DM container has not been upgraded, and it's not in the root.list. This container must be upgraded first. This check applies to the upgradeConfigurations API as well.
    * A connection cannot be established using the connection information given. Applies to the upgradeConfigurations API as well.
    * A configuration in the root.list or upgrade.list cannot be found. Applies to the upgradeConfigurations API as well.
    * A ds.xml is set in the properties file but none of the containers in root.list or upgrade.list is a primary or standalone domain manager.
    * The path to ds.xml is given and it cannot be found.
    * An MF container with an ESB container with a DBService instance is to be upgraded, and the DBService jar files have not been loaded. Applies to the upgradeConfigurations API as well.
    * An MF container with an ESB container with a DBService instance is to be upgraded, and there's no DBService control code in the properties file. Applies to the upgradeConfigurations API as well.
    * An MF container with an ESB container with a BPEL instance is to be upgraded, and the BPEL jar files have not been loaded. Applies to the upgradeConfigurations API as well.
    * An MF container with an ESB container is to be upgraded, and the ESB archive file has not been loaded. Applies to the upgradeConfigurations API as well.
    * An MF container with an Event Monitor is to be upgraded, and there's no Event Monitor control code in the properties file. Applies to the upgradeConfigurations API as well.
    * An MF container with a broker is to be upgraded, and there's no MQ control code in the properties file. Applies to the upgradeConfigurations API as well.
    * A <containerName>.db.action of COPY or MOVE is specified and mqstore.db.connect and recovery.log.file are not both specified for the container.
    * <containerName>.previous.working.directory is not specified and there are relative paths in the broker DB and log path fields or the DS storage directory.
    * A <containerName>.previous.working.directory is specified but it cannot be found.
    * The domain contains any pre 7.5 configuration. All configurations have to be upgraded to 7.5 or above before the 8.5 migration tool can run. 
    * Applies to the upgradeConfigurations API as well.

	 */
	private void checkErrors() throws Exception
	{
		checkRootList();
		checkUpgradeList();
		checkContainerProperties();
	}
	
    private void checkRootList()
    throws Exception
    {
        // String dsXml = m_migrateProps.getProperty(DSBOOTFILEPROP);
        // if (dsXml != null && !(new File(dsXml)).exists())
        // throw new MigrationException("ds.xml file specified cannot be found " + dsXml);
        // make sure we can find the versioned directory under the launcher if we are setting up
        // root containers
        if (!m_rootList.isEmpty())
        {
            String version = findLauncherVersion();
            if (version == null)
            {
                throw new MigrationException("The upgrade could not find the container setup directories of the launcher. Check that your installation has a Launcher/" + CURRENT_PRODUCT_VERSION + ".x.x directory in it");
            }
        }
        Iterator rootIT = m_rootList.iterator();
        while (rootIT.hasNext())
        {
            String containerID = (String)rootIT.next();
            IDirElement container = m_fileSystemService.getFSElement(containerID, false);
            debug("found " + containerID + " in root.list");
            if (container == null)
            {
                throw new MigrationException(containerID + " is not found in the DS");
            }
            // We need to keep the RELEASE_VERSION because storage for DS and brokers
            // will not migrate after 8.0. Containers with the current version
            // appear in the root list when they've already been upgrade but need to
            // be migrated. We use a null value to signal those containers
            HashMap<String, String> containerProps = getContainerProps(getContainerName(containerID));
            String relVersion = container.getIdentity().getReleaseVersion();
            if (!relVersion.equals(CURRENT_RELEASE_VERSION))
            {
                containerProps.put("RELEASE_VERSION", relVersion);
            }
            // warn the user that we're going to try to migrate this container, if the new directories
            // are there already
            boolean migrate = willMigrate(relVersion) || relVersion.equals(CURRENT_RELEASE_VERSION);
            if (migrate)
            {
                String containerName = getContainerName(containerID);
                File containerWD = ContainerSetup.findContainerDir(m_containersDir, m_migrateProps.getProperty(DOMAINPROP), containerName);
                if (containerWD.exists())
                {
                    throw new Exception("The directory " + containerWD + " already exists and would be overwritten if we continue. Delete the directory and try again");
                }
            }
            if (checkComponents(container, false, true))
            {
                if (m_DMElement != null) // we already found a DM
                {
                    // we would only get here if the user mucked the properties file
                    throw new MigrationException("The upgrade tool can only upgrade one domain manager at a time. There are at least two containers with a directory service in the root.list. Put one back in the upgrade.list before running the upgrade, then run upgradeProps and upgrade for the second one");
                }
                // In the case where there is more than one instance of DirectoryService, we
                // force them to connect to the domain represented by that instance.
                // Most of the time, this will be a mistake; for instance, they have upgraded
                // from 7.0 to 8.5, but want to upgrade the DS that was the backup. They could
                // but we make then upgrade through a connection to its DS storage. Hopefully
                // it makes the user think twice about whether that's what they really want to do
                if (isOnlineConnection() && ((DirectoryServiceProxy)m_fileSystemService).getDirectoryServiceReleaseVersion().startsWith(CURRENT_PRODUCT_VERSION))
                {
                    throw new MigrationException("You're connected to a running, upgraded domain manager and wish to upgrade another domain manager. You must connect to the domain specified by " + containerID + " in order to upgrade it");
                }
                m_DMElement = container;
                // This is a DM upgrade, so check that we can find sonicsw.properties
                checkInstallProperties();
                // save the mgmt broker name for the workbench upgrade. The Id of the
                // mgmt broker is needed to seed the samples (property is broker.path)
                m_migrateProps.put(BROKERPATHPROP, containerProps.get(BROKERPATHPROP));
            }
            // don't break. Check all component related errors of the root containers here
            // Check all container attributes that are supposed to be paths on this host
            checkRootContainerPaths(container);
        }
        // if (m_DMElement != null && dsXml == null)
        // throw new MigrationException("The domain manager container " + m_DMElement.getIdentity().getName() +
        // " is being upgraded but no DS boot file was specified.");
        if (m_DMElement != null)
         {
            return;
        // if (dsXml != null && m_DMElement == null)
        // throw new
        // MigrationException("ds.xml was specified but the domain manager container is not found in the root.list. If the domain manager was already upgraded, remove the ds.xml file specification from the properties file. If you wish to upgrade the domain manager container, add it to the root.list property");
        }

        if (m_DMElement == null)
        {
            // find the upgraded DM container
            Query typeQuery = new Query();
            From typeFrom = new FromElementType("MF_CONTAINER");
            typeQuery.setFrom(typeFrom);
            IDirElement[] configs = null;

            configs = m_fileSystemService.getFSElements(typeQuery, false);
            for (int i = 0; i < configs.length; i++)
            {
                IDirElement container = configs[i];
                if (checkComponents(container, true, false))
                {
                    if (!container.getIdentity().getReleaseVersion().equals(CURRENT_RELEASE_VERSION))
                    {
                        throw new MigrationException("Domain Manager container must be upgraded first and it must be in the root.list.");
                    }
                }
            }
        }
    }
	
	// check that the path properties for the root container are legal paths on this host
	private void checkRootContainerPaths(IDirElement container) throws Exception
	{
		String containerName = (String)container.getAttributes().getAttribute(IContainerConstants.CONTAINER_NAME_ATTR);
		File containerWD = ContainerSetup.findContainerDir(m_containersDir, m_migrateProps.getProperty(DOMAINPROP), containerName);
		HashMap containerProps = (HashMap)m_containerProps.get(containerName);
		if (containerProps != null)
		{
			// check cache directory
			String prop = (String)containerProps.get(CACHEPROP);
			File propFile = null;
			if (prop != null)
			{	
			    propFile = new File(prop);
			    if (!propFile.isAbsolute())
                {
                    propFile = new File(containerWD, prop);
                }
			    if (!testValidFile(propFile))
                {
                    throw new MigrationException(prop + " is not a valid cache directory for " + containerName + " on this host.");
                }
			}
			// check log directory
			prop = (String)containerProps.get(CONTLOGPROP);
			if (prop != null)
			{
			    propFile = new File(prop);
			    if (!propFile.isAbsolute())
                {
                    propFile = new File(containerWD, prop);
                }
			    if (!testValidFile(propFile))
                {
                    throw new MigrationException(prop + " is not a valid container log file for " + containerName + " on this host.");
                }
			}
			// Test SSL Certificate Dir
			/*prop = (String)containerProps.get(SSLCERTDIRPROP);
			if (prop != null)
			{
				propFile = new File(prop);
			    if (!testValidFile(propFile))
				    throw new MigrationException(prop + " is not a valid SSL certificate directory for " + containerName + " on this host.");
			
			}*/
		}
	}
	
	private boolean checkComponents(IDirElement container, boolean findDSOnly, boolean rootContainer) throws Exception
	{
		debug("checkComponents for " + container.getIdentity().getName());
		boolean containsDS = false;
		IAttributeSet compSet = (IAttributeSet)container.getAttributes().getAttribute("COMPONENTS");
		if (compSet != null)
        {
            HashMap componentsMap = compSet.getAttributes();
            Collection componentSets = componentsMap.values();
            Iterator componentSetsIt = componentSets.iterator();
            while (componentSetsIt.hasNext())
            {
                IAttributeSet componentSet = (IAttributeSet)componentSetsIt.next();
                Reference componentRef = (Reference)componentSet.getAttribute("CONFIG_REF");
                String componentID = componentRef.getElementName();
                debug("checkComponents component " + componentID);
                IElementIdentity compID = null;
                try
                {
                    compID = m_fileSystemService.getFSIdentity(componentID);
                }
                catch (Exception e)
                {
                	compID = m_fileSystemService.getFSIdentity(componentID + "/_Default");
                }
		        if (compID.getType().equals("MF_DIRECTORY_SERVICE"))
                {
                    containsDS = true;
                }
		        if (!findDSOnly)
                {
                    checkComponent(compID, container, rootContainer);
                }
		        if (containsDS && findDSOnly)
                {
                    // we're done. we're not checking the upgradability of the container, so return the answer
		        	return containsDS;
                }
            }
        }
		return containsDS;
	}
	
    private void checkComponent(IElementIdentity compID, IDirElement container, boolean rootContainer) throws Exception
    {
        String containerVersion = container.getIdentity().getReleaseVersion();
        String type = compID.getType();
        IElementIdentity containerID = container.getIdentity();
        HashMap<String, String> containerProps = getContainerProps(getContainerName(containerID.getName()));
        String relVersion = containerProps.get("RELEASE_VERSION");
        boolean migrate = relVersion != null ? willMigrate(relVersion) : true;
        // skip this check if component has been upgraded already....
        if (type.equals("XQ_CONTAINER") && !containerVersion.equals(CURRENT_RELEASE_VERSION))
        {
            checkXQContainer(compID);
            // if we're running this code, the MF archives have been loaded,
            // so don't need to check for logger archives.
            // skip this check if component has been upgraded already....
        }
        else if (type.equals("MQ_BROKER") || type.equals("MQ_BACKUPBROKER"))
        {
            if (migrate)
            {
                checkBrokerComponent(compID, containerID);
            }
            else
            {
                // save the name of the broker. If this is the DM it will be
                // needed for the workbench upgrade to tailor the samples
                containerProps.put(BROKERPATHPROP, compID.getName());
            }
        }
        else if (type.equals("MF_DIRECTORY_SERVICE"))
        {
            checkDSComponent(compID, container, rootContainer);
            // save the fact that there is a DS in this container. ContainerSetup
            // needs to know so that it can write the ds boot file
        }
        else if (type.equals("MF_BACKUP_DIRECTORY_SERVICE"))
        {
            checkBackupDSComponent(compID, containerID);
        }
    }
	
	private String checkWD(HashMap containerProps, String containerName) throws Exception
	{
		if (containerProps == null)
        {
            throw new MigrationException("previous.working.directory is required to migrate container " + containerName + " and was not specified.");
        }
		String prevWD = (String)containerProps.get("previous.working.directory");
		if (prevWD == null)
        {
            throw new MigrationException("previous.working.directory is required to migrate container " + containerName + " and was not specified.");
        }
		if (!(new File(prevWD)).isAbsolute())
        {
            throw new MigrationException("An absolute value for previous.working.directory is required to migrate container " + containerName + " and was specified as a relative path: " + prevWD);
        }
		if (!(new File(prevWD)).exists())
        {
            throw new MigrationException("The directory named by the previous.working.directory property for container " + containerName + " does not exist: " + prevWD);
        }
		return prevWD;
	}
	
	private String checkLaunchContainer(HashMap containerProps, String containerName) throws Exception
    {
        if (containerProps == null)
        {
            throw new MigrationException("previous.working.directory is required to find the launchcontainer script for container " + containerName + " and was not specified.");
        }
        String prevWD = (String)containerProps.get("previous.working.directory");
        if (prevWD == null)
        {
            throw new MigrationException("previous.working.directory is required to find the launchcontainer script for container " + containerName + " and was not specified.");
        }
        if (!(new File(prevWD)).isAbsolute())
        {
            throw new MigrationException("An absolute value for previous.working.directory is required for container " + containerName + " and was specified as a relative path: " + prevWD);
        }
        if (!(new File(prevWD)).exists())
        {
            throw new MigrationException("The directory named by the previous.working.directory property for container " + containerName + " does not exist: " + prevWD);
        }
        if (!((new File(prevWD, "launchcontainer.bat")).exists() || (new File(prevWD, "launchcontainer.sh")).exists()))
        {
            throw new MigrationException("The container startup scripts are not found in the specified previous.working.directory property for container " + containerName + ": " + prevWD + ". The upgrade will not be able to restart the container");
        }
        return prevWD;
    } 
	
	// esbContainer is passed in for the error message. It is the ID of the ESB container
	private void checkServices(String esbContainer, IAttributeSet services, IDirectoryAdminService dsAdmin)
	    throws Exception
	{
        // we have checked for null before getting here
        HashMap servicesMap = services.getAttributes();
        Collection servicesMaps = servicesMap.values();
        Iterator servicesMapsIt = servicesMaps.iterator();
        while (servicesMapsIt.hasNext())
        {
            IAttributeSet serviceSet = (IAttributeSet)servicesMapsIt.next();
            String serviceName = (String)serviceSet.getAttribute("service_ref");
            IDirElement serviceEl = dsAdmin.getElement("/xqServices/" + serviceName, false);
            // cover the case when a service instance has been deleted but an ESB container
            // still refers to it.
            if (serviceEl != null)
            {
                String serviceType = (String)serviceEl.getAttributes().getAttribute("type_ref");
                if (serviceType.equals("DatabaseServiceType"))
                {
                    if (!m_isRDBMSInstalled)
                    {
                        throw new MigrationException("DBService has not been installed, so cannot proceed with the upgrade of " + esbContainer);
                    }
                }
                if ((serviceType.equals("XMLServiceType") || serviceType.equals("OrchestrationServiceType") || serviceType.equals("BPELServiceType")) && !m_isWBUpgrade)
                {
                    throw new MigrationException("Services of type XMLServiceType, OrchestrationServiceType and BPELServiceType cannot be upgraded to " + CURRENT_PRODUCT_VERSION + ", so " + esbContainer + " cannot be upgraded");
                }
            }
        }
	}
	
	/*
	 * public static String BOOTFILEPWDPROP = "boot.file.password";
    public static String CENTRALURLPROP = "central.connection.url";
    public static String CENTRALUSERPROP = "central.connection.username";
    public static String CENTRALPWPROP = "central.connection.password";
    public static String CENTRALRTPROP = "central.connection.request.timeout";
    public static String CENTRALCTPROP = "central.connection.connect.timeout";
    public static String CENTRALSCPROP = "central.connection.socket.connect";
    public static String CENTRALLBPROP = "central.connection.load.balancing";
    public static String CENTRALCONNPROPSPROP = "central.connection.system.properties";
    public static String CONTLOGPROP = "log.dir";
    public static String CACHEPROP = "cache.dir";
    public static String DBACTIONPROP = "db.action";
    public static String MQSTOREPROP = "mqstore.db.connect";
    public static String LOGFILEPROP = "recovery.log.file";
    public static String WSERVICEPROP = "windows.service.name";
    public static String SSLCERTDIRPROP = "ssl.certificates.dir";
    public static String SSLCERTCHAINPROP = "ssl.certificate.chain";
    public static String WDPROP = "previous.working.directory";
    public static String HOSTDIRPROP = "host.directory";
	 */
	
	private void readContainerProps() throws Exception
	{
		Enumeration propKeys = m_migrateProps.keys();
		while (propKeys.hasMoreElements())
		{
			String key = (String)propKeys.nextElement();
			if (key.indexOf(".") > -1)
			{
				String containerName = key.substring(0, key.indexOf("."));
				String containerPropName = key.substring(key.indexOf(".") + 1);
				debug("readContainerProps containerName == " + containerName
						+ " containerPropName == " + containerPropName);
				// Is this a container prop
				if (containerPropName.startsWith("central.connection")
						|| containerPropName.equals(BOOTFILEPWDPROP)
						|| containerPropName.equals(CACHEPROP)
						|| containerPropName.equals(CONTLOGPROP)
						|| containerPropName.equals(DBACTIONPROP)
						|| containerPropName.equals(MQSTOREPROP)
						|| containerPropName.equals(LOGFILEPROP)
						|| containerPropName.equals(SSLCERTDIRPROP)
						|| containerPropName.equals(SSLCERTCHAINPROP)
						|| containerPropName.equals(WSERVICEPROP)
						|| containerPropName.equals(WDPROP)
						|| containerPropName.equals(HOSTDIRPROP)) 
				{
					HashMap containerProps = getContainerProps(containerName);
					String propValue = (String)m_migrateProps.get(key);
					// check that the placeholder <containersDir> has been replaced with something
					if (propValue.indexOf(CONTAINERSDIRPLACEHOLDER) > -1)
                    {
                        throw new MigrationException("The property " + key + " still contains the place holder " + CONTAINERSDIRPLACEHOLDER + ".");
                    }
					containerProps.put(containerPropName, m_migrateProps.get(key));
				}
			}
		}
	}
	
	private void checkContainerProperties() throws Exception
	{
		debug("checkContainerProperties");
		Iterator containerIT = m_containerProps.keySet().iterator();
		while (containerIT.hasNext())
		{
			String containerName = (String) containerIT.next();
			File containerWD = ContainerSetup.findContainerDir(m_containersDir, m_migrateProps.getProperty(DOMAINPROP), containerName);
			HashMap containerProps = (HashMap)m_containerProps.get(containerName);
			if (containerProps != null)
			{
				String dbAction = (String) containerProps.get(DBACTIONPROP);
				if (dbAction != null) 
				{
					if ("copy".equalsIgnoreCase(dbAction) || "move".equalsIgnoreCase(dbAction)) 
					{
						String dbConnect = (String) containerProps.get(MQSTOREPROP);
						String logFile = (String) containerProps.get(LOGFILEPROP);
						if (dbConnect == null)
                        {
                            throw new MigrationException("db.action of " + dbAction + " was specified for container "
											+ containerName + " but mqstore.db.connect value was not specified.");
                        }
                        else if (logFile == null)
                        {
                            throw new MigrationException("db.action of " + dbAction + " was specified for container "
											+ containerName + " but recvery.log.file was not specified.");
                        }

						try 
						{
							File dbConnectFile = new File(dbConnect);
							if (!dbConnectFile.isAbsolute())
                            {
                                dbConnectFile = new File(containerWD, dbConnect);
                            }
							if (!testValidFile(dbConnectFile))
                            {
                                throw new MigrationException(MQSTOREPROP + " value " + dbConnect + " of "
												+ containerName + " is invalid for upgrade.");
                            }
						} 
						catch (Exception e) 
						{
							throw new MigrationException(MQSTOREPROP + " value " + dbConnect + " of "
											+ containerName + " is invalid for upgrade.");
						}

						try						
						{
							File logFileDir = new File(logFile);
							if (!logFileDir.isAbsolute())
                            {
                                logFileDir = new File(containerWD, logFile);
                            }
							if (!testValidFile(logFileDir))
                            {
                                throw new MigrationException(LOGFILEPROP + " value " + logFile + " of "
											 + containerName + " is invalid for upgrade.");
                            }
						} 
						catch (Exception e) 
						{
							throw new MigrationException(LOGFILEPROP + " value " + logFile + " of "
											+ containerName + " is invalid for upgrade.");
						}
					}
				}
			}
		}
	}
	
	private boolean testValidFile(File test)
	{
	    return (test.exists() || test.mkdirs());
	}
	
	private void checkUpgradeList() throws Exception
	{
		debug("checkUpgradeList");
		Iterator upgradeIT = m_upgradeList.iterator();
		while (upgradeIT.hasNext())
		{
			String containerID = (String)upgradeIT.next();
			IDirElement container = m_fileSystemService.getFSElement(containerID, false);
			if (container == null)
            {
                throw new MigrationException(containerID + " is not found in the DS");
            }
			HashMap<String, String> containerProps = getContainerProps(getContainerName(containerID));
			String relVersion = container.getIdentity().getReleaseVersion();
			if (!relVersion.equals(CURRENT_RELEASE_VERSION))
            {
                containerProps.put("RELEASE_VERSION", relVersion);
            }
			//if (container.getIdentity().getReleaseVersion().equals(CURRENT_RELEASE_VERSION))
				//throw new MigrationException(containerID + " has already been upgraded. Remove it from upgrade.list in the properties file and restart the upgrade");
			checkComponents(container, false, false);
		}				
	}
	
	private String doMigration() throws Exception
	{
		// do the domain manager first
		m_configUpgrade = new ConfigUpgrade(m_migrateProps, m_containerProps);
		
		if (m_DMElement != null)
		{
			debug("doMigration about to call migrateDomainManager");
			String releaseVersion = m_DMElement.getIdentity().getReleaseVersion();
			migrateDomainManager();
			debug("doMigration call to migrateDomainManager done");
		// open a new connection to the DS to upgrade all other containers
		    m_ds = new DirectoryService((IDirElement)m_migrateProps.get(DMDSELEMENTPROP), null, m_migrateProps.getProperty(DSHOSTPROP), willMigrate(releaseVersion));
		    m_fileSystemService = m_ds.getFileSystemService();
		}
        else
        {
            // connect to migrate and upgrade non DM containers
			connect(null);
        }
		m_configUpgrade.setConnection(m_fileSystemService);
		// the other root containers
		for (int containerIndex=0; containerIndex < m_rootList.size(); containerIndex++)
        {
            m_fileSystemService.suspendChangeNotifications((String)m_rootList.get(containerIndex), null);
        }
		for (int containerIndex=0; containerIndex < m_upgradeList.size(); containerIndex++)
        {
            m_fileSystemService.suspendChangeNotifications((String)m_upgradeList.get(containerIndex), null);
        }
		debug("doMigration about to call upgradeRootContainers");
		String message = upgradeRootContainers();
		message = Utils.addToUpgradeMessage(message, upgradeContainers());
		message = Utils.addToUpgradeMessage(message, upgradeTemplates());
		debug("doMigration about to call migrateRootContainers");
		migrateRootContainers();
		debug("doMigration about to call disconnect");
		disconnect();
		// if we're running from a Sonic launcher install, restart root containers that
		// were stopped during this migration
		if (m_restartContainers.size() > 0 && !(new File(m_archives)).exists())
        {
            restartRootContainers();
            // save it if we're running from a DM install regardless, because the ant
            // task will call Migrate with the file.
        }
        else if ((new File(m_archives)).exists())
		{
			debug("doMigration calling saveProps");
			saveProps();
		}
		return message;
	}
	
	private void migrateDomainManager() throws Exception
	{
		debug("Migrating domain manager");
		if (isOnlineConnection())
        {
            shutdownDM();
        }
		String releaseVersion = m_DMElement.getIdentity().getReleaseVersion();
		if (willMigrate(releaseVersion))
		{
		    copyDSStorage();
		    debug("About to migrate broker DB");
	        migrateDMBrokerDB(); 
		}
		// save the DS version for the upgrade
		m_migrateProps.put(DSVERSIONPROP, releaseVersion);
        upgradeDomainManager();
		// we need a connection before to call setup or undoReadOnlyBackup
		
		m_ds = new DirectoryService((IDirElement)m_migrateProps.get(DMDSELEMENTPROP), null, m_migrateProps.getProperty(DSHOSTPROP), willMigrate(releaseVersion));
		m_fileSystemService = m_ds.getFileSystemService();
		if (willMigrate(releaseVersion))
        {
		    File wd = setupDomainManager();
		// and, wickedly, the property ds.working.dir which the install ant scripts
		// expect is not the working directory of the DS, but the working
		// directory of the domain manager container; where the DS boot file
		// will be found, and so, we save this working directory to be written out
		// to the property file....
		    m_migrateProps.setProperty(NEWWDPROP, wd.getCanonicalPath());
		    String OS = System.getProperty("os.name").toLowerCase();
    	    if (OS.indexOf("windows") > -1)
            {
                createDMShortcuts(wd);
            }		    
		}
        fixLauncherJREForOldContainer(m_DMElement.getIdentity().getName(), releaseVersion);
        undoReadOnlyBackup();
	    disconnect();
	}

	
	// if we need to shutdown the domain manager, we want to shutdown other containers
	// that we're going to migrate on this host, otherwise, we won't have a mgmt connection
	// to do it from
	// If we don't have a mgmt connection right now, nothing gets shutdown, and
	// nothing gets restarted afterwards, as we don't know what was running when we 
	// started the upgrade
	private void shutdownDM() throws Exception
	{
		String dmID = m_DMElement.getIdentity().getName();
		Iterator rootListIT = m_rootList.iterator();
		while (rootListIT.hasNext())
		{
			//the DM gets shutdown last
			String containerID = (String)rootListIT.next();
			if (!containerID.equals(dmID))
            {
                shutdownContainer(containerID, true);
            }
		}
		Iterator upgradeIT = m_upgradeList.iterator();
		while (upgradeIT.hasNext())
		{
			String containerID = (String)upgradeIT.next();
			shutdownContainer(containerID, false);
		}
		// now shutdown the DM
		// setBackupReadOnly bypasses the 7.0 backup
		setBackupReadOnly();
		m_restartDM = true;
		String containerObjectName = getAgentObjectName((String)m_DMElement.getAttributes().getAttribute(IContainerConstants.CONTAINER_NAME_ATTR));
		debug("domain manager object name == " + containerObjectName);
		System.out.println("upgrade shutting down domain manager container " + dmID + "...");
		m_ds.shutdownContainer(containerObjectName, 180000);
		System.out.println("upgrade shutting down domain manager container " + dmID + "...done");
		m_ds = null;
	}
	
		private void copyDSStorage() throws Exception
		{
			String domainName = m_migrateProps.getProperty(DOMAINPROP);
			String hostDirectory = (String)m_migrateProps.get(DSHOSTSRCPROP);
			File storageDir = new File(hostDirectory);	
			String destinationDir = m_migrateProps.getProperty(DSHOSTPROP);
			File DSStorageDir = new File(storageDir, domainName);
			// Don't copy if source and destination are the same. The user is keeping
			// the DS storage where they already have it
			if (storageDir.equals(new File(destinationDir)))
            {
                return;
            }
			waitForDMToClose(DSStorageDir);
			System.out.println("upgrade copying DS storage from " + DSStorageDir.getCanonicalPath() + " to " + destinationDir + "/" + domainName + "..." );
			Utils.copyAll(DSStorageDir, new File(destinationDir, domainName));
			System.out.println("upgrade copying DS storage from " + DSStorageDir.getCanonicalPath() + " to " + destinationDir + "/" + domainName + "...done" );
		}

	
	private void waitForDMToClose(File DSStorage) throws Exception
	{
		File lockFile; 
		
		lockFile = new File(DSStorage, "data.odb/lock");
		
		while (lockFile.exists())
		{
			debug("Waiting for DS lock file " + lockFile.getCanonicalPath() + " to be deleted");
			try
			{
			    Thread.sleep(1000);
			}
			catch (InterruptedException intE)
			{
				throw new Exception("Unable to wait for DS to close; aborting upgrade");
			}
		}
	}
	
	private void migrateDMBrokerDB() throws Exception
	{
		migrateBrokerDB(m_DMElement.getIdentity().getName(), true, true);
	}
	
	private void upgradeDomainManager() throws Exception
	{
		m_migrateProps.setProperty(DMELEMENTPROP, m_DMElement.getIdentity().getName());
		System.out.println("upgrade upgrading the domain manager configuration and its components...");
		m_configUpgrade.upgradeDM(m_DMElement);
		System.out.println("upgrade upgrading the domain manager configuration and its components...done");
	}
	
	private File setupDomainManager() throws Exception
	{
		String containerName = (String)m_DMElement.getAttributes().getAttribute(IContainerConstants.CONTAINER_NAME_ATTR);
		Properties non_config_Props = extractSetupProperties(containerName);	
		HashMap containerProps = (HashMap)m_containerProps.get(containerName);
		String version = findLauncherVersion(); //we've already checked that we could find one
		System.out.println("upgrade creating launch scripts and container working area for " + m_DMElement.getIdentity().getName() + "...");
		File wd = ContainerSetup.setupContainer(m_ds.m_dsAdmin, m_DMElement.getIdentity().getName(), (String)containerProps.get(DSELEMENTPROP), version, m_sonicHome, non_config_Props, (String)containerProps.get(BOOTFILEPWDPROP));
		System.out.println("upgrade creating launch scripts and container working area for " + m_DMElement.getIdentity().getName() + "...done");
		return wd;
	}

    //Earlier this method was executed only for the 8.0 product version since it had different JRE (1.5)
    //from 10.0 we ship 1.7 hence this method has to be executed for all the product versions.
    private void fixLauncherJREForOldContainer(String containerID, String releaseVersion) throws Exception
    {
        IDirElement container = m_fileSystemService.getFSElement(containerID, false);
        if(releaseVersion == null)
        {
            releaseVersion = container.getIdentity().getReleaseVersion();
        }
        if (releaseVersion.compareTo(FIRST_CIPGRADE_RELEASE_VERSION) > 0)
        {
            String OS = System.getProperty("os.name").toLowerCase();
            String containerName = getContainerName(containerID);
            String contWorkingDir = m_containersDir + File.separator +  m_migrateProps.getProperty(DOMAINPROP) + "." + containerName;
            File scriptFile = null;
            FileWriter fw = null;
            String jreCommandStr ="SONIC_LAUNCHER_JRE=" + System.getProperty("sonicmq.jre");
            debug("jreCommandStr=" + jreCommandStr);
            try
            {
                if (OS.indexOf("windows") > -1)
                {
                    debug("Sciprt File=" + contWorkingDir + File.separator + "set_launcher_jre.bat");
                    scriptFile = new File(contWorkingDir + File.separator + "set_launcher_jre.bat");
                    if (scriptFile != null && scriptFile.exists())
                    {
                        fw = new FileWriter(scriptFile);
                        debug("New File Content=" + "set " + jreCommandStr);
                        fw.write("set " + jreCommandStr);
                    }

                }
                else
                {
                    debug("Sciprt File=" + contWorkingDir + File.separator + "set_launcher_jre.sh");
                    scriptFile = new File(contWorkingDir + File.separator + "set_launcher_jre.sh");
                    if (scriptFile != null && scriptFile.exists())
                    {
                        fw = new FileWriter(scriptFile);
                        StringBuilder stringBuilder = new StringBuilder("#!/bin/sh");
                        stringBuilder.append(System.getProperty("line.separator"));
                        stringBuilder.append(jreCommandStr);
                        debug("New File Content=" + stringBuilder.toString());
                        fw.write(stringBuilder.toString());
                    }
                }
            }
            finally
            {
                if(fw != null)
                {
                    fw.close();
                }
            }
        }
    }

    private void createDMShortcuts(File wd) throws Exception
	{
		ShortcutCreator creator = new ShortcutCreator(m_debug);
	    
	    File sonicHome = new File(m_sonicHome);
	    File sonicmqHome = new File(sonicHome, SONICMQDIR);
	    // we checked already if we could find sonicsw.properties
	    String programGroup = SONICSW_PROPS.getProperty(INSTALL_PROGRAM_GROUP);
	    if (programGroup != null)
	    {
	    	File launchScriptFile = new File(wd, LAUNCH_SCRIPT);
	    	File shutdownScriptFile = new File(wd, SHUTDOWN_SCRIPT);
	    	if (!launchScriptFile.exists() || !shutdownScriptFile.exists())
            {
                throw new MigrationException("Upgrade could not find the scripts for the domain manager; the container setup step did not finish successfully.");
            }
	    	File iconFile = new File(sonicmqHome, MQCONTAINER_ICON_LOCATION);
	    	String iconFileName;
	    	if (!iconFile.exists())
            {
                iconFileName = null;
            }
            else
            {
                iconFileName = iconFile.getCanonicalPath();
            }
	        creator.createShortcut(launchScriptFile.getCanonicalPath(), programGroup,
	            LAUNCH_VERBIAGE, iconFileName, true);
	        creator.createShortcut(shutdownScriptFile.getCanonicalPath(), programGroup,
		        SHUTDOWN_VERBIAGE, iconFileName, true);
	    }
    }
	
	private void undoReadOnlyBackup() throws Exception
	{
		// while we're upgrading and migrating the primary domain manager, undo the
		// read only backup mode used during the upgrade.
		String containerName = (String)m_DMElement.getAttributes().getAttribute(IContainerConstants.CONTAINER_NAME_ATTR);
	    HashMap props = getContainerProps(containerName);
	    String dsID = (String)props.get(DSELEMENTPROP);
	    IDirElement dsElement = m_fileSystemService.getFSElement(dsID, true);
	    IAttributeSet dsTopSet = dsElement.getAttributes();
		IAttributeSet replParams = (IAttributeSet)dsTopSet.getAttribute(IDirectoryServiceConstants.REPLICATION_PARAMETERS_ATTR);
	    if (replParams == null)
         {
            return; // not fault tolerant
        }
	    replParams.setBooleanAttribute(IDirectoryServiceConstants.BACKUP_FAILOVER_READ_ONLY_ATTR, Boolean.FALSE);
	    m_fileSystemService.updateFSElement((IDeltaDirElement)dsElement.doneUpdate());
	}
	
	private Properties extractSetupProperties(String containerName)
	{
		Properties returnThis = new Properties();
		HashMap containerProps = (HashMap)m_containerProps.get(containerName);
		if (containerProps != null)
		{
			// a container could end up with no properties in the property file,
			// and so its properties could be null. The example is the instance of
			// a template, where the user has not changed the cache and log
			// directories. Odd, but possible, especially if each instance is 
			// running on a different host, there's no collision with "."
		    String WSName = (String)containerProps.get(WSERVICEPROP);
		    if (WSName != null)
            {
                returnThis.setProperty(WSNAMEPROP, WSName);
            }
		}
		return returnThis;
	}
	
	private HashMap getContainerProps(String containerName)
	{
		HashMap props = (HashMap)m_containerProps.get(containerName);
		if (props == null)
		{
			props = new HashMap();
			m_containerProps.put(containerName, props);
		}
		return props;
	}
	
	// migrate all non DM root containers and their components
	// For 8.5 - If the original version is 8.0, no migration is
	// required
	private void migrateRootContainers() throws Exception
	{
		Iterator rootContainers = m_rootList.iterator();
		while (rootContainers.hasNext())
		{
			
			String containerID = (String)rootContainers.next();
			HashMap<String, String> containerProps = getContainerProps(getContainerName(containerID));
			String relVersion = containerProps.get("RELEASE_VERSION");
			debug("migrateRootContainers container " + containerID + " relVersion == " + relVersion);
			boolean migrate = relVersion != null ? willMigrate(relVersion) : true;
			if (m_DMElement == null || !containerID.equals(m_DMElement.getIdentity().getName()))
			{
			    if (migrate)
			    {
				    debug("migrateRootContainers migrating " + containerID);
				    migrateContainer(containerID);	
			    }
                else
                {
                    recycleContainer(containerID);
                }
			}
		}
	}
	
	// upgrade all non DM root containers and their components
	private String upgradeRootContainers() throws Exception
	{
		String message = null;
		Iterator rootContainers = m_rootList.iterator();
		while (rootContainers.hasNext())
		{
			
			String containerID = (String)rootContainers.next();
			debug("upgradeRootContainers iterating over " + containerID);
			if (m_DMElement == null || !containerID.equals(m_DMElement.getIdentity().getName()))
            {
                message = Utils.addToUpgradeMessage(message, upgradeContainer(containerID));
            }			
		}
		return message;
	}
	
	// This is for root containers that are not the DM. 
	private void migrateContainer(String containerID) throws Exception
	{
	    
		setupContainer(containerID);
		debug("migrateContainer about to call migrateBrokerDB for " + containerID);
		migrateBrokerDB(containerID, false, true);
	}
	
	private void setupContainer(String containerID) throws Exception
	{
		String containerName = getContainerName(containerID);
		Properties non_config_Props = extractSetupProperties(containerName);
		HashMap containerProps = (HashMap)m_containerProps.get(containerName);
		debug("setupContainer " + containerID);
		boolean shutDownService = true;
		if (non_config_Props.get(WSNAMEPROP) != null)
		{
			shutDownService = shutdownWService((String)non_config_Props.get(WSNAMEPROP), containerID);
		// remove the windows service name from the properties - if we were not able
		// to stop the service, do not write over it.
		    if (!shutDownService)
            {
                non_config_Props.remove(WSNAMEPROP);
            }
		}
        else
        {
            shutdownContainer(containerID, true);
        } 
		String dsElementPropValue = null;
		String bootFilePWD = null;
		// If the user is not changing anything about the container, so they erased the values in
		// the property file, its containerProps could be null
		if (containerProps != null)
		{
			dsElementPropValue = (String)containerProps.get(DSELEMENTPROP);
			bootFilePWD = (String)containerProps.get(BOOTFILEPWDPROP);
		}
		System.out.println("upgrade creating launch scripts and container working area for " + containerID + "...");
        String version = findLauncherVersion(); //we've already checked that we could find one
		ContainerSetup.setupContainer(m_ds.m_dsAdmin, containerID, dsElementPropValue, version, m_sonicHome, non_config_Props, bootFilePWD);
		System.out.println("upgrade creating launch scripts and container working area for " + containerID + "...done");
	}
	
    private void migrateBrokerDB(String containerID, boolean dm, boolean root)
    throws Exception
    {
        // find out if there's a broker DB to move. Its properties would have the broker
        // properties set
        debug("Entering migrateBrokerDB for " + containerID);
        String containerName = getContainerName(containerID);
        HashMap containerProps = (HashMap)m_containerProps.get(containerName);
        if (containerProps == null)
         {
            return; // nothing to do
        }
        String dbAction = (String)containerProps.get(DBACTIONPROP);
        debug("migrateBrokerDB, action == " + dbAction);
        if (dbAction != null)
        {
            dbAction = ((String)containerProps.get(DBACTIONPROP)).toLowerCase();
            if (dbAction.equals("copy") || dbAction.equals("move"))
            {
                // if the container is running, it's time to shut it down
                if (m_fileSystemService != null && m_fileSystemService instanceof DirectoryServiceProxy)
                {
                    // find out if the container is running
                    // The DM was already shutdown so no need to check if it's running
                    // if the container is a root container, no need to
                    // shut it down, because we already tried
                    String containerObjectName = getAgentObjectName(containerName);
                    if (!dm && !root && pingContainer(containerObjectName))
                    {
                        System.out.println("upgrade shutting down " + containerID + " to copy/move its broker database...");
                        boolean shutdown = shutdownContainer(containerID, root);
                        if (shutdown)
                        {
                            System.out.println("upgrade shutting down " + containerID + " to copy/move its broker database...done");
                        }
                        else
                        {
                            System.out.println("upgrade shutting down " + containerID + " to copy/move its broker database failed. Database and logs will not be copied");
                            return;
                        }
                    }
                    // check whether the shutdown was successful for a root container
                    // shutdown earlier
                    // containerInfo will be of the form methodForStartup,logDirectory
                    String containerInfo = m_restartContainers.get(containerID);
                    String restartStatus = (String)m_restartContainers.get(containerID);
                    if (restartStatus != null && restartStatus.startsWith("ERROR"))
                    {
                        System.out.println("upgrade shutting down " + containerID + " had failed earlier. Database and logs will not be copied");
                        return;
                    }
                }
                File containerDir = ContainerSetup.findContainerDir(m_containersDir, m_migrateProps.getProperty(DOMAINPROP), containerName);
                String mqStore = (String)containerProps.get(MQSTORESRCPROP);
                String recoveryLog = (String)containerProps.get(LOGFILESRCPROP);
                File brokerDBDir = new File(mqStore);
                String mqStoreNew = (String)containerProps.get(MQSTOREPROP);
                File mqStoreNewFile = new File(mqStoreNew);
                if (!mqStoreNewFile.isAbsolute())
                {
                    mqStoreNewFile = new File(containerDir, mqStoreNew);
                }
                System.out.println("upgrade copying broker DB from " + brokerDBDir.getCanonicalPath() + " to " + mqStoreNewFile.getCanonicalPath() + "...");
                Utils.copyAll(brokerDBDir, mqStoreNewFile);
                System.out.println("upgrade copying broker DB from " + brokerDBDir.getCanonicalPath() + " to " + mqStoreNewFile.getCanonicalPath() + "...done");
                File recoveryLogDir = new File(recoveryLog);
                String logNew = (String)containerProps.get(LOGFILEPROP);
                File logNewFile = new File(logNew);
                if (!logNewFile.isAbsolute())
                {
                    logNewFile = new File(containerDir, logNew);
                }
                System.out.println("upgrade copying broker log from " + recoveryLogDir.getCanonicalPath() + " to " + logNewFile.getCanonicalPath() + "...");
                Utils.copyAll(recoveryLogDir, logNewFile);
                System.out.println("upgrade copying broker log from " + recoveryLogDir.getCanonicalPath() + " to " + logNewFile.getCanonicalPath() + "...done");
                if (dbAction.equals("move"))
                // delete source
                {
                    System.out.println("upgrade removing broker log from " + recoveryLogDir.getCanonicalPath() + "...");
                    Utils.recursiveDeleteDirectory(recoveryLogDir);
                    System.out.println("upgrade removing broker log from " + recoveryLogDir.getCanonicalPath() + "...done");
                    System.out.println("upgrade removing broker database from " + brokerDBDir.getCanonicalPath() + "...");
                    Utils.recursiveDeleteDirectory(brokerDBDir);
                    System.out.println("upgrade removing broker database from " + brokerDBDir.getCanonicalPath() + "...done");
                }
            }
        }
    }
	
	private String upgradeContainers() throws Exception
	{
		String message = null;
		Iterator containers = m_upgradeList.iterator();
		while (containers.hasNext())
		{			
			String containerID = (String)containers.next();
			debug("upgradeContainers iterating over " + containerID);	
		    message = Utils.addToUpgradeMessage(message, upgradeContainer(containerID));
		    HashMap<String, String> containerProps = getContainerProps(getContainerName(containerID));
			String relVersion = containerProps.get("RELEASE_VERSION");
			boolean migrate = relVersion != null ? willMigrate(relVersion) : true;
			debug("upgradeContainers containerId " + containerID + " RELEASE_VERSION " + relVersion);
			if (migrate)
            {
                migrateBrokerDB(containerID, false, false);
            }							
		}
		return message;
	}
	
	// only non container templates end up in the template list
	private String upgradeTemplates() throws Exception
	{
		String message = null;
		Iterator templates = m_templateList.iterator();
		while (templates.hasNext())
		{			
			String elID = (String)templates.next();
			debug("upgradeTemplates iterating over " + elID);
		    message = Utils.addToUpgradeMessage(message, m_configUpgrade.upgradeConfig(elID));							
		}
		return message;
	}
	
	private String upgradeContainer(String containerID) throws Exception
	{
		System.out.println("upgrade upgrading container configuration " + containerID + "...");
        fixLauncherJREForOldContainer(containerID, null);
        String message = m_configUpgrade.upgradeContainer(containerID);
		System.out.println("upgrade upgrading container configuration " + containerID + "...done");
		return message;
	}
	
	private void restartRootContainers() throws Exception
	{
		// if MGMTCONNURLSPROP was saved, it means we need to try to find the
		// backup DM and shut it down before restarting the primary DM
	
		String mgmtUrls = m_migrateProps.getProperty(MGMTCONNURLSPROP);
		
		if (mgmtUrls != null)
		{
			boolean connected = false;
			try
			{
			   connect(mgmtUrls);
			   connected = true;
			}
			catch (Exception e)
			{
				// decide that we didn't connect to a backup DS, so no need to shut it down
			}
			if (connected)
			{
				String backupDSContainerName = m_migrateProps.getProperty(BACKUPDSCONTAINERPROP);
				String containerObjectName = getAgentObjectName(backupDSContainerName);
				System.out.println("upgrade shutting down your backup DM before restarting the upgraded primary DM...");
				m_ds.shutdownContainer(containerObjectName, 180000);
				System.out.println("upgrade shutting down your backup DM before restarting the upgraded primary DM...done");
				// the mgmt connection is gone.
				m_ds = null;
			}
		}
		
		// if WDPROP has been saved, we're restarting the DM. Use that value to find
		// the working directory of the DM and its launchcontainer script, otherwise, 
		// either containers are migrated to 8.0 and SONIC_HOME is applicable, or
		// not migrated in which case they're recycled.
		// if WDPROP has been saved, we assume the DM is the only container being restarted OR
		// all containers being restarted share a parent Containers directory. Can't think of a case
		// when this wouldn't be true.
		String ctWD = m_migrateProps.getProperty(WDPROP);
		File sonicHome = ctWD == null ? new File(m_sonicHome) : (new File(ctWD)).getParentFile().getParentFile();
		LauncherDriver driver = new LauncherDriver(sonicHome.getCanonicalPath());
		File containersDir = ContainerSetup.findSonicContainersDir(sonicHome);
		String domainName = m_migrateProps.getProperty(DOMAINPROP);
		for (String containerID : m_restartContainers.keySet())
		{	
			String containerName = getContainerName(containerID);
			String method = (String)m_restartContainers.get(containerID);
			if (method.startsWith("ERROR"))
             {
                continue; // an error shutting down the container is marked so we don't try
            }
			              // to shutdown the container again
			File containerDir = ContainerSetup.findContainerDir(containersDir, domainName, containerName);
			ILauncherContainerDriver containerDriver = driver.getContainer(containerDir.getCanonicalPath());
			String OS = System.getProperty("os.name").toLowerCase();
	    	if (OS.indexOf("windows") > -1)
	    	{
	    		if (method.startsWith("SERVICE"))
	    		{
	    			System.out.println("upgrade launching " + containerID + " as a windows service...");
	    		    containerDriver.launchContainerAsWindowsService();
	    		    System.out.println("upgrade launching " + containerID + " as a windows service...done");
	    		}
	    		else
	    		{
	    			System.out.println("upgrade launching " + containerID + " using its launchcontainer script...");
	    			containerDriver.launchContainerWithBAT();
	    			System.out.println("upgrade launching " + containerID + " using its launchcontainer script...done");
	    		}
	    	}
	    	else
	    	{
	    		System.out.println("upgrade launching " + containerID + " using its launchcontainer script...");
	    		containerDriver.launchContainerWithSH();
	    		System.out.println("upgrade launching " + containerID + " using its launchcontainer script...done");
	    	}
		}
		debug("restartRootContainers exiting");
	}

	
	private void parseRestartContainers()
	{
		String origValue = m_migrateProps.getProperty(RESTARTCONTAINERSPROP);
		// we tested for null already
		StringTokenizer tokenizer = new StringTokenizer(origValue, ";", false);
		while (tokenizer.hasMoreTokens())
		{
		    String containerInfo = tokenizer.nextToken();
			String containerID = containerInfo.substring(0, containerInfo.indexOf(","));
			String startupType = containerInfo.substring(containerInfo.indexOf(",") + 1);
			m_restartContainers.put(containerID, startupType);
		}
	}
	
	private boolean shutdownContainer(String containerID, boolean root) throws Exception
	{
		// This is called for containers other than the DM container
		// if the container is already in the restart list, we already
		// shut it down so leave
		if (m_restartContainers.get(containerID) != null)
		{
			if (m_restartContainers.get(containerID).startsWith("ERROR"))
			{
				System.out.println("shutdownContainer found ERROR already, so returning false");
			    return false;
			}
            else
            {
                return true;
            }
		}
		// if the DM isn't running, we can't do this
		if (m_fileSystemService == null || !(m_fileSystemService instanceof DirectoryServiceProxy))
		{
			System.out.println("shutdownContainer doesn't have an instance of DirectorySErviceProxy");
			return false;
		}
		String dmID = m_DMElement == null ? null : m_DMElement.getIdentity().getName();
		String containerName = getContainerName(containerID);
		String containerObjectName = getAgentObjectName(containerName);
		if (!containerID.equals(dmID))
		{
			HashMap containerProps = getContainerProps(containerName);
			String dbCopyAction = (String)containerProps.get(DBACTIONPROP);
			// shutdown all root containers, and all containers needing a DB migration
			if ( (root || (dbCopyAction != null && (dbCopyAction.equals("COPY") || 
					dbCopyAction.equals("MOVE")))) 
				&& pingContainer(containerObjectName))
			{
				System.out.println("upgrade shutting down " + containerID + (root ? " (will get restarted later)" : "") + "...");
				boolean shutdown = m_ds.shutdownContainer(containerObjectName, 180000);
				if (shutdown)
				{
					System.out.println("upgrade shutting down " + containerID + "...done");
				    if (root)
				    {
					    String windowsService = (String)containerProps.get(WSERVICEPROP);
				        if (windowsService != null)
                        {
                            m_restartContainers.put(containerID, "SERVICE");
                        }
                        else
                        {
                            m_restartContainers.put(containerID, "COMMANDLINE");
                        }
				   }
				   return true;
				}
				else
				{
					   // save a marker that says we already tried to shutdown this container
					   System.out.println("upgrade shutting down " + containerID + " failed. Container will not be restarted after the upgrade");
					   m_restartContainers.put(containerID, "ERROR");
					   return false;
				}
			}
		}
		System.out.println("shutdowncontainer fell down to the end and returning false");
		return false;
	}
	
	private boolean pingContainer(String containerObjectName) throws Exception
	{
		// if there's no connection to the DS, or the connection is a direct
		// connection, we don't know if the container is up, so return false
		debug("pinging " + containerObjectName);
		if (m_ds == null || !(m_fileSystemService instanceof DirectoryServiceProxy))
		{
			debug("Container " + containerObjectName + " is not alive");
			return false;
		}
        else
        {
            return m_ds.pingContainer(containerObjectName);
        }
	}
	
	private boolean shutdownWService(String serviceName, String containerID) throws IOException, DirectoryServiceException
	{
	    // remember the log file location
	    // we know we have a connection to the DS at this point
        String[] exeCmd = new String[]{"sc", "stop", serviceName};
        System.out.println("Calling sc stop service " + serviceName + "...");
        IRemoteExecResult result = ExecUtility.exec(exeCmd, null, (new File(".")).getCanonicalPath(), null);

        // the following will probably never happen, as even when sc stop fails, 
        // it seems to return 0
        if (!result.isSuccessful())
        {
            System.out.println("Tried to stop service " + serviceName + " but finished with an error "+ result.getOutput() + "; windows service will not be installed");
            return false;
        }

        String wsResultText = ExecUtility.execResultOutputToString(result);
        // we look for specific error codes in this text.
        // 1060 means the service is not installed - good
        // 1062 means the service is not running - also good
        // anything else is probably a problem
        int errorIndex = wsResultText.indexOf("FAILED");
        if (errorIndex == -1)
        {
        	// there was actually no error. The service is stopping. 
        	// keep looping until we get error 1062
        	// wait a maximum of 3 minutes
        	System.out.println("Waiting until service is really shutdown...");
        	long originalTime = System.currentTimeMillis();
        	long counter = 0;
        	boolean stopped = false;
        	while (!stopped && counter < 180000)
        	{
        		try 
        		{
					Thread.sleep(5000);
					counter = System.currentTimeMillis() - originalTime;
					System.out.println("Waiting until service is really shutdown...");
				} catch (InterruptedException e) {}
        		IRemoteExecResult secondTry = ExecUtility.exec(exeCmd, null, (new File(".")).getCanonicalPath(), null);
        		String secondText = ExecUtility.execResultOutputToString(secondTry);
        		if (secondText.indexOf("FAILED 1062") > -1)
        		{
        			System.out.println("Calling sc stop service " + serviceName + "...done");
        			m_restartContainers.put(containerID,"SERVICE");
        			return true; // the service has stopped
        		}
        		else if (secondText.indexOf("FAILED 1061") > -1)
        		{
        			// keep going, it's shutting down
        		}
        		else if (secondText.indexOf("FAILED") > -1)
        		{
        			System.out.println("Tried to stop service " + serviceName + " but finished with an error: " 
                			+ IContainer.NEWLINE + secondText + IContainer.NEWLINE +
                			"Windows service will not be installed");
        			return false; // we failed for some other odd reason.
        		}
        	} 
        	// If we get here, the service did not finish stopping
        	System.out.println("Tried to stop service " + serviceName + " for 3 minutes, but stop never finished, Windows service will not be installed");
        	return false;
        }
        else if (wsResultText.indexOf("1060") > -1 || wsResultText.indexOf("1062") > -1)
        {
        	System.out.println("Service was either not installed or already stopped, assuming container was not running, so new service " + serviceName + " will not be started");
        	return true;
        }
        else
        {
        	System.out.println("Tried to stop service " + serviceName + " but finished with an error: " 
        			+ IContainer.NEWLINE + wsResultText + IContainer.NEWLINE +
        			"Windows service will not be installed");
            return false;
        }
	}
	
	private void setBackupReadOnly() throws Exception
	{
		// find out if there is a backup
		String containerName = getContainerName(m_DMElement.getIdentity().getName());
	    HashMap props = getContainerProps(containerName);
	    String dsID = (String)props.get(DSELEMENTPROP);
	    IDirElement dsElement = m_fileSystemService.getFSElement(dsID, true);
	    
	    IAttributeSet dsTopSet = dsElement.getAttributes();
	    IAttributeSet refs = (IAttributeSet)dsTopSet.getAttribute(IDirectoryServiceConstants.CONFIG_ELEMENT_REFERENCES_ATTR);
	    if (refs != null)  // FT DS?
	    {
	    	Reference backup = (Reference)refs.getAttribute(IDirectoryServiceConstants.BACKUP_CONFIG_ELEMENT_REF_ATTR);
	    	if (backup != null)
	    	{
	    		boolean backupIsRunning = false;
	    		String backupObjectName = null;
	    		// is the backup running?
	    		ArrayList containerIds = Utils.findContainersForComponent(backup.getElementName(), m_fileSystemService);
	    		// typically containerIds will be only one, but we'll just find the first one that
	    		// is running
	    		Iterator idsIterator = containerIds.iterator();
	    		while (idsIterator.hasNext())
	    		{
	    			String backupCID = (String)idsIterator.next();
	    			String backupCName = getContainerName(backupCID);
	    			backupObjectName = getAgentObjectName(backupCName);
	    			backupIsRunning = pingContainer(backupObjectName);
	    		}
	    		if (backupIsRunning)
	    		{
	    		    IAttributeSet replParams = (IAttributeSet)dsTopSet.getAttribute(IDirectoryServiceConstants.REPLICATION_PARAMETERS_ATTR);
	    		    if (replParams == null)
                    {
                        replParams = dsTopSet.createAttributeSet(IDirectoryServiceConstants.REPLICATION_PARAMETERS_ATTR);
                    }
	    		    replParams.setBooleanAttribute(IDirectoryServiceConstants.BACKUP_FAILOVER_READ_ONLY_ATTR, Boolean.TRUE);
	    		    m_fileSystemService.updateFSElement((IDeltaDirElement)dsElement.doneUpdate());
	    		        // now restart the backup to have the setting take effect
	    		    System.out.println("upgrade restarting the backup DM after setting failover read only mode...");
	    		    m_ds.restartContainer(backupObjectName, getAMObjectName(containerName));
	    		    System.out.println("upgrade restarting the backup DM after setting failover read only mode...done");
                }	    		
	        }
	    }
	}
	
	private String getContainerName(String containerID)
	{
		return containerID.substring(containerID.lastIndexOf(IMFDirectories.MF_DIR_SEPARATOR) + 1);
	}
	
	private String getAgentObjectName(String containerName)
	{
		return m_migrateProps.getProperty(DOMAINPROP) + "." +  containerName+ ":ID=" + IAgentProxy.ID;
	}
	
	private String getAMObjectName(String containerName)
	{
		return m_migrateProps.getProperty(DOMAINPROP) + "." +  IAgentManagerProxy.GLOBAL_ID+ ":ID=" + IAgentManagerProxy.GLOBAL_ID;
	}
	 
	private void printWarning(String warning)
	{
		System.err.println();
		System.err.println("*****************************  Upgrade Message ************************");
		System.err.println();
		System.err.println(warning);
		System.err.println();
		System.err.println("***********************************************************************");
	}
	
	// is the DS that we're upgrading running? It's possible (though not likely) that
	// there is more than one DS instance and we should handle them, without breaking
	// If there is no management connection, we can't really check.
	private boolean isOnlineConnection() throws Exception
	{
		return m_fileSystemService != null && m_fileSystemService instanceof DirectoryServiceProxy;
	}
	
	private String findLauncherVersion()
	{
		File sonicHome = new File(m_sonicHome);
		File launcherDir = new File(sonicHome, "Launcher");
		File[] subs = launcherDir.listFiles();
		for (int i=0;i<subs.length;i++)
		{
			File fileOrDir = subs[i];
			if (fileOrDir.isDirectory() && fileOrDir.getName().startsWith(CURRENT_PRODUCT_VERSION))
            {
                return fileOrDir.getName();
            }
		}
		return null;
	}
	
	// Check that we have DBService control codes if required, check that the appropriate
	// products have been installed to support the services
	private void checkXQContainer(IElementIdentity compID) throws Exception
	{
		if (!m_isESBInstalled)
        {
            throw new MigrationException("ESB has not been installed so we cannot upgrade "
							+ compID.getName());
        }
		// find the services and check for BPEL and DBService
		// switch to the storage API for services
		IDirectoryAdminService dsAdmin = m_ds.getAdminService();
		String storageXQName = m_fileSystemService.logicalToStorage(compID.getName());
		IDirElement xqContainer = dsAdmin.getElement(storageXQName, false);
		IAttributeSet servicesSet = (IAttributeSet) xqContainer.getAttributes().getAttribute("services");
		if (servicesSet != null)
        {
            checkServices(compID.getName(), servicesSet, dsAdmin);
        }
	}
	
	// if we have relative paths in the broker's DB and/or log locations, we must know
	// the working directory of the pre-upgrade container in order to copy the logs and
	// database. We throw an error if we don't have that information. We also check that these
	// paths can be found on the host
	private void checkBrokerComponent(IElementIdentity compID, IElementIdentity containerID) throws Exception
	{
		String relVersion = compID.getReleaseVersion();
		String type = compID.getType();
		// just check paths if it has been upgraded already
		if (type.equals("MQ_BROKER") && m_brokerCN == null && !relVersion.equals(CURRENT_RELEASE_VERSION))
        {
            throw new MigrationException("Broker control code is not known, so cannot proceed with the upgrade of " + compID.getName());
        }	
		
		// check the paths to the database and log files for brokers on this host
		IDirElement brokerEl = m_fileSystemService.getFSElement(compID.getName(), false);
	    IAttributeSet DBParams = (IAttributeSet)brokerEl.getAttributes().getAttribute("BROKER_DATABASE_PARAMETERS");
		IAttributeSet logParam = (IAttributeSet)brokerEl.getAttributes().getAttribute("BROKER_RECOVERY_LOG_PARAMETERS");
		String containerName = containerID.getNameComponents()[containerID.getNameComponents().length-1];
		HashMap containerProps = getContainerProps(containerName);
		String dbAction = (String)containerProps.get(DBACTIONPROP);
		// dbaction will be null if ConfigDependencies finds that this broker
		// is probably running on some other host. The user can always change this,
		// but the default is to assume that for FT brokers, they're running on 
		// different hosts
		if (dbAction == null)
        {
            return;
        }
		String mqStore = (String)DBParams.getAttribute("MQSTORE_DB_CONNECT");
		debug("Migrate.checkComponent broker's MQSTORE_DB_CONNECT == " + mqStore);
		String recoveryLog = (String)logParam.getAttribute("RECOVERY_LOG_PATH");
		debug("Migrate.checkComponent broker's RECOVERY_LOG_PATH == " + recoveryLog);
		if (mqStore == null || mqStore.length() == 0)
        {
            mqStore = "./SonicMQStore";
        }
		if (recoveryLog == null || recoveryLog.length() == 0)
        {
            recoveryLog = "./log";
        }
		File mqStoreFile = new File(mqStore);
		File recoveryLogFile = new File(recoveryLog);
		if (!mqStoreFile.isAbsolute() ||
		    !recoveryLogFile.isAbsolute())	
		{
			String prevWD = checkWD(containerProps, containerName);	
			if (!mqStoreFile.isAbsolute())
			{
			    mqStoreFile = new File(prevWD, mqStore);
			    if (!mqStoreFile.exists())
                {
                    throw new MigrationException("The source broker database directory of broker component " + brokerEl.getIdentity().getName() + " has been computed to be " + mqStoreFile.getCanonicalPath()
			    			+ ". This path cannot be found. Double check the setting of previous.working.directory for container " + containerName);
                }
			}
			if (!recoveryLogFile.isAbsolute())
			{
				recoveryLogFile = new File(prevWD, recoveryLog);
				if (!recoveryLogFile.exists())
                {
                    throw new MigrationException("The source recovery log file location of broker component " + brokerEl.getIdentity().getName() + " has a value of " + recoveryLogFile.getCanonicalPath()
			    			+ ". This path cannot be found. Double check the setting of previous.working.directory for container " + containerName);
                }
			}
		}
		containerProps.put(MQSTORESRCPROP, mqStoreFile.getCanonicalPath());
		containerProps.put(LOGFILESRCPROP, recoveryLogFile.getCanonicalPath());	
		containerProps.put(BROKERPATHPROP, brokerEl.getIdentity().getName());
		if (dbAction.equals("LEAVE"))
		{
			// the source and the destination are the same, and the values read in from the
			// property file are ignored. Save the computed src values as the destination values
			containerProps.put(MQSTOREPROP, mqStoreFile.getCanonicalPath());
			containerProps.put(LOGFILEPROP, recoveryLogFile.getCanonicalPath());
		}
	}
	// Make sure we can find the DS storage for pre 8.0 DMs. If the path is a relative path, we must know the
	// working directory of the container.
	private void checkDSComponent(IElementIdentity compID, IDirElement container, boolean rootContainer) throws Exception
	{
		IElementIdentity containerID = container.getIdentity();
		String containerName = containerID.getNameComponents()[containerID.getNameComponents().length-1];
		HashMap containerProps = getContainerProps(containerName);
		IDirElement dsEl = m_fileSystemService.getFSElement(compID.getName(), false);
		String hostDir = (String)dsEl.getAttributes().getAttribute("HOST_DIRECTORY");
		String domainName = m_migrateProps.getProperty(DOMAINPROP);;
					
		if (rootContainer) // calculate the host directory only for the root DS,
							// not other instances
		{
			containerProps.put(DSELEMENTPROP, compID.getName());
			m_migrateProps.put(DMDSELEMENTPROP, dsEl);

			// check that we can find the source storage
			if (hostDir == null || hostDir.length() == 0)
            {
                hostDir = ".";
            }
			File hostDirFile = new File(hostDir);
			if (!hostDirFile.isAbsolute()) 
			{
			    String prevWD = checkWD(containerProps, containerName);
                hostDirFile = new File(prevWD, hostDir);
			}
			// if this is an upgrade from 8.0 or beyond, check that we can find the launchcontainer script
			if (!willMigrate(compID.getReleaseVersion()))
            {
                checkLaunchContainer(containerProps, containerName);
            }
			
			// save the source storage
			debug("checkDSComponent setting DSHOSTSRCPROP to "
					+ hostDirFile.getCanonicalPath());
			// make sure hostDirFile can be found
			
			if (!(new File(hostDirFile, domainName)).exists())
            {
                throw new MigrationException("The directory service storage cannot be found in " + hostDirFile.getCanonicalPath());
            }
			m_migrateProps.put(DSHOSTSRCPROP, hostDirFile.getCanonicalPath());
			File newDSStorageFile = null;
            if (willMigrate(compID.getReleaseVersion()))
            {
			// check the destination
			    newDSStorageFile = checkNewDSHostProp(containerProps,
					containerName);
            }
			if (newDSStorageFile == null) 
			{
				// the user did not specify a change to the DS storage, or
				// this is an upgrade from 8.0 or beyond, so use the source
				// the source
				debug("checkDSComponent setting DSHOSTPROP to "
						+ hostDirFile.getCanonicalPath());
				m_migrateProps.put(DSHOSTPROP, hostDirFile.getCanonicalPath());
				newDSStorageFile = hostDirFile;
			}
			// finally, if the src and destination are not the same, and
			// there are files
			// in the destination directory, throw an error CR 75272
			File newDomainDir = new File(newDSStorageFile,
					m_migrateProps.getProperty(DOMAINPROP));
			if (!hostDirFile.equals(newDSStorageFile) && newDomainDir.exists()
					&& (newDomainDir.list().length > 0))
            {
                throw new MigrationException(
						"There are files in the new storage area "
								+ newDomainDir.getCanonicalPath()
								+ " for the directory service. Remove these files and restart the upgrade.");
            }

		}
	}
	
	private void checkBackupDSComponent(IElementIdentity compID, IElementIdentity containerID) throws Exception
	{
		String containerName = containerID.getNameComponents()[containerID.getNameComponents().length-1];
		HashMap containerProps = getContainerProps(containerName);
		containerProps.put(DSELEMENTPROP, compID.getName());
		// save the connection URLS of the primary, to save them and try and
		// shutdown the backup before restarting the primary
		if (m_DMElement != null)
		{
		    String urls = (String)((IAttributeSet)m_DMElement.getAttributes().getAttribute(IContainerConstants.CONNECTION_ATTR)).getAttribute(IContainerConstants.CONNECTIONURLS_ATTR);
		    m_migrateProps.put(MGMTCONNURLSPROP, urls);
		    m_migrateProps.put(BACKUPDSCONTAINERPROP, containerName);
		}
	}
	
	// add the list of containers to be restarted to the saved properties, and the
	// mgmt connection urls if required to find backup containers
	private void addRestartContainersList(Properties seedingProps)
	{
		int restartSize = m_restartContainers.size();
		String restartString = "";
		if (m_restartDM || restartSize > 0)
		{
		    if (m_restartDM)
		    {
		        // remember the log directory of the DM. Other log directories for the other
		        // containers have already been saved in the values of m_restartContainers
		    	restartString = restartString + m_DMElement.getIdentity().getName() + ",";
		    	HashMap dmProps = getContainerProps((String)m_DMElement.getAttributes().getAttribute(IContainerConstants.CONTAINER_NAME_ATTR));
		    	String windowsService = (String)dmProps.get(WSERVICEPROP);
		    	if (windowsService != null)
                {
                    restartString = restartString + "SERVICE";
                }
                else
                {
                    restartString = restartString + "COMMANDLINE";
                }
		    }
		    if (restartSize > 0)
		    {
		    	for (String id : m_restartContainers.keySet())
		    	{
		    	    if (restartString.length() > 0)
                    {
		    	        // there's a previous container there,
		    	        // add a separator character
		    	        restartString = restartString + ";";
                    }
		    		String method = (String)m_restartContainers.get(id);
		    		if (method.startsWith("ERROR"))
                    {
                        continue;
                    }
		    		restartString = restartString + id + "," + method;
		    	}
		    }
		    seedingProps.put(RESTARTCONTAINERSPROP, restartString);
		    // if the DM needs to be restarted, remember the mgmt connection URLs
		    // so the backup can be stopped before the primary is restarted
		    String mgmtUrls = null;
		    if (m_restartDM && ((mgmtUrls = m_migrateProps.getProperty(MGMTCONNURLSPROP)) != null))
		    {	    	
		    	// if there's a backup DS you expect a value for connectionURLs
			    IAttributeSet connectionProps;
			    connectionProps = (IAttributeSet)m_DMElement.getAttributes().getAttribute(IContainerConstants.CONNECTION_ATTR);
			    String username = null; String password = null; String node = null;
			    String domain = null; String conProps = null;
			    if (connectionProps != null)
			    {
			    	username = (String)connectionProps.getAttribute(IContainerConstants.DEFAULTUSER_ATTR);
			    	if (username == null || username.length() == 0)
			    	{
			    		username = DEFAULTUSER;
			    	}
			    	password = (String)connectionProps.getAttribute(IContainerConstants.DEFAULTPASSWORD_ATTR);
			    	node = (String)connectionProps.getAttribute(IContainerConstants.MANAGEMENT_NODE_ATTR);
			    	domain = m_migrateProps.getProperty(DOMAINPROP);
			    	conProps = m_migrateProps.getProperty(CONNPROPSPROP);
			    }
			   
		    	seedingProps.put(MGMTCONNURLSPROP, mgmtUrls); 
		    	if (username != null)
		    	{
		    		seedingProps.put(USERNAMEPROP, username);
		    	}
		    	if (password != null)
                {
                    seedingProps.put(PASSWORDPROP, password);
                }
		    	if (node != null)
                {
                    seedingProps.put(NODEPROP, node);
                }
		    	seedingProps.put(DOMAINPROP, domain); 	
		    	if (conProps != null)
                {
                    seedingProps.put(CONNPROPSPROP, conProps);
                } 
		    	String backupDSContainer = (String)m_migrateProps.getProperty(BACKUPDSCONTAINERPROP);
		    	if (backupDSContainer != null)
                {
                    seedingProps.put(BACKUPDSCONTAINERPROP, backupDSContainer);
                }
		    }
		}
	}
	 
	private void checkInstallProperties() throws MigrationException, FileNotFoundException, IOException
	{
		File sonicHome = new File(m_sonicHome);
    	File sonicswPropertiesFile = new File(sonicHome, SONICSW_PROPERTIES);
    	if (!sonicswPropertiesFile.exists())
        {
            throw new MigrationException("Could not find the sonicsw.properties files required to create shortcuts. Something went wrong during the installation.");
        }
    	FileInputStream propsStream = new FileInputStream(sonicswPropertiesFile);
	    SONICSW_PROPS.load(propsStream);
	    propsStream.close();
		if (m_wbLocation != null)
		{
			// then this should be a workbench install and eclipse should have
			// been installed
			String wbInstalled = SONICSW_PROPS.getProperty(WB_INSTALLED_PROP);
			if (wbInstalled == null || !wbInstalled.equals("true"))
            {
                throw new MigrationException("The workbench location argument cannot be used if a workbench has not been installed. Remove the argument and try again.");
            }
			String eclipseInstalled = SONICSW_PROPS.getProperty(ECLIPSE_CHOICE_PROP);
			if (eclipseInstalled == null || !eclipseInstalled.equals("NEW"))
            {
                throw new MigrationException("The workbench location argument cannot be used if eclipse is not installed with the workbench. Remove the argument and try again.");
            }			
		}
	}
	
	// Make sure the directory specified by newStorageDir can be created; ie.
	// none of the intermediate paths are files, we have permission to write, etc.
	// if it's a relative path find it in the new "Containers" working area.
	// Return the new DS storage directory if specified by the user,
	// otherwise null.
	private File checkNewDSHostProp(HashMap containerProps, String containerName) throws MigrationException, IOException
	{
		String domainName = m_migrateProps.getProperty(DOMAINPROP);
		File containers = ContainerSetup.findSonicContainersDir(new File(m_sonicHome));
		File containerDir = ContainerSetup.findContainerDir(containers, domainName, containerName);
		File destinationDir = null;
		// containerDir is the parent of the parent of the domain name directory
		String enteredDSHost = (String)containerProps.get(HOSTDIRPROP);
		File enteredDir = null;
		if (enteredDSHost != null)
		{
			enteredDir = new File(enteredDSHost);
			if (enteredDir.isAbsolute())
            {
                destinationDir = enteredDir;
            }
            else
            {
                destinationDir = new File(containerDir, enteredDSHost);
            }
			checkDirectory(destinationDir, containerName + "." + DSHOSTPROP);
			debug("checkNewDSHostProp setting DSHOSTPROP to " + destinationDir.getCanonicalPath());
			m_migrateProps.put(DSHOSTPROP, destinationDir.getCanonicalPath());	
			return destinationDir;
		}
        else
        {
            return null;
        }
	}
	
	// takes in a property name for error reporting, since this could be used generically to 
	// test directory specifications.
	private void checkDirectory(File destinationDir, String propertyName) throws MigrationException, IOException
	{
		if (destinationDir == null)
        {
            return;
        }
		if (destinationDir.exists())
		{
			if (destinationDir.isFile())
            {
                throw new MigrationException(destinationDir.getCanonicalPath() + " is an existing file and cannot be used as a directory for " + propertyName);
            }			
			// otherwise check that we can write
			if (!destinationDir.canWrite())
            {
                throw new MigrationException("The upgrade is not able to write to the directory " + destinationDir.getCanonicalPath()+ " specified in "+ propertyName + ".");
            }
		}
		else
		{
			if (destinationDir.getParentFile() == null)
            {
                // we're at the top and the directory does not exist
				throw new MigrationException("Drive accessed by " + destinationDir + " does not exist");
            }
			checkDirectory(destinationDir.getParentFile(), propertyName);
		}
	}
	
	private boolean willMigrate(String releaseVersion)
	{
		return releaseVersion.compareTo(FIRST_CIPGRADE_RELEASE_VERSION) < 0;
	}
	
	private void recycleContainer(String containerID) throws Exception
	{
	    String cName = getContainerName(containerID);
        String cObjectName = getAgentObjectName(cName);
        // pingContainer will return false if what we have is a direct connection to the DS
        // or if the DS is not running, which is ok, because we cannot communicate with the 
        // container to restart it in those cases.
	    if (pingContainer(cObjectName))
	    {
	        System.out.println("upgrade restarting " + cObjectName + "...");
	        m_ds.restartContainer(cObjectName, getAMObjectName(cName));
	        System.out.println("upgrade restarting " + cObjectName + "...done");
	    }
	}

}
