package com.sonicsw.mf.common.config.upgrade;

import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Vector;

import com.sonicsw.mf.common.IDSTransaction;
import com.sonicsw.mf.common.config.ConfigException;
import com.sonicsw.mf.common.config.IAttributeSet;
import com.sonicsw.mf.common.config.IElementIdentity;
import com.sonicsw.mf.common.config.IMFDirectories;
import com.sonicsw.mf.common.config.ReadOnlyException;
import com.sonicsw.mf.common.config.Reference;
import com.sonicsw.mf.common.config.query.AttributeName;
import com.sonicsw.mf.common.dirconfig.DirectoryServiceException;
import com.sonicsw.mf.common.dirconfig.IDeltaDirElement;
import com.sonicsw.mf.common.dirconfig.IDirElement;
import com.sonicsw.mf.mgmtapi.config.constants.IActivationDaemonConstants;
import com.sonicsw.mf.mgmtapi.config.constants.IAgentManagerConstants;
import com.sonicsw.mf.mgmtapi.config.constants.ICollectionsMonitorConstants;
import com.sonicsw.mf.mgmtapi.config.constants.IContainerConstants;
import com.sonicsw.mf.mgmtapi.config.constants.IDirectoryServiceConstants;

public class TypedConfigUpgrade implements IMigrationProcess
{
	UpgradeEnv m_upgradeEnv = null;
	Utils m_utils = null;

	private static String CONFIG_TYPE_DIRECTORY = "/mx/configTypes/";
	private static String INITIAL_VALUES_ATTR = "initialValues";
	private Class[] UPGRADE_METHOD_SIGNATURE = new Class[] {IDirElement.class, UpgradeEnv.class};
	
	public TypedConfigUpgrade(UpgradeEnv env)
	{
		m_upgradeEnv = env;
		m_utils = m_upgradeEnv.getUtils();
	}
	
	private String doVersionedUpgrade(IDirElement el, String releaseVersion, String type)
	throws Exception 
    {
		IVersionedConfigUpgrade upgrade = UpgradeRegistry.getConfigUpgrade(releaseVersion);
		if (upgrade == null)
        {
            throw new Exception("Unable to find a config upgrade method for type" + type + " for version " + releaseVersion);
        }
		java.lang.reflect.Method upgradeMethod = upgrade.getClass().getMethod(
				"upgrade" + type, UPGRADE_METHOD_SIGNATURE);
		String message = (String) upgradeMethod.invoke(upgrade, new Object[] {
				el, m_upgradeEnv });
		if (!upgrade.getUpgradedReleaseVersion().equals(CURRENT_RELEASE_VERSION))
        {
            message = m_utils.addToUpgradeMessage(message, doVersionedUpgrade(
					el, upgrade.getUpgradedReleaseVersion(), type));
        }
		return message;
	}
	
	
	public String upgradeMF_AGENT_MANAGER(IDirElement AM) throws Exception
    {
		// There are no templates of MF_AGENT_MANAGER allowed in 6.X, so there's
		// no template code here.
		// do versioned upgrades first
		String relVersion = AM.getIdentity().getReleaseVersion();
		String returnVal = doVersionedUpgrade(AM, relVersion, "MF_AGENT_MANAGER");
		setArchiveName(AM, IAgentManagerConstants.ARCHIVE_NAME_ATTR, "MF/" + CURRENT_PRODUCT_VERSION + "/MFmanager.car");
		m_upgradeEnv.dsFileSystem.updateFSElement((IDeltaDirElement)AM.doneUpdate());
		m_utils.fixProductAndReleaseVersion(AM.getIdentity().getName());
		return returnVal;
	}

	public String upgradeMF_BACKUP_DIRECTORY_SERVICE(IDirElement backup) throws Exception
	{
		//do versioned upgrades first
		String relVersion = backup.getIdentity().getReleaseVersion();
		
		String returnVal = null;
		if (!relVersion.equals(CURRENT_RELEASE_VERSION))
        {
            returnVal = doVersionedUpgrade(backup, relVersion, "MF_BACKUP_DIRECTORY_SERVICE");
        }  
		String containerName = (String)m_upgradeEnv.getProp(CONTAINERNAMEPROP);
		String enteredHostDir = null;
		if (containerName != null)
        {
            enteredHostDir = (String)m_upgradeEnv.getContainerProp(containerName, IMigrationProcess.HOSTDIRPROP);
        }
		if (enteredHostDir != null)
        {
            m_utils.fixHOST_DIRECTORY(backup, enteredHostDir);
        }
		m_upgradeEnv.dsFileSystem.updateFSElement((IDeltaDirElement)backup.doneUpdate());
		if (!relVersion.equals(CURRENT_RELEASE_VERSION))
        {
            m_utils.fixProductAndReleaseVersion(backup.getIdentity().getName());
        }
		return returnVal;
	}

    // currently MF_BACKUP_DIRECTORY_SERVICE does not go through this logic.
    public String upgradeMF_DIRECTORY_SERVICE(IDirElement DS) throws Exception
    { 
        //do versioned upgrades first
		String relVersion = DS.getIdentity().getReleaseVersion();
		String returnVal = doVersionedUpgrade(DS, relVersion, "MF_DIRECTORY_SERVICE");        	 
        String dsHostProp = (String)m_upgradeEnv.getProp(IMigrationProcess.DSHOSTPROP);
        if (dsHostProp != null)
        {
            m_utils.fixHOST_DIRECTORY(DS, dsHostProp);
        }
        
        m_upgradeEnv.dsFileSystem.updateFSElement((IDeltaDirElement)DS.doneUpdate());
        m_utils.fixProductAndReleaseVersion(DS.getIdentity().getName());
        // if there is a REPLICATION_CONNECTION element, update its version
        IAttributeSet configRefs = (IAttributeSet) DS.getAttributes().getAttribute(IDirectoryServiceConstants.CONFIG_ELEMENT_REFERENCES_ATTR);
        if (configRefs != null)
        {
        	Reference replRef = (Reference) configRefs.getAttribute(IDirectoryServiceConstants.REPLICATION_CONNECTIONS_ELEMENT_REF_ATTR);
        	if (replRef != null)
            {
                m_utils.fixProductAndReleaseVersion(replRef.getElementName());
            }
        }
        
        if (m_upgradeEnv.isDEBUG())
        {
            System.out.println("Done upgrading MF_DIRECTORY_SERVICE " + DS.getIdentity().getName() );
        }
        return returnVal;
    }
    
    
    public String upgradeMF_CONTAINER(IDirElement container)
        throws Exception
    {
    	IElementIdentity id = container.getIdentity();
    	 if (m_upgradeEnv.isDEBUG())
        {
            System.out.println("TypedConfigUpgrade.upgradeMF_CONTAINER upgrading " + id.getName());
        }
    	if (id.getReleaseVersion().equals(CURRENT_RELEASE_VERSION))
    	{
    		// we're here only to upgrade the paths
    		upgradeContainerPaths(container);
    		m_upgradeEnv.dsFileSystem.updateFSElement((IDeltaDirElement)container.doneUpdate());
    		return null;
    	}
		if (container.isTemplate()) 
		{
			String[] instanceNames = container.getSubclassedList();
			for (int i = 0; i < instanceNames.length; i++)
            {
                upgradeMF_CONTAINER(m_upgradeEnv.dsFileSystem.getFSElement(
						instanceNames[i], true, true));
            }
		}

        //do versioned upgrades first
		String relVersion = container.getIdentity().getReleaseVersion();
		String message = doVersionedUpgrade(container, relVersion, "MF_CONTAINER"); 
		if (m_upgradeEnv.isDEBUG())
        {
            System.out.println("TypedConfigUpgrade.upgradeMF_CONTAINER ID after doVersionedUpgrade " + container.getIdentity().getName());
        }
		upgradeContainerPaths(container);
		if (m_upgradeEnv.isDEBUG())
        {
            System.out.println("TypedConfigUpgrade.upgradeMF_CONTAINER ID after upgradeContainerPaths " + container.getIdentity().getName());
        }
		setArchiveName(container, IContainerConstants.ARCHIVE_NAME_ATTR, "MF/" + CURRENT_PRODUCT_VERSION + "/MFcontainer.car");
		setArchiveName(container, IContainerConstants.ACTIONAL_SDK_ARCHIVE_NAME_ATTR, "MF/" + CURRENT_PRODUCT_VERSION + "/actional-sdk.jar");
		setArchiveName(container, IContainerConstants.ACTIONAL_PLUGMAKER_ARCHIVE_NAME_ATTR, "MF/" + CURRENT_PRODUCT_VERSION + "/actional-plugmaker.jar");
		message = Utils.addToUpgradeMessage(message, setArchiveSearchPath(container, "sonicfs:///Archives"));
		checkCLI(container);
		m_upgradeEnv.dsFileSystem.updateFSElement((IDeltaDirElement)container.doneUpdate());
		if (m_upgradeEnv.isDEBUG())
        {
            System.out.println("TypedConfigUpgrade.upgradeMF_CONTAINER ID after updateFSElement " + container.getIdentity().getName());
        }
		m_utils.fixProductAndReleaseVersion(container.getIdentity().getName());
		return message;
	}
    
    public String upgradeMF_ACTIVATION_DAEMON(IDirElement daemon)
        throws Exception
    {
        if (daemon.isTemplate())
        {
            String[] instanceNames = daemon.getSubclassedList();
            for (int i=0; i<instanceNames.length; i++)
            {
                upgradeMF_ACTIVATION_DAEMON(m_upgradeEnv.dsFileSystem.getFSElement(instanceNames[i], true, true));
            }
        }
        //do versioned upgrades first
		String relVersion = daemon.getIdentity().getReleaseVersion();
		String message = doVersionedUpgrade(daemon, relVersion, "MF_ACTIVATION_DAEMON");      
        if (daemon.getSuperElementName() == null)
        {
            setArchiveName(daemon, IActivationDaemonConstants.ARCHIVE_NAME_ATTR, "MF/" + CURRENT_PRODUCT_VERSION + "/MFdaemon.car");
        }       
        m_upgradeEnv.dsFileSystem.updateFSElement((IDeltaDirElement)daemon.doneUpdate());
        m_utils.fixProductAndReleaseVersion(daemon.getIdentity().getName());
        return message;
    }

    public String upgradeMF_COLLECTION_MONITOR(IDirElement monitor)
        throws Exception
    {
        // no template code for monitor
        //do versioned upgrades first
		String relVersion = monitor.getIdentity().getReleaseVersion();
		String message = doVersionedUpgrade(monitor, relVersion, "MF_COLLECTION_MONITOR");
    	setArchiveName(monitor, ICollectionsMonitorConstants.ARCHIVE_NAME_ATTR, "MF/" + CURRENT_PRODUCT_VERSION + "/MFmonitor.car");       
        m_upgradeEnv.dsFileSystem.updateFSElement((IDeltaDirElement)monitor.doneUpdate());
        m_utils.fixProductAndReleaseVersion(monitor.getIdentity().getName());
        return message;
        // TEST
        //IDirElement test = dsFileSystem.getFSElement(monitor.getIdentity().getName(), false);
        //String archiveName = (String)test.getAttributes().getAttribute("ARCHIVE_NAME");
    }
    
    public String upgradeMQ_CLUSTER(IDirElement cluster) throws Exception
    {
    	// nothing to be done if cluster has already been upgraded
    	if (cluster.getIdentity().getReleaseVersion().equals(CURRENT_RELEASE_VERSION))
        {
            return null;
        }
        if (cluster.isTemplate())
        {
              String[] instanceNames = cluster.getSubclassedList();
              for (int i=0; i<instanceNames.length; i++)
            {
                upgradeMQ_CLUSTER(m_upgradeEnv.dsFileSystem.getFSElement(instanceNames[i], true, true));
            }
        }
        //do versioned upgrades first
		String relVersion = cluster.getIdentity().getReleaseVersion();
		String message = doVersionedUpgrade(cluster, relVersion, "MQ_CLUSTER");
        m_upgradeEnv.dsFileSystem.updateFSElement((IDeltaDirElement)cluster.doneUpdate());
        upgradeClusterVersions(cluster);
        return message;
    }
    
    public String upgradeMQ_BROKER(IDirElement broker)
        throws Exception
    {
        // This method handles both MQ_BROKER and MQ_BACKUPBROKER
         IElementIdentity id = broker.getIdentity();
         String relVersion = id.getReleaseVersion();
         if (relVersion.equals(CURRENT_RELEASE_VERSION))
         {
        	 // then we're here only to change paths that are host related
        	 IDSTransaction transaction = m_upgradeEnv.dsFileSystem.createTransaction();
        	 m_utils.upgradeBrokerAndDBPaths(broker);
        	 transaction.addUpdateElement((IDeltaDirElement)broker.doneUpdate());
             m_upgradeEnv.dsFileSystem.executeTransaction(transaction);
             return null;
         }
        // Mari 2/25/2004  Upgrade the instances before the templates. Modified attributes
        // in the subclasses are DELETED when the modified attribute is deleted from the template, so edits
        // in the subclass DISAPPEAR! if we upgrade the template first

        if (m_upgradeEnv.isDEBUG())
        {
            System.out.println("ConfigUpgrade.upgradeBrokerInstance, broker == " + broker.getIdentity().getName());
        }
        if (broker.isTemplate())
        {
              String[] instanceNames = broker.getSubclassedList();
              for (int i=0; i<instanceNames.length; i++)
            {
                upgradeMQ_BROKER(m_upgradeEnv.dsFileSystem.getFSElement(instanceNames[i], true, true));
            }
        }
        IDSTransaction transaction = m_upgradeEnv.dsFileSystem.createTransaction();
        
        //do versioned upgrades first
		
		String message = doVersionedUpgrade(broker, relVersion, "MQ_BROKER");
        upgradeBrokerPaths(broker);
        // set the control number
        IAttributeSet topSet = broker.getAttributes();
        IAttributeSet prodInfo = (IAttributeSet) topSet.getAttribute("PRODUCT_INFORMATION");
        // controlCode will be null if we're upgrading from the SMC
        if (prodInfo != null && !prodInfo.getAttributeMetaData("CONTROL_NUMBER").isFromTemplate())
        {
        	prodInfo.setStringAttribute("CONTROL_NUMBER", (String)m_upgradeEnv.getProp(IMigrationProcess.BROKERCNPROP));
        	if (m_upgradeEnv.isDEBUG())
            {
                System.out.println("Changed the broker control number to " + m_upgradeEnv.getProp(IMigrationProcess.BROKERCNPROP));
            }
        }
        // set the ARCHIVE_NAME
        if (broker.getSuperElementName() == null)
        {
            if (m_upgradeEnv.isDEBUG())
            {
                System.out.println("Super element name for " + broker.getIdentity().getName() + " was null");
            }
            setArchiveName(broker, IContainerConstants.ARCHIVE_NAME_ATTR, "MQ/" + CURRENT_PRODUCT_VERSION + "/MQbroker.car");
        }
        // save the broker config
        transaction.addUpdateElement((IDeltaDirElement)broker.doneUpdate());
        // save all the changes before upgrading the configuration version
        m_upgradeEnv.dsFileSystem.executeTransaction(transaction);
        // if this broker is in a cluster being upgraded, upgrade the cluster configuration
        String clusterID = findCluster(broker);
        if (clusterID != null)
        {
        	// upgradeMQ_CLUSTER will check whether the cluster has been upgraded already or not.        	
        	IDirElement clusterEl = m_upgradeEnv.dsFileSystem.getFSElement(clusterID, true);
        	upgradeMQ_CLUSTER(clusterEl);
        }
        upgradeBrokerVersions(broker);
        return message;
    }
    
    public String upgradeMF_LOGGER(IDirElement logger)
        throws Exception
    {

        if (m_upgradeEnv.isDEBUG())
        {
            System.out.println("ConfigUpgrade.upgradeLoggerConfig, logger == " + logger.getIdentity().getName());
        }
        if (logger.isTemplate())
        {
            String[] instanceNames = logger.getSubclassedList();
            for (int i = 0; i < instanceNames.length; i++)
            {
                upgradeMF_LOGGER(m_upgradeEnv.dsFileSystem.getFSElement(instanceNames[i], true, true));
            }
        }
       
       //do versioned upgrades first
		String relVersion = logger.getIdentity().getReleaseVersion();
		String message = doVersionedUpgrade(logger, relVersion, "MF_LOGGER");
        // set the ARCHIVE_NAME
        if (logger.getSuperElementName() == null)
        {
            if (m_upgradeEnv.isDEBUG())
            {
                System.out.println("Super element name for " + logger.getIdentity().getName() + " was null");
            }
            setArchiveName(logger, IContainerConstants.ARCHIVE_NAME_ATTR, "MF/" + CURRENT_PRODUCT_VERSION + "/MFlogger.car");
        }
        m_upgradeEnv.dsFileSystem.updateFSElement((IDeltaDirElement)logger.doneUpdate());
        m_utils.fixProductAndReleaseVersion(logger.getIdentity().getName());
        return message;
    }
    
    public String upgradeXQ_CONTAINER(IDirElement XQContainer) throws Exception
    {
        // no template code necessary for XQ_CONTAINERs
        fixProductVersion(XQContainer.getIdentity().getName());
        IAttributeSet topSet = XQContainer.getAttributes();
        topSet.setStringAttribute("ARCHIVE_NAME", "ESB/" + CURRENT_PRODUCT_VERSION + "/ESBcontainer.car");
        m_upgradeEnv.dsFileSystem.updateFSElement((IDeltaDirElement)XQContainer.doneUpdate());
        
     // remove the control code of DBService instances
        String storageXQName = m_upgradeEnv.dsFileSystem.logicalToStorage(XQContainer.getIdentity().getName());         
        IDirElement xqContainer = m_upgradeEnv.dsAdmin.getElement(storageXQName, false);
        IAttributeSet servicesSet = (IAttributeSet)xqContainer.getAttributes().getAttribute("services");
        if (servicesSet != null)
        {
            HashMap servicesMap = servicesSet.getAttributes();
            Collection servicesMaps = servicesMap.values();
            Iterator servicesMapsIt = servicesMaps.iterator();
            while (servicesMapsIt.hasNext())
            {
                IAttributeSet serviceSet = (IAttributeSet)servicesMapsIt.next();
                String serviceName = (String)serviceSet.getAttribute("service_ref");
                IDirElement serviceEl = m_upgradeEnv.dsAdmin.getElement("/xqServices/" + serviceName, true);
                // cover the case where there is a service reference to a service instance that has been removed
                if (serviceEl != null)
                {
                    String serviceType = (String)serviceEl.getAttributes().getAttribute("type_ref");
                    if (serviceType.equals("DatabaseServiceType"))
                    {
                        IAttributeSet params = (IAttributeSet)serviceEl.getAttributes().getAttribute("params");
                        params.deleteAttribute("controlNumber");
                        m_upgradeEnv.dsAdmin.setElement(serviceEl.doneUpdate(), null);
                    }
                }
            }
        }

        return null;
    }

    public String upgradeMF_HOST_MANAGER(IDirElement hostManager) throws Exception
    {
        // nothing to be done if hostmanager has already been upgraded
        if (hostManager.getIdentity().getReleaseVersion().equals(CURRENT_RELEASE_VERSION))
        {
            return null;
        }
        // do versioned upgrades first
        String relVersion = hostManager.getIdentity().getReleaseVersion();
        String message = doVersionedUpgrade(hostManager, relVersion, "MF_HOST_MANAGER");
        setArchiveName(hostManager, IAgentManagerConstants.ARCHIVE_NAME_ATTR, "MF/" + CURRENT_PRODUCT_VERSION + "/MFhostmanager.car");
        m_upgradeEnv.dsFileSystem.updateFSElement((IDeltaDirElement)hostManager.doneUpdate());
        m_utils.fixProductAndReleaseVersion(hostManager.getIdentity().getName());
        return message;
    }

    void upgradeBrokerVersions(IDirElement broker)
    throws Exception
    {
    	m_upgradeEnv.m_curTransaction = m_upgradeEnv.dsFileSystem.createTransaction();
        try
        {
            String brokerLogicalName = broker.getIdentity().getName();
            // strip off the _Default
            brokerLogicalName = brokerLogicalName.substring(0, brokerLogicalName.lastIndexOf("/"));
            IAttributeSet topSet = broker.getAttributes();
            m_utils.fixProductAndReleaseVersion(broker.getIdentity().getName());
            IAttributeSet configRefs = (IAttributeSet)topSet.getAttribute("CONFIG_ELEMENT_REFERENCES");
            Reference routesRef = (Reference)configRefs.getAttribute("ROUTING_CONFIG_ELEMENT_REF");
            if ((routesRef != null) && (!configRefs.getAttributeMetaData("ROUTING_CONFIG_ELEMENT_REF").isFromTemplate()))
            {
                IDirElement routesEl = m_upgradeEnv.dsFileSystem.getFSElement(routesRef.getElementName(), true, true);
                IAttributeSet routeList = (IAttributeSet)routesEl.getAttributes().getAttribute("ROUTES");
                if (routeList == null)
                {
                    routeList = routesEl.getAttributes().createAttributeSet("ROUTES");
                }
                if (!routesEl.getAttributes().getAttributeMetaData("ROUTES").isFromTemplate())
                {
                    HashMap allRoutes = routeList.getAttributes();
                    Collection routeElements = allRoutes.values();
                    Iterator routeElementsIt = routeElements.iterator();
                    while (routeElementsIt.hasNext())
                    {
                        Reference route = (Reference)routeElementsIt.next();
                        m_utils.fixProductAndReleaseVersion(route.getElementName());
                    }
                }
                m_utils.fixProductAndReleaseVersion(routesEl.getIdentity().getName());
            }
            // upgrade the product version of the queues element in the view
            Reference queueReference = (Reference)configRefs.getAttribute("QUEUE_CONFIG_ELEMENT_REF");
            if ((queueReference != null) && (!configRefs.getAttributeMetaData("QUEUE_CONFIG_ELEMENT_REF").isFromTemplate()))
            {
                m_utils.fixProductAndReleaseVersion(queueReference.getElementName());
            }

            // upgrade the replication connections

            Reference replicationsReference = (Reference)configRefs.getAttribute("REPLICATION_CONNECTIONS_ELEMENT_REF");
            if ((replicationsReference != null) && (!configRefs.getAttributeMetaData("REPLICATION_CONNECTIONS_ELEMENT_REF").isFromTemplate()))
            {
                m_utils.fixProductAndReleaseVersion(replicationsReference.getElementName());
            }

            // upgrade the product versions of the acceptors in the view

            Reference acceptorConfigRef = (Reference)configRefs.getAttribute("ACCEPTOR_CONFIG_ELEMENT_REF");
            if ((acceptorConfigRef != null) && (!configRefs.getAttributeMetaData("ACCEPTOR_CONFIG_ELEMENT_REF").isFromTemplate()))
            {
                IDirElement acceptorConfig = m_upgradeEnv.dsFileSystem.getFSElement(acceptorConfigRef.getElementName(), true, true);
                Vector acceptorVersionUpgrade = upgradeAcceptorConfig(acceptorConfig);
                for (int s = 0; s < acceptorVersionUpgrade.size(); s++)
                {
                    IDirElement item = (IDirElement)acceptorVersionUpgrade.get(s);
                    m_utils.fixProductAndReleaseVersion(item.getIdentity().getName());
                }
            }

            // upgrade the product version of global routing element in the view

            Reference globalSubsRef = (Reference)configRefs.getAttribute("GLOBAL_SUBSCRIPTIONS_CONFIG_ELEMENT_REF");
            if ((globalSubsRef != null) && (!configRefs.getAttributeMetaData("GLOBAL_SUBSCRIPTIONS_CONFIG_ELEMENT_REF").isFromTemplate()))
            {
                m_utils.fixProductAndReleaseVersion(globalSubsRef.getElementName());
            }
            m_upgradeEnv.dsFileSystem.executeTransaction(m_upgradeEnv.m_curTransaction);
        }
        finally
        {
        	m_upgradeEnv.m_curTransaction = null;
        }
    }

    // mostly go down the acceptor configuration and get a list of the acceptors, protocls, urls, etc. to change their versions
    // fix 6.1 upgrade bug by adding the AUTHENTICATION attribute map to the HTTP protocol urls.
    private Vector upgradeAcceptorConfig(IDirElement acceptorConfig) throws Exception
    {
        Vector versionChange = new Vector();
        versionChange.add(acceptorConfig);
        IAttributeSet acceptorsMap = (IAttributeSet)acceptorConfig.getAttributes().getAttribute("ACCEPTORS");
        HashMap acceptorsHMap = acceptorsMap.getAttributes();
        Collection acceptorsRefs = acceptorsHMap.values();
        Iterator acceptorsRefsIt = acceptorsRefs.iterator();
        while (acceptorsRefsIt.hasNext())
        {
            Reference acceptorRef = (Reference) acceptorsRefsIt.next();
            String acceptorName = acceptorRef.getElementName();
            IDirElement acceptorEl = m_upgradeEnv.dsFileSystem.getFSElement(acceptorName, true, true);
            if (m_upgradeEnv.isDEBUG())
            {
                System.out.println("ConfigUpgrade.upgradeDirectURLS upgrading release version of " + acceptorName);
            }
            versionChange.add(acceptorEl);
            if (acceptorEl.getIdentity().getType().equals("MQ_ACCEPTOR_DIRECT"))
            {
                IAttributeSet protocols = (IAttributeSet)acceptorEl.getAttributes().getAttribute("PROTOCOLS");
                if (protocols != null)
                {
                    HashMap protocolsHMap = protocols.getAttributes();
                    Collection protocolsRefs = protocolsHMap.values();
                    Iterator protocolsRefsIt = protocolsRefs.iterator();
                    while (protocolsRefsIt.hasNext())
                    {
                        Reference protocolRef = (Reference) protocolsRefsIt.next();
                        IDirElement protocolEl = m_upgradeEnv.dsFileSystem.getFSElement(protocolRef.getElementName(), true, true);
                        if (m_upgradeEnv.isDEBUG())
                        {
                            System.out.println("ConfigUpgrade.upgradeDirectURLS upgrading release version of " + protocolRef.getElementName());
                        }
                        versionChange.add(protocolEl);
                        IAttributeSet urlMap = (IAttributeSet)protocolEl.getAttributes().getAttribute("DIRECT_URL_MAP");
                        if (urlMap != null)
                        {
                            HashMap urlsHMap = urlMap.getAttributes();
                            Collection urlsRef = urlsHMap.values();
                            Iterator urlsRefIt = urlsRef.iterator();
                            while (urlsRefIt.hasNext())
                            {
                                Reference urlRef = (Reference)urlsRefIt.next();
                                IDirElement urlEl = m_upgradeEnv.dsFileSystem.getFSElement(urlRef.getElementName(), true);
                                if (m_upgradeEnv.isDEBUG())
                                {
                                    System.out.println("ConfigUpgrade.upgradeDirectURLS upgrading release version of " + urlRef.getElementName());
                                }
                                versionChange.add(urlEl);
                                if (urlEl.getAttributes().getAttribute("AUTHENTICATION") == null)
                                {
                                    if (m_upgradeEnv.isDEBUG())
                                    {
                                        System.out.println("ConfigUpgrade.upgradeAcceptorsElement adding AUTHENTICATION attribute set");
                                    }
                                    urlEl.getAttributes().createAttributeSet("AUTHENTICATION");
                                    m_upgradeEnv. dsFileSystem.updateFSElement((IDeltaDirElement)urlEl.doneUpdate());
                                }
                             }
                          }
                      }
                  }
              }
          }
          return versionChange;
      }
    
    private void setArchiveName(IDirElement el, String attributeName, String archiveName) throws Exception
    {
        IAttributeSet topSet = el.getAttributes();
        // meta data will be null if the attribute has never been added before, like the 
        // actional libraries in 8.5
        if (topSet.getAttributeMetaData(attributeName) == null || !topSet.getAttributeMetaData(attributeName).isFromTemplate())
        {
            topSet.setStringAttribute(attributeName, archiveName);
        }
    }


    // will return a message if the ARCHIVE_SEARCH_PATH is not reset because the value found
    // was not the initial value
    private String setArchiveSearchPath(IDirElement el, String archiveSearchPath) throws Exception
    {
        IElementIdentity elID = el.getIdentity();
        String initialValue = getInitialValue(elID.getType(), elID.getReleaseVersion(), IContainerConstants.ARCHIVE_SEARCH_PATH_ATTR);
    	IAttributeSet topSet = el.getAttributes();
    	String searchPathValue = (String)topSet.getAttribute(IContainerConstants.ARCHIVE_SEARCH_PATH_ATTR);
    	// set ARCHIVE_SEARCH_PATH only if the user has not changed the value
    	if (initialValue.equals(searchPathValue) && 
    			!topSet.getAttributeMetaData(IContainerConstants.ARCHIVE_SEARCH_PATH_ATTR).isFromTemplate())
    	{
            topSet.setStringAttribute(IContainerConstants.ARCHIVE_SEARCH_PATH_ATTR, archiveSearchPath);
            return null;
    	}
        else
        {
            return "The ARCHIVE_SEARCH_PATH of " + elID.getName() + " was set to the non default value of " + searchPathValue +
    		    " so it was not reset to the new default value of " + archiveSearchPath;
        }
    }
    
    private void fixProductVersion(String viewName) throws Exception
    {
      m_utils.fixViewVersionsHelper(viewName, false);
      if (viewName.indexOf("_Default") > -1)
      {
          String parentFolder = viewName.substring(0, viewName.lastIndexOf("/"));
          // Change the version of the parent directory as well
          m_utils.fixViewVersionsHelper(parentFolder, false);
      }

    }
    
    private void upgradeClusterVersions(IDirElement cluster)
    throws Exception
    {
    	m_upgradeEnv.m_curTransaction = m_upgradeEnv.dsFileSystem.createTransaction();
        try
        {
            IAttributeSet topSet = cluster.getAttributes();
            m_utils.fixProductAndReleaseVersion(cluster.getIdentity().getName());
            IAttributeSet configRefs = (IAttributeSet)topSet.getAttribute("CONFIG_ELEMENT_REFERENCES");
            Reference routesRef = (Reference)configRefs.getAttribute("ROUTING_CONFIG_ELEMENT_REF");
            if ((routesRef != null) && (!configRefs.getAttributeMetaData("ROUTING_CONFIG_ELEMENT_REF").isFromTemplate()))
            {
                IDirElement routesEl = m_upgradeEnv.dsFileSystem.getFSElement(routesRef.getElementName(), true, true);
                IAttributeSet routeList = (IAttributeSet)routesEl.getAttributes().getAttribute("ROUTES");
                if (routeList == null)
                {
                    routeList = routesEl.getAttributes().createAttributeSet("ROUTES");
                }
                if (!routesEl.getAttributes().getAttributeMetaData("ROUTES").isFromTemplate())
                {
                    HashMap allRoutes = routeList.getAttributes();
                    Collection routeElements = allRoutes.values();
                    Iterator routeElementsIt = routeElements.iterator();
                    while (routeElementsIt.hasNext())
                    {
                        Reference route = (Reference)routeElementsIt.next();
                        m_utils.fixProductAndReleaseVersion(route.getElementName());
                    }
                }
                m_utils.fixProductAndReleaseVersion(routesEl.getIdentity().getName());
            }

            Reference globalSubsRef = (Reference)configRefs.getAttribute("GLOBAL_SUBSCRIPTIONS_CONFIG_ELEMENT_REF");
            if ((globalSubsRef != null) && (!configRefs.getAttributeMetaData("GLOBAL_SUBSCRIPTIONS_CONFIG_ELEMENT_REF").isFromTemplate()))
             {
                m_utils.fixProductAndReleaseVersion(globalSubsRef.getElementName());
            // Fix the routing folder
            // fixVersionsHelper(clusterLogicalName + "/Routing");
            // fixVersionsHelper(clusterLogicalName + "/Routing/Global Subscriptions");
            }

            // upgrade the product version of the queues element to 6.1 in the view
            Reference queueReference = (Reference)configRefs.getAttribute("QUEUE_CONFIG_ELEMENT_REF");
            if ((queueReference != null) && (!configRefs.getAttributeMetaData("QUEUE_CONFIG_ELEMENT_REF").isFromTemplate()))
            {
                m_utils.fixProductAndReleaseVersion(queueReference.getElementName());
            }
            m_upgradeEnv.dsFileSystem.executeTransaction(m_upgradeEnv.m_curTransaction);
        }
        finally
        {
        	m_upgradeEnv.m_curTransaction = null;
        }
    }

    public String basicConfigUpgrade(IDirElement config)
    throws Exception
    {
		if (config.isTemplate()) 
		{
			String[] instanceNames = config.getSubclassedList();
			for (int i = 0; i < instanceNames.length; i++)
            {
                basicConfigUpgrade(m_upgradeEnv.dsFileSystem.getFSElement(
						instanceNames[i], true, true));
            }
		}

		m_utils.fixProductAndReleaseVersion(config.getIdentity().getName());
		return null;
	}
    
    void upgradeContainerPaths(IDirElement container)
    throws Exception
    {
    	
    	String containerName = (String)container.getAttributes().getAttribute(IContainerConstants.CONTAINER_NAME_ATTR);
    	if (m_upgradeEnv.isDEBUG())
        {
            System.out.println("TypedConfigUpgrade.upgradeContainerPaths " + containerName);
        }
    	String newCache = (String)m_upgradeEnv.getContainerProp(containerName, IMigrationProcess.CACHEPROP);
        String newLog = (String)m_upgradeEnv.getContainerProp(containerName, IMigrationProcess.CONTLOGPROP);
		AttributeName cacheDirectory = new AttributeName("CACHE");
		cacheDirectory.setNextComponent("CACHE_DIRECTORY");
		if (newCache != null)
		{
			if (m_upgradeEnv.isDEBUG())
            {
                System.out.println("TypedConfigUpgrade.upgradeContainerPaths " + containerName + " setting cache dir to " + newCache);
            }
			m_utils.fixPath(container.getAttributes(), cacheDirectory, newCache);
		}
		if (newLog != null)
		{
			if (m_upgradeEnv.isDEBUG())
            {
                System.out.println("TypedConfigUpgrade.upgradeContainerPaths " + containerName + " setting log dir to " + newLog);
            }
			m_utils.fixPath(container.getAttributes(), new AttributeName("LOG_FILE"), newLog);
		}
	}
    

    void upgradeBrokerPaths(IDirElement broker) throws Exception
    {
    	String containerName = (String)m_upgradeEnv.getProp(CONTAINERNAMEPROP);
    	String sslCertChain = (String)m_upgradeEnv.getContainerProp(containerName, SSLCERTCHAINPROP);
        String certDir = (String)m_upgradeEnv.getContainerProp(containerName, SSLCERTDIRPROP);
        IAttributeSet topSet = broker.getAttributes();
        AttributeName fixPath = new AttributeName("BROKER_SSL_PARAMETERS");
        fixPath.setNextComponent("SSL_CERTIFICATE_CHAIN");
        if (sslCertChain != null)
        {
            m_utils.fixPath(topSet, fixPath, sslCertChain);
        }
        fixPath = new AttributeName("BROKER_SSL_PARAMETERS");
        fixPath.setNextComponent("SSL_CA_CERTIFICATES_DIR");
        if (certDir != null)
        {
            m_utils.fixPath(topSet, fixPath, certDir);
        }        
        m_utils.upgradeBrokerAndDBPaths(broker);
        
    }
    
    // If the container has COMMAND_LINE set to the default value of false,
    // remove the attribute. During Workbench initialization of the Domain
    // MAnager container, it looks to the Agent like we've made a modification
    // to this attribute (delete it) and then forced the CLI=true, which causes
    // a benign warning message to be printed in the container log once, the first
    // time the container is started. Do this for the main container being
    // upgraded, because we want a generated container boot file to match the
    // change in this attribute.
    private void checkCLI(IDirElement container) throws ReadOnlyException, ConfigException
    {
    	// This container contains the MF_DIRECTORY_SERVICE instance
    	String DMElementName = (String)m_upgradeEnv.getProp(DMELEMENTPROP);
    	if ( DMElementName != null && DMElementName.equals(container.getIdentity().getName()))
    	{
    	    IAttributeSet topSet = container.getAttributes();
    	    Boolean cl = (Boolean)topSet.getAttribute(IContainerConstants.COMMAND_LINE_ATTR);
    	    if (cl != null && !cl.booleanValue())
            {
                topSet.deleteAttribute(IContainerConstants.COMMAND_LINE_ATTR);
            }
    	}
    }
    
    private String getInitialValue(String type, String version, String attributeName) throws DirectoryServiceException
    {
    	IDirElement  typeEl = m_upgradeEnv.dsAdmin.getElement(CONFIG_TYPE_DIRECTORY + version + IMFDirectories.MF_DIR_SEPARATOR + type, false);
    	IAttributeSet topSet = typeEl.getAttributes();
    	String initialValues = (String)topSet.getAttribute(INITIAL_VALUES_ATTR);
    	// find the beginning of the initial value
    	String attrValueBeginDelimiter = "<" + attributeName + ">";
    	int delimiterPos = initialValues.indexOf(attrValueBeginDelimiter);
    	int initialValueBegin = delimiterPos + attrValueBeginDelimiter.length();
    	// find the end of the initial value
    	String attrValueEndDelimiter = "</" + attributeName + ">";
    	int endDelimiterPos = initialValues.indexOf(attrValueEndDelimiter);
    	String iValue = initialValues.substring(initialValueBegin, endDelimiterPos);
    	return iValue;
    }
    
    private String findCluster(IDirElement broker)
    {
    	IAttributeSet brokerReferences = (IAttributeSet)broker.getAttributes().getAttribute("CONFIG_ELEMENT_REFERENCES");
        Reference clusterReference = (Reference)brokerReferences.getAttribute("CLUSTER_CONFIG_ELEMENT_REF");
        if (clusterReference != null)
        {
            return clusterReference.getElementName();
        }
        else
        {
            return null;
        }
    }

}
