package com.sonicsw.mf.common.config.upgrade;

import java.util.HashMap;

public class UpgradeRegistry 
{
	private static final boolean DEBUG = System.getProperty("DebugAll.debug", "false").equals("true");
	/**
	 * A map of release versions and objects which implement the 
	 * upgrade for the given versions. For now, this class registers the
	 * implementations of IVersionedConfigUpgrade and IVersionedDomainManagerUpgrade;
	 * this registration could be designed some other way so that the generic
	 * upgrade is not aware of the implementation classes, maybe like the
	 * plugins are registered in the DS, for instance.
	 */
	private static HashMap CONFIG_UPGRADE_MAP = new HashMap();
	
	/**
	 * A map of release versions and classes which implement the
	 * upgrade 
	 */
	private static HashMap DOMAIN_UPGRADE_MAP = new HashMap();
	
	static 
	{
		DOMAIN_UPGRADE_MAP.put("102", new DomainManager75());
		DOMAIN_UPGRADE_MAP.put("103", new DomainManager76());
		DOMAIN_UPGRADE_MAP.put("104", new DomainManager80());
		DOMAIN_UPGRADE_MAP.put("105", new DomainManager85());
		DOMAIN_UPGRADE_MAP.put("106", new DomainManager86());
		CONFIG_UPGRADE_MAP.put("102", new TypedConfigUpgrade75());
		CONFIG_UPGRADE_MAP.put("103", new TypedConfigUpgrade76());
		CONFIG_UPGRADE_MAP.put("104", new TypedConfigUpgrade80());
		CONFIG_UPGRADE_MAP.put("105", new TypedConfigUpgrade85());
		CONFIG_UPGRADE_MAP.put("106", new TypedConfigUpgrade86());
	}
	/**
	 * Register an implementation of IVersionedConfigUpgrade.
	 * @param releaseVersion The version handled by the upgrade object.
	 * @param upgrade The upgrade implementation for the given release version.
	 */
	public static synchronized void registerConfigUpgrade(String releaseVersion, IVersionedConfigUpgrade upgradeObject)
	{
		if (DEBUG)
        {
            System.out.println("UpgradeRegistry.registerConfigUpgrade registering " + upgradeObject + " for version " + releaseVersion);
        }
		if (CONFIG_UPGRADE_MAP == null)
        {
            CONFIG_UPGRADE_MAP = new HashMap();
        }
		CONFIG_UPGRADE_MAP.put(releaseVersion, upgradeObject);
	}
	
	/**
	 * Register an implementation of IVersionedDomainManagerUpgrade.
	 * @param releaseVersion The release version handled by the object being 
	 * registered.
	 * @param domainUpgrade The object which implements 
	 * IVersionedDomainManagerUpgrade
	 */
	public static synchronized void registerDomainUpgrade(String releaseVersion, IVersionedDomainManagerUpgrade domainUpgrade)
	{
		if (DEBUG)
        {
            System.out.println("UpgradeRegistry.registerDomainUpgrade registering " + domainUpgrade + " for version " + releaseVersion);
        }
		if (DOMAIN_UPGRADE_MAP == null)
        {
            DOMAIN_UPGRADE_MAP = new HashMap();
        }
		DOMAIN_UPGRADE_MAP.put(releaseVersion, domainUpgrade);
	}
	
	/**
	 * 
	 * Return the registered implementation of IVersionedConfigUpgrade which
	 * handles the releaseVersion
	 * @param releaseVersion The version that is handled by the returned object
	 * @return The implementation of IVersionedConfigUpgrade
	 * @throws Exception If there's no upgrade object registered for the given
	 * release version.
	 */
	public static synchronized IVersionedConfigUpgrade getConfigUpgrade(String releaseVersion)
	    throws Exception
	{
		if (CONFIG_UPGRADE_MAP == null)
        {
            throw new Exception("There is no upgrade object registered to handle the upgrade from version " + releaseVersion);
        }
		IVersionedConfigUpgrade upgrade = (IVersionedConfigUpgrade)CONFIG_UPGRADE_MAP.get(releaseVersion);
		if (upgrade == null)
        {
            throw new Exception("There is no upgrade class registered to handle the upgrade from version " + releaseVersion);
        }
		return upgrade;
	}
	
	/**
	 * Return the upgrade object to handle the domain manager upgrade from the
	 * given version.
	 * @param releaseVersion Version of the domain to be upgraded.
	 * @return the upgrade object that handles an upgrade from the given release
	 * version.
	 * @throws Exception If there is no domain manager upgrade object registered.
	 */
	public static synchronized IVersionedDomainManagerUpgrade getConfigDomainUpgrade(String releaseVersion)
    throws Exception
    {
	    if (DOMAIN_UPGRADE_MAP == null)
        {
            throw new Exception("There is no upgrade class registered to handle the domain upgrade from version " + releaseVersion);
        }
	    IVersionedDomainManagerUpgrade upgrade = (IVersionedDomainManagerUpgrade)DOMAIN_UPGRADE_MAP.get(releaseVersion);
	    if (upgrade == null)
        {
            throw new Exception("There is no upgrade class registered to handle the domain upgrade from version " + releaseVersion);
        }
	    return upgrade;
    }

}
