package com.sonicsw.mf.common.dirconfig;

/**
 * The status of the current or most recent DS backup operation.
 * <p>
 * If no backup operation has ever been invoked:
 * <ul>
 * <li>isInProgress() == false
 * <li>getStartTime() == -1
 * <li>getCompletionTime() == -1
 * <li>getLocation() == ""
 * </ul>
 * <p>
 * If a backup operation is in progress:
 * <ul>
 * <li>isInProgress() == true
 * <li>getStartTime() == <invocation time at DS in millis>
 * <li>getCompletionTime() == -1
 * <li>getLocation() == backupPath
 * </ul>
 * <p>
 * If the most recent backup operation succeeded:
 * <ul>
 * <li>isInProgress() == false
 * <li>getStartTime() == <invocation time at DS in millis>
 * <li>getCompletionTime() == <completion time at DS in millis>
 * <li>getLocation() == backupPath
 * </ul>
 * <p>
 * If the most recent backup operation failed:
 * <ul>
 * <p>isInProgress() == false
 * <p>getStartTime()() == <invocation time at DS in millis>
 * <p>getCompletionTime() == -1
 * <p>getLocation() == backupPath
 * <ul>
 */

import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.HashMap;

public class BackupStatus implements java.io.Serializable, com.sonicsw.mf.common.runtime.IBackupStatus
{
	private static final long serialVersionUID = 0L;
    private final static int SERIALIZATION_VERSION = 1;
    
    private static final ThreadLocal<SimpleDateFormat> DATE_PARSER_THREAD_LOCAL = new ThreadLocal<SimpleDateFormat>() {
        @Override
        protected SimpleDateFormat initialValue() {
            return new SimpleDateFormat("yy/MM/dd HH:mm:ss");
        }
    };
    HashMap m_status; 
    
    // you can add to these, but not delete. They're also used by DirectoryService to name the attributes of the
    // element where the backup status is persisted.
    public static final String IN_PROGRESS = "IN_PROGRESS";
    public static final String START_TIME = "START_TIME";
    public static final String COMPLETION_TIME = "COMPLETION_TIME";
    public static final String LOCATION = "LOCATION";
    
    public BackupStatus()
    {
       m_status = new HashMap();
       m_status.put(IN_PROGRESS, Boolean.FALSE);
       m_status.put(START_TIME, new Long(-1));
       m_status.put(COMPLETION_TIME, new Long(-1));
       m_status.put(LOCATION, "");
    }
    // implement writeObject and readObject in case we want to change serialization 
    // in a later version
    private void writeObject(java.io.ObjectOutputStream s) throws java.io.IOException
    {
    	s.writeInt(SERIALIZATION_VERSION);
    	s.writeObject(m_status);
    }
    
    private void readObject(java.io.ObjectInputStream s) throws java.io.IOException
    {
    	s.readInt(); //SERIALIZATION_VERSION, we can ignore it for now
        try
        {
            m_status = (HashMap)s.readObject();
        }
        catch (ClassNotFoundException notFound)
        {
            throw new java.io.IOException("Could not read BackupStatus: " + notFound.toString());
        }    	
    }
    
    
    public void startBackup(String backupLocation)
    {   
        m_status.put(IN_PROGRESS, Boolean.TRUE);
        m_status.put(START_TIME, new Long(System.currentTimeMillis()));
        m_status.put(COMPLETION_TIME, new Long(-1));
        m_status.put(LOCATION, backupLocation);
    }
    
    public void failBackup()
    {
        m_status.put(IN_PROGRESS, Boolean.FALSE);
        m_status.put(COMPLETION_TIME, new Long(-1));
    }
    
    public void finishBackup()
    {
        m_status.put(IN_PROGRESS, Boolean.FALSE);
        m_status.put(COMPLETION_TIME, new Long(System.currentTimeMillis()));
    }
    
    @Override
    public boolean isInProgress()
    {
        return ((Boolean)m_status.get(IN_PROGRESS)).booleanValue();
    }
    
    @Override
    public long getStartTime()
    {
        return ((Long)m_status.get(START_TIME)).longValue();
    }
    
    @Override
    public long getCompletionTime()
    {
        return ((Long)m_status.get(COMPLETION_TIME)).longValue();
    }
    
    @Override
    public String getLocation()
    {
        return (String)m_status.get(LOCATION);
    }
    
    @Override
    public String toString()
    {
        String statusString;
        if (getStartTime() == -1)
        {
            statusString = "Start time: -1";
        }
        else
        {
            statusString = "Start time: " + DATE_PARSER_THREAD_LOCAL.get().format(new Date(getStartTime()));
        }
        statusString = statusString + ", location: " + getLocation();
        if (getCompletionTime() == -1)
        {
            statusString = statusString + ", completion time: -1";
        }
        else
        {
            statusString = statusString + ", completion time: " + DATE_PARSER_THREAD_LOCAL.get().format(new Date(getCompletionTime()));
        }
        statusString = statusString + ", in progress: " + isInProgress();

        return statusString;
    }
    
    public void setInProgress(Boolean inProgress)
    {
        m_status.put(IN_PROGRESS, inProgress);
    }
    
    public void setStartTime(Long startTime)
    {
        m_status.put(START_TIME, startTime);
    }
    
    public void setCompletionTime(Long completionTime)
    {
        m_status.put(COMPLETION_TIME, completionTime);
    }
    
    public void setLocation(String location)
    {
        m_status.put(LOCATION, location);
    }
}
