/* Copyright (c) 2007 Progress Software Corporation.  All Rights Reserved. */

package com.sonicsw.mf.common.dirconfig;

import java.io.StringReader;

import com.sonicsw.mf.common.config.impl.Element;
import com.sonicsw.mf.common.xml.Validator;

/**
 * ElementFactory creates IDirElement object programmatically.
 */
public final class ElementFactory
{
   /**
    * Creates a new IDirElement object.
    *
    * @param elementName the name of the element in the directory hierarchy
    * @param type the type of the element (e.g. messageServer)
    * @param releaseVersion the release version of the element (e.g. "3.5")
    *
    * @return Empty instance of IDirElement
    *
    */
    public static IDirElement createElement(String elementName, String type, String releaseVersion)
    {
        return new Element(elementName, type, releaseVersion);
    }

   /**
    * Creates a new IDirElement object.
    *
    * @param elementName the name of the element in the directory hierarchy
    * @param type the type of the element (e.g. messageServer)
    * @param releaseVersion the release version of the element (e.g. "3.5")
    * @param createTemplate if true then the new element is a template
    *
    * @return Empty instance of IDirElement
    *
    */
    public static IDirElement createElement(String elementName, String type, String releaseVersion, boolean createTemplate)
    {
        return new Element(elementName, type, releaseVersion, createTemplate);
    }

   /**
    * Creates a new IDirElement instance object. When the object is submitted for creation at the DS side, the
    * DS calculates the delta bewteen the templateName template and this element and subclass elementName with that 
    * delta
    *
    * @param elementName the name of the element in the directory hierarchy
    * @param type the type of the element (e.g. messageServer)
    * @param releaseVersion the release version of the element (e.g. "3.5")
    * @param templateName the name of the template to sublass from
    *
    * @return Empty instance of IDirElement instance
    *
    */
    public static IDirElement createInstanceElement(String elementName, String type, String releaseVersion, String templateName)
    {
        return new Element(elementName, type, releaseVersion, templateName);
    }

   /**
    * Creates an element from an XML document. The created element is in the same state
    * as a newly created element - it has an INITIAL_VERSION version and no creation timestamp (value is 0).
    *
    * @param XMLDocument the XML representation of the element
    * @param domainName specifies domain name, can be null
    *
    * @return An IDirElement object
    *
    * @exception InvalidXMLException if the XML document is invalid
    *
    */
    public static IDirElement importFromXML(String XMLDocument, String domainName) throws InvalidXMLException 
    { 
        if(XMLDocument.length() == 0 || XMLDocument == null)
        {
            throw new InvalidXMLException("XML file can't be empty");
        }
        Validator xmlValidator = new Validator(new StringReader(XMLDocument));    
        //do validation first
        xmlValidator.createSAXParser();
        xmlValidator.setDirectoryService(null);
        xmlValidator.setValidation(true);
        try{
            xmlValidator.parseData();  //throws exception wrapped into SAXParser exception  
            //do import 
            xmlValidator.resetXMLData(new StringReader(XMLDocument));
            xmlValidator.setContentHandler(domainName);
            xmlValidator.parseData(); //throws exception wrapped into SAXParser exception
        }
        catch(Exception e)
        {
            throw new InvalidXMLException(e.getMessage());
        }
        return xmlValidator.getContentHandler().getElement();
    }

    /**
     * Creates elements from an XML document. The elements have an INITIAL_VERSION version and 
     * no creation timestamp (value is 0).
     *
     * @param XMLDocument the XML representation of the elements
     * @param domainName specifies domain name, can be null
     *
     * @return An array of IDirElement
     *
     * @exception InvalidXMLException if the XML document is invalid
     *
     */
    public static IDirElement[] importElementsFromXML(String XMLDocument, String domainName) throws InvalidXMLException 
    { 
        if(XMLDocument.length() == 0 || XMLDocument == null)
        {
            throw new InvalidXMLException("XML file can't be empty");
        }
        
        Validator xmlValidator = new Validator(new StringReader(XMLDocument));    
        //do validation first
        xmlValidator.setKeepElements(true);
        xmlValidator.createSAXParser();
        xmlValidator.setDirectoryService(null);
        xmlValidator.setValidation(true);
        try{
            xmlValidator.parseData();  //throws exception wrapped into SAXParser exception  
            //do import 
            xmlValidator.resetXMLData(new StringReader(XMLDocument));
            xmlValidator.setContentHandler(domainName);
            xmlValidator.parseData(); //throws exception wrapped into SAXParser exception
        }
        catch(Exception e)
        {
            throw new InvalidXMLException(e.getMessage());
            }
            
        return xmlValidator.getContentHandler().getElements();
    }

    public static IDirElement importElementFromXML(String XMLDocument, String domainName, String elType)
        throws InvalidXMLException
    {
        IDirElement[] elements = importElementsFromXML(XMLDocument, domainName);
        for (int i=0; i<elements.length; i++)
        {
            if (elements[i].getIdentity().getType().equals(elType))
            {
                return elements[i];
            }
        }
        return null;
    }
}
