/*
 * Copyright (c) 2001 Sonic Software. All Rights Reserved.
 */

package com.sonicsw.mf.common.dirconfig;
import java.io.Serializable;

import com.sonicsw.mf.common.config.IBasicElement;
import com.sonicsw.mf.common.config.IDeltaElement;
import com.sonicsw.mf.common.config.IElement;
import com.sonicsw.mf.common.config.INextVersionToken;
import com.sonicsw.mf.common.config.ReadOnlyException;

/**
 * This interface represents a Configuration Element object which resides in the MF Directory, the local cache,
 * or a loose configuration element used by tools and utilities. An IElement  object can be 
 * modified; the modification is applied to the MF Directory central repository by creating
 * a IDeltaDirElement from this modified element and passing it to the central repository through the DirectoryService interface.
 * Each Configuration Element version has a unique
 * identity represented by an IElementIdentity object. The uniqueness of the identity is created by its
 * name, creation timestamp and version number. The IElementIdentity object also contains the element's type and release version.
 * All the information of the IElement object is contained in a single IAttributeSet object. 
 * 
 *INote that there is no direct support for element renaming, an element can be renamed only by
 * the recreation of a copy with a new name and the of the original element.
 */

public interface IDirElement extends IElement, Serializable
{

 /**
  * Sets a new element name. setName can be only called in new elements, before doneUpdate is called.
  *
  * @param elementName the name of the element in the directory hierarchy
  *
  * @exception ReadOnlyException if this is a read-only element
  */
  public void setName(String elementName) throws ReadOnlyException;

 /**
  * Creates a writable copy of this (writable or read-only) object. 
  * A call to IElement.doneUpdate() in the new copy return the delta between this object and the modified version of the clone.
  * This functionality allows the administration tool to use the clone as a cratch pad and then use the delta to apply the modifications
  * to the "real" element. Note that the delta of the clone CANNOT BE USED in IDirectoryAdminService.setElement to modify the element in the
  * directory service - use IDirectoryAdminService.cloneElement for that.
  *
  * NOTE: The purpose of this function is to create scratch pads where values can be locally modifed.
  *
  * NOTE: Clones are not cached in a limited size memory cache, so care must be taken not to keep too many and/or too large clones.
  *
  * @return the created IElement object
  *
  */
  @Override
public IElement createWritableClone();

 /**
  * Returns the delta between the current version of the object in the central repository and the modified element.
  * If the element is new (does not yet exist in the central repository) then the delta is between an empty element
  * and the current state of the element. The doneUpdate() call turns this element into a read-only element. Further doneUpdate() 
  * calls return null. 
  *
  * @return a delta element object if this element was modified or an element object if this element is new
  *
  * @exception ReadOnlyException if this is a read-only element
  *
  */
  public IBasicElement doneUpdate() throws ReadOnlyException;

 /**
  * Returns the delta between the current version of the object in the central repository and the modified element for subclassing
  * purpose If the element is new (does not yet exist in the central repository) then the delta is between an empty element
  * and the current state of the element. The doneUpdate() call turns this element into a read-only element. Further doneUpdate()
  * calls return null. The resulting delta does not contain any IDeltaAttributeSet objects but rather complete IAttributeList objects
  * that overwrite the IAttributeList value in the super class.
  *
  * @return a delta element object if this element was modified or an element object if this element is new
  *
  * @exception ReadOnlyException if this is a read-only element
  *
  */
  public IBasicElement doneUpdateForSubclassing() throws ReadOnlyException;


  
  /**
  *
  * Returns true if this element was modified and doneUpdate() was not called yet.
  *
  * @return true if this element was modified.
  */
  public boolean isDirty();

  /**
  *
  * Returns the list of element names sub-classed from this.
  *
  * @return the list of element names sub-classed from this (a 0 length array if non).
  */
  public String[] getSubclassedList();

  /**
  *
  * Returns the name of the super element of this (if sub-classed).
  *
  * @return  the name of the super element of this (null if not sub-classed)
  */
  public String getSuperElementName();

 
  /**
  * Called on a just updated read-only element, to create an updatapble next version element. the token is returned
  * from DS setElement and setElements calls.
  * Example:
  * 
  * elements[0] = element0.doneUpdate();
  * elements[1] = element1.doneUpdate();
  * INextVersionToken token = ds.setElements(elements, null, view.doneUpdate())
  * element0 = element0.setNextVersion(token);
  * element1 = element1.setNextVersion(token);
  * view = view.setNextVersion(token);
  *
  * @param the next version token returned from the Directory Service
  *
  * @return an updatable next version element
  *
  */
  public IDirElement getNextVersion(INextVersionToken token);

  /**
   * Returns true if this element is a template.
   *
   * @return true if this element is a template.
   */
  public boolean isTemplate();

  /**
   * Returns true if this element is sub-classed.
   *
   * @return true if this element is sub-classed.
   */
  public boolean isSubclassedElement();

  /**
   * Returns the sub-classing delta of this sub-classed element.
   *
   * @return the sub-classing delta or null if it's not a sub-classed element or if 
   *         the caller didn't request to attach the delta.
   */
  public IDeltaElement getSubclassingDelta();

}
