package com.sonicsw.mf.common.info;

import java.lang.reflect.Method;

import com.sonicsw.mf.common.info.impl.AttributeInfo;
import com.sonicsw.mf.common.info.impl.NotificationInfo;
import com.sonicsw.mf.common.info.impl.OperationInfo;
import com.sonicsw.mf.common.runtime.INotification;

/**
 * This factory is used to create the objects needed to describe a components management interface.
 *
 * When needed, the container requests the components management interface (described via the
 * factory created meta data classes).
 */
public final class InfoFactory
{
    /**
     * This method creates an object that encapsulates the meta-info for a management attribute
     * to be exposed by a component.
     *
     * @param name        The programmatic name of the attribute.
     * @param type        The class name of the attribute type.
     * @param description A hman readable description of the attribute (should be suitable for
     *                    displaying as help in a tool).
     * @param isReadable  The attribute is readable.
     * @param isWritable  The attribute is writeable.
     */
    public static IAttributeInfo createAttributeInfo(String name, String type, String description, boolean isReadable, boolean isWritable)
    {
        return new AttributeInfo(name, type, description, isReadable, isWritable);
    }

    /**
     * This method creates an object that encapsulates the meta-info for a management operation
     * to be exposed by a component.
     *
     * @param description A hman readable description of the operation.
     * @param method      The java.lang.reflect.Method object describing the operation.
     */
    public static IOperationInfo createOperationInfo(String description, Method method)
    {
        return new OperationInfo(description, method);
    }

    /**
     * This method creates an object that encapsulates the meta-info for a management notification
     * to be exposed by a component.
     *
     * @param category    Sets the major category for this notification. Allowable values are defined
     *                    using <category name>_CATEGORY constants defined in this class.
     * @param subCategory Sets the sub category for this notification. The sub-category must be a single
     *                    word containing only alphabetical or numeric characters (no delimiters).
     * @param eventName   This uniquely identifies the notification within category/subCategory/type. The
     *                    name must be a single word containing only alphabetical or numeric characters
     *                    (no delimiters). Examples names might be "connect", "messageDiscarded",
     *                    "containerShutdown".
     * @param description A hman readable description of the notification.
     */
    public static INotificationInfo createNotificationInfo(short category, String subCategory, String eventName, String description)
    {
        String[] notifTypes = new String[3];
        notifTypes[0] = INotification.CATEGORY_TEXT[category];
        notifTypes[1] = subCategory;
        notifTypes[2] = eventName;

        return new NotificationInfo(notifTypes, INotification.CLASSNAME, description);
    }
}
