/*
 * Copyright (c) 2002, Sonic Software Corporation. All Rights Reserved.
 *
 * This software is the confidential and proprietary information of Sonic
 * Software Corporation. ("Confidential Information").  You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Progress.
 *
 * PROGRESS MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE
 * SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 * PURPOSE, OR NON-INFRINGEMENT. PROGRESS SHALL NOT BE LIABLE FOR ANY DAMAGES
 * SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR DISTRIBUTING
 * THIS SOFTWARE OR ITS DERIVATIVES.
 *
 * CopyrightVersion 1.0
 *
 */

package com.sonicsw.mf.common.license;

/*
 * This class builds on the normal LicenseDecoder, which takes the basic
 * Sonic control code and converts it to:
 *  <li>serial number
 *  <li>product code
 *  <li>use count
 * <p>
 * This class returns all those items, plus *some* "inferred" information, such as:
 *  <li>getProductName() -- name of Product Code
 *  <li>isEvaluationMode() -- true, for eval editions
 *  <li>etc.
 * Concrete implementations of ProductDecoder may provide additonal "inferred" information.
 * <p>
 * Originally this class was contained in the progress.message.broker package,
 * and was a concrete final class.
 * However, it was modified and move in order to provide a "common"
 * set of license management and validation code, that could be used
 * by various products such as SonicMQ, ESB, the Logger product, etc.
 * <p>
 * This class is tied to a particular version of a product.  When the product
 * changes, this should change as well.
 *
 */
public abstract class ProductDecoder
{
    // Member variables
    private String m_controlNumber;     // 15 character Control Number
    private String m_version;           // Full version string e.g 5.0.2a
    protected int m_count = -1 ;          //Use Count
    protected int m_serial = -1;        //Serial Number
    protected int m_productCode = -1;   //Product Code

    // Computed values, based on Product Code
    protected int m_productIndex = -1;     //Index for the product code
    protected boolean m_eval = false;      //Evaluation Mode
    protected int m_evalPeriod = -1;       //Evaluation period, days (-1 = unlimited)

    /**
     * Create a ProductDecoder based on a Control Number. This can be
     * used to get the capabilities of the specific version of a product.
     * This throws the same Exceptions as LicenseDecoder.
     * <p>
     * One difference between this and LicenceDecoder is that with ProductDecoder
     * you can check for a valid Product Code by calling <code>isValidProduct()</code>
     * (which will return false if the product code is not recognized).
     * @param version The version as a String
     * @param controlNumber The control number for the version.
     * @throws Exception if there is a problem
     */
    public ProductDecoder (String version, String controlNumber)
    throws Exception
    {
        // Use LicenseDecoder to check things like CRC, password, etc.
        m_controlNumber = controlNumber;
        m_version = version;
    }

    /**
     *
     *
     * @return
     */
     abstract protected void validate(LicenseDecoder ld);

    /**
     * Get the product name as a String.
     * @return the Product Name as a string, or null if the product were invalid
     */
    abstract public String getProductName();

    /**
     * Return a String Array containing a list of features supported
     * by the version.  This can be used with <code>getFeatureValue</code>
     * to get a list of properties (as Strings).
     * @return A list (String[]) of features.
     */
    abstract public String[] getFeatureList();

    /**
     * Return a String value for a given feature.
     * <p>
     * Programmers might want to use methods, like <code>isEvaluationMode()</code>
     * instead of this String property, as the former is cast as the correct
     * primative data type.
     * @param featureKey the key (name) of the feature.
     * @return The value of the feature, or null, if the feature is not supported.
     */
    abstract public String getFeatureValue(String feature);

    /**
     * Get the int for the number of days in the Evaluation Period.
     * @return trial period, in days (or -1 for unlimited)
     */
    public int getEvaluationPeriod()
    {
       return m_evalPeriod;
    }

    /**
     * Get the Serial Number as an int (from the Control Number)
     * @return Serial Number, or -1 if Control Number was invalid.
     */
    public final int getSerialNumber()
    {
        return m_serial;
    }

    /**
     * Get the "Use Count" as an int (from the Control Number)
     * @return Count, or -1 if Control Number was invalid.
     */
    public final int getCount()
    {
        return m_count;
    }

    /**
     * Get the Product Code as an int (from the Control Number).
     * @return Product Code, or -1 if Control Number was invalid.
     */
    public final int getProductCode()
    {
        return m_productCode;
    }

    /**
     * Return true if the product code contained in the Control Number
     * is recognized for this version.  (from the Control Number).
     * @return true, only if Control Number is valid, and Product Code is recognized.
     */
    public final boolean checkProductCode()
    {
        return (m_productIndex < 0 ? false : true);
    }

    /**
     * Get the Control Number passed into the constructor.
     * @return Control Number (15 character String)
     */
    public final String getControlNumber()
    {
        return m_controlNumber;
    }

    /**
     * Get the Version String passed into the constructor.
     * @return Version (e.g. "5.01a")
     */
    public final String getVersion()
    {
        return m_version;
    }
}