package com.sonicsw.mf.common.metrics;

import java.io.Serializable;

/**
 * Individual metrics are identified as leaves in a tree. A metric identity identifies one
 * of the following:
 * <pre>
 *  - a branch of a tree of related metrics (node)
 *  - a metric group (node)
 *  - an individual metric (leaf)
 * </pre>
 * Each node/leaf in a metric tree has a name component (a string) and is fully described
 * by an array of name components that describe its path in the tree starting from the root.
 * A canonical form of the name components is evaluated by creating a '.' delimited list of
 * those name components (however the canonical form cannot be reduced to the original
 * name components as an individual name component is allowed to contain the '.' character).
 * <p>
 * A metric tree may be as small as a single node.
 */
public interface IMetricIdentity
extends Serializable
{
    public static final IMetricIdentity[] EMPTY_METRIC_IDENTITY_ARRAY = new IMetricIdentity[0];

    /**
     * Get the name components the uniquely identify the metric identity.
     */
    public String[] getNameComponents();

    /**
     * Get the canonical name of the metric identity where by each name component is
     * delimited by a period '.' character. If any of the name components contain
     * a period '.' or percent '%' character these will not be be escaped (this is
     * different to the absolute name) e.g. "queue.message.Count.SoincMQ.routingQueue"
     * <p>
     * Note: The return value from this method should not be used to create a new metric
     *       identity instance (using the MetricsFactory).
     *
     * @see #getAbsoluteName()
     */
    public String getName();

    /**
     * Get the absolute name of the metric identity where by each name component is
     * delimited by a period '.' character. If any of the name components contain
     * a period '.' or percent '%' character these will be escaped using a
     * percent character e.g. "queue.message.Count.SoincMQ%.routingQueue"
     *
     * @see #getName()
     */
    public String getAbsoluteName();

    /**
     * Tests if this metric is an instance of the given metric identity (branch
     * of a tree of metrics or metric group). If given identity is an instance
     * pattern e.g. "a.b.c*", then tests if this metric matches the pattern.
     */
    public boolean isInstanceOf(IMetricIdentity id);
}
