/**
 * Copyright (c) 1998-2002 Sonic Software Corporation. All Rights Reserved.
 *
 * This software is the confidential and proprietary information of Sonic
 * Software Corpoation. ("Confidential Information").  You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Progress.
 *
 * SONIC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE
 * SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 * PURPOSE, OR NON-INFRINGEMENT. SONIC SHALL NOT BE LIABLE FOR ANY DAMAGES
 * SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR DISTRIBUTING
 * THIS SOFTWARE OR ITS DERIVATIVES.
 *
 * CopyrightVersion 1.0
 */
package com.sonicsw.mf.common.metrics;

import java.util.ArrayList;
import java.util.StringTokenizer;

import com.sonicsw.mx.util.IEmptyArray;

import com.sonicsw.mf.common.metrics.impl.AggregateMetric;
import com.sonicsw.mf.common.metrics.impl.Alert;
import com.sonicsw.mf.common.metrics.impl.HistoricalMetric;
import com.sonicsw.mf.common.metrics.impl.Metric;
import com.sonicsw.mf.common.metrics.impl.MetricIdentity;
import com.sonicsw.mf.common.metrics.impl.MetricInfo;
import com.sonicsw.mf.common.metrics.impl.MetricsData;

/**
 * Component developers are required to use the factory to create instances of implementations
 * of the core metrics interfaces (of this pakage). The exception to this rule is that component
 * developers can create their own IMetricsManager implementation - however the preferred model is
 * to use the default manager provided by this factory.
 */
public final class MetricsFactory
{
    /**
     * Returns a metric identity. The values supplied should be valid for the life cycle of the
     * component.
     *
     * @param nameCompnents An array of strings that describe the path of the metric in
     *                      in the components tree of metrics.
     */
    public static IMetricIdentity createMetricIdentity(String[] nameComponents)
    {
        return new MetricIdentity(nameComponents);
    }

    /**
     * Returns a metric identity. The values supplied should be valid for the life cycle of the
     * component.
     *
     * @param parentID      The metric identity of the parent.
     * @param nameComponent The name component of the child of the parent. Special characters '.'
     *                      and '\' are treated as part of the name component rather than as further
     *                      delimiting.
     */
    public static IMetricIdentity createMetricIdentity(IMetricIdentity parentID, String nameComponent)
    {
        String[] parentNameComponents = parentID.getNameComponents();
        String[] nameComponents = new String[parentNameComponents.length + 1];
        System.arraycopy(parentNameComponents, 0, nameComponents, 0, parentNameComponents.length);
        nameComponents[parentNameComponents.length] = nameComponent;

        return createMetricIdentity(nameComponents);
    }

    /**
     * Returns a metric identity. The values supplied should be valid for the life cycle of the
     * component.
     *
     * @param absoluteName  An a period '.' delimited string that describe the path of the metric in
     *                      in the components tree of metrics (each token representing a component
     *                      of the metric name). If a component of a metric name contains periods '.'
     *                      or percents '%' then those should be escaped by a percent e.g. "%."
     */
    public static IMetricIdentity createMetricIdentity(String absoluteName)
    {
        ArrayList nameComponents = new ArrayList();
        StringTokenizer st = new StringTokenizer(absoluteName, ".%", true);

        String nameComponent = null;
        while (st.hasMoreTokens())
        {
            String token = st.nextToken();
            if (token.equals("."))
            {
                if (nameComponent == null)
                {
                    nameComponents.add("");
                }
                else
                {
                    nameComponents.add(nameComponent);
                    nameComponent = null;
                }
            }
            else
            if (token.equals("%"))
            {
                // save the next token (there should always be one)
                if (nameComponent == null)
                {
                    nameComponent = st.nextToken();
                }
                else
                {
                    nameComponent += st.nextToken();
                }
            }
            else
            {
                if (nameComponent == null)
                {
                    nameComponent = token;
                }
                else
                {
                    nameComponent += token;
                }
            }
        }

        // anything left ?
        if (nameComponent != null)
        {
            nameComponents.add(nameComponent);
        }

        return createMetricIdentity((String[])nameComponents.toArray(IEmptyArray.EMPTY_STRING_ARRAY));
    }

    /**
     * Returns a metric that encapsulates the metric identity and its value.
     */
    public static IMetric createMetric(IMetricIdentity id, long value, long currencyTimestamp)
    {
        return new Metric(id, value, currencyTimestamp);
    }

    /**
     * Returns a historical metric that encapsulates the metric identity, its value and the source
     * of the metric. For use by the framework and not application components.
     */
    public static IHistoricalMetric createMetric(String source, IMetric metric)
    {
        return new HistoricalMetric(source, metric.getMetricIdentity(), metric.getValue(), metric.getCurrencyTimestamp());
    }

    /**
     * Returns an aggregate metric that encapsulates the metric identity, its aggregate values and the sources
     * that contributed to the aggregated value of the metric. For use by the framework and not
     * application components.
     */
    public static IAggregateMetric createMetric(String[] sources, IMetricIdentity id, long[] values, long currencyTimestamp)
    {
        return new AggregateMetric(sources, id, values, currencyTimestamp);
    }

    /**
     * Returns a info object that encapsulates the meta-data pertaining to an individual metric
     * or instance metric.
     *
     * @param id               The metric identity.
     * @param valueType        The metric value type (see IValueType defined types).
     * @param description      The optional free form description of the metric.
     * @param extendedDataData The optional free form data associated with the metric.
     * @param isInstanceMetric Indicates the meta-data describes metrics that are parented by the instance metric.
     * @param isDynamic        Indicates the metric can be enabled/disabled at runtime.
     * @param supportsHighAlerts   Indicates the metric supports high threshold alerts.
     * @param supportsLowAlerts    Indicates the metric supports low threshold alerts.
     * @param units            The units that this metric uses.
     *
     * @see #createMetricInfo(IMetricIdentity, short, String, String, boolean, boolean)
     * @see IValueType
     */
    public static IMetricInfo createMetricInfo(IMetricIdentity id, short valueType, String description, String extendedDataData,
            boolean isInstanceMetric, boolean isDynamic,
            boolean supportsHighAlerts, boolean supportsLowAlerts, String units)
    {
        return new MetricInfo(id, valueType, description, extendedDataData, isInstanceMetric, isDynamic, supportsHighAlerts, supportsLowAlerts, units);
    }

    /**
     * Returns a info object that encapsulates the meta-data pertaining to an individual metric
     * or instance metric. Does not specify alerts.
     *
     * @see #createMetricInfo(IMetricIdentity, short, String, String, boolean, boolean, boolean, boolean, String)
     */
    public static IMetricInfo createMetricInfo(IMetricIdentity id, short valueType, String description, String extendedDataData, boolean isInstanceMetric, boolean isDynamic)
    {
        return createMetricInfo(id, valueType, description, extendedDataData, isInstanceMetric, isDynamic, false, false, null);
    }

    /**
     * Returns a metrics data object that encapsulates the given metrics. The currency timestamp
     * of the metrics data object will be set to the provided value.
     */
    public static IMetricsData createMetricsData(IMetric[] metrics, long currencyTimestamp)
    {
        return new MetricsData(metrics, currencyTimestamp);
    }

    /**
     * Returns an aggregate metrics data object that encapsulates the given aggregrate metrics. The currency timestamp
     * of the metrics data object will be set to the provided value.
     */
    public static IAggregateMetricsData createMetricsData(IAggregateMetric[] metrics, long currencyTimestamp)
    {
        return new MetricsData(metrics, currencyTimestamp);
    }

    /**
     * Creates an Alert object
     *
     * @param IMetricIdentity id    The metric id this alert is associated with
     * @param isHighThreshold       Whether this is a high or low threshold
     * @param threshold             The threshold to set the alert to
     */
    public static IAlert createAlert(IMetricIdentity metricId, boolean isHighThreshold, long threshold)
    {
        if (metricId == null)
        {
            throw new IllegalArgumentException("Metric identity cannot be null");
        }
        return new Alert(metricId, isHighThreshold, threshold);
    }
    
    /**
     * Creates an array of Alert objects from a comma delimited threshold string
     *
     * @param IMetricIdentity id    The metric id these alert are associated with
     * @param isHighThreshold       Whether these are high or low thresholds
     * @param threshold             A comma delimited list of threshold values
     */
    public static IAlert[] createAlerts(IMetricIdentity metricId, boolean isHighThreshold, String thresholds)
    {
        if (metricId == null)
        {
            throw new IllegalArgumentException("Metric identity cannot be null");
        }

        String[] tokens;
        ArrayList alertsList = new ArrayList();
        if (thresholds != null)
        {
            tokens = thresholds.split(",");
            for (int i = 0; i < tokens.length; i++)
            {
                try
                {
                    long value = Long.decode(tokens[i]).longValue();
                    IAlert alert = MetricsFactory.createAlert(metricId, isHighThreshold, value);
                    alertsList.add(alert);
                }
                // Skip threshold if value is invalid
                catch (NumberFormatException nfe)
                {
                }
            }
        }
        return (IAlert[])alertsList.toArray(new IAlert[alertsList.size()]);
    }
}
