/**
 * Copyright (c) 1998-2002 Sonic Software Corporation. All Rights Reserved.
 *
 * This software is the confidential and proprietary information of Sonic
 * Software Corpoation. ("Confidential Information").  You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Progress.
 *
 * SONIC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE
 * SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 * PURPOSE, OR NON-INFRINGEMENT. SONIC SHALL NOT BE LIABLE FOR ANY DAMAGES
 * SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR DISTRIBUTING
 * THIS SOFTWARE OR ITS DERIVATIVES.
 *
 * CopyrightVersion 1.0
 */
package com.sonicsw.mf.common.metrics.impl;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.Serializable;
import java.util.GregorianCalendar;
import java.util.Objects;

import com.sonicsw.mf.common.metrics.IAlert;
import com.sonicsw.mf.common.metrics.IMetricIdentity;
import com.sonicsw.mf.common.metrics.MetricsFactory;

/**
 * Encapsulates an alert and its properties and behavior
 * Is associated with a metric.
 */
public class Alert implements IAlert, Serializable
{
    private IMetricIdentity m_metricId;
    private long m_threshold = 0;
    private boolean m_hiLo = true;
    private long m_lastValue = -1;
    private GregorianCalendar m_initialAlertTime = null;
    private boolean m_isRepeat = false;
    private boolean m_isExceeded = false;
    private static final long serialVersionUID = 1589193560369980527L;
    private static final short m_serialVersion = 0;

    public Alert() {}   // for serialization only

    public Alert(IMetricIdentity metricId,  boolean hiLo, long threshold)
    {
        m_metricId = metricId;
        m_threshold = threshold;
        m_hiLo = hiLo;
    }
    // IAlert
    @Override
    public IMetricIdentity getMetricIdentity(){ return m_metricId; }
    @Override
    public boolean isHighThreshold() { return m_hiLo; }
    @Override
    public long getThresholdValue() { return m_threshold; }

    // internal use
    public boolean isExceeded() { return m_isExceeded; }
    public GregorianCalendar getInitialAlertTime() { return m_initialAlertTime; }
    public boolean isRepeat() { return this.m_isRepeat; }

    // set threshold value (for internal use)
    public boolean setThresholdValue(long newThreshold)
    {
        boolean ret = false;
        m_threshold = newThreshold;
        if (m_lastValue >=0)    // initialized
        {
            if ((m_hiLo && newThreshold > m_lastValue) || (!m_hiLo && newThreshold < m_lastValue))
            {
                m_isExceeded = true;
                ret = true;
            }
        }
        return ret;
    }


    // Check whether new value should trigger notification
    public boolean check(long value, boolean repeat)
    {
        // update last value
        m_lastValue = value;

        // drop thru condition is that the threshold has been exceeded
        if (m_hiLo) // high threshold
        {
            if (value > m_threshold)
            {
                if (m_isExceeded && !repeat)
                {
                    return false;
                }
            }
            else
            {
                // reset
                m_isExceeded = false;
                m_initialAlertTime = null;
                m_isRepeat = false;
                return false;
            }
        }
        else
        {
            if (value < m_threshold)
            {
                if (m_isExceeded && !repeat)
                {
                    return false;
                }
            }
            else
            {
                // reset
                m_isExceeded = false;
                m_initialAlertTime = null;
                m_isRepeat = false;
                return false;
            }
        }

        m_isExceeded = true;
        
        // set the date the first time its exceeded
        if (m_initialAlertTime == null)
        {
            m_initialAlertTime = new GregorianCalendar();
        }
        else
        {
            m_isRepeat = true;
        }
        
        return true;
    }

    public boolean isInstanceOf(IAlert parent)
    {
        if (!(parent instanceof IAlert))
        {
            return false;
        }

        if(this.getMetricIdentity().isInstanceOf(parent.getMetricIdentity())
               && (this.isHighThreshold() == parent.isHighThreshold())
                && (this.getThresholdValue() == parent.getThresholdValue()) )
        {
            return true;
        }
        else
        {
            return false;
        }
    }

    @Override
    public boolean equals(Object obj)
    {
        if (!(obj instanceof IAlert))
        {
            return false;
        }

        IAlert other = (IAlert) obj;
        if (this.getMetricIdentity().equals(other.getMetricIdentity())
                && (this.isHighThreshold() == other.isHighThreshold())
                && (this.getThresholdValue() == other.getThresholdValue()) )
        {
            return true;
        }
        else
        {
            return false;
        }
    }
    
    @Override
    public int hashCode() {
        return Objects.hash(getMetricIdentity(), isHighThreshold(), getThresholdValue());
    }

    //TODO: override hashCode() for easy test of contains, etc...

    @Override
    public String toString()
    {
        return '[' + m_metricId.getName() + "]hiLo=" + m_hiLo + ",threshold=" + m_threshold;
    }

    //
    // Serialization
    //

    // You can add to these, but never remove
    private static final short ID_NAME_FIELD = 0;
    private static final short THRESHOLD_FIELD = 1;
    private static final short IS_HIGH_THRESHOLD_FIELD = 2;

    // This is done more efficiently than in other places where we employ version
    // handling serialization code .. as there may be many metrics to describe

    private void writeObject(ObjectOutputStream stream)
    throws IOException
    {
        short fieldCount = 3;

        // we know how many fields we will write and serial version will always be written
        stream.writeShort(fieldCount);
        stream.writeShort(m_serialVersion);

        // Alert specific fields
        stream.writeShort(ID_NAME_FIELD);
        stream.writeObject(m_metricId.getNameComponents());

        stream.writeShort(THRESHOLD_FIELD);
        stream.writeLong(m_threshold);

        stream.writeShort(IS_HIGH_THRESHOLD_FIELD);
        stream.writeBoolean(m_hiLo);
    }

    private void readObject(ObjectInputStream stream)
    throws IOException, ClassNotFoundException
    {
        // read the number of items and stuff them in a hash map
        short numFields = stream.readShort();
        short serialVer = stream.readShort();
        for (int i = 0; i < numFields; i++)
        {
            short field = stream.readShort();
            switch (serialVer)
            {
                // case olderVersion<n> ...
                default:
                {
                    if (field == ID_NAME_FIELD)
                    {
                        String[] nameComponents = (String[])stream.readObject();
                        m_metricId = MetricsFactory.createMetricIdentity(nameComponents);
                    }
                    else if (field == THRESHOLD_FIELD)
                    {
                        m_threshold = stream.readLong();
                    }
                    else if (field == IS_HIGH_THRESHOLD_FIELD)
                    {
                        m_hiLo = stream.readBoolean();
                    }
                    break;
                }
            }
        }
    }

    @Override
    public boolean sameMetricIdentity(IMetricIdentity other) {
        if(other == null)
            return false;

        return this.getMetricIdentity().equals(other);
    }
}

