package com.sonicsw.mf.common.metrics.impl;

import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.util.StringTokenizer;

import com.sonicsw.mf.common.metrics.IMetricIdentity;

/**
 * Class is not for reference and construction, rather you should use the factory and interface.
 *
 * @see com.sonicsw.mf.common.metrics.MetricsFactory#createMetricIdentity(String[] nameComponents, long hash)
 */
public final class MetricIdentity
implements IMetricIdentity
{
    static final long serialVersionUID = 619761350679132533L;
    private static final short m_serialVersion = 0;

    private String[] m_nameComponents;
    private long m_hash;

    public MetricIdentity(String[] nameComponents)
    {
        m_nameComponents = nameComponents;
    }

    /**
     * @see com.sonicsw.mf.common.metrics.IMetric#getNameComponents()
     */
    @Override
    public String[] getNameComponents() { return m_nameComponents; }

    /**
     * @see com.sonicsw.mf.common.metrics.IMetric#getName()
     */
    @Override
    public String getName()
    {
        StringBuffer sb = new StringBuffer();

        for (int i = 0; i < m_nameComponents.length; i++)
        {
            if (i > 0)
            {
                sb.append('.');
            }
            sb.append(m_nameComponents[i]);
        }

        return sb.toString();
    }

    /**
     * @see com.sonicsw.mf.common.metrics.IMetric#getAbsoluteName()
     */
    @Override
    public String getAbsoluteName()
    {
        StringBuffer sb = new StringBuffer();

        for (int i = 0; i < m_nameComponents.length; i++)
        {
            if (i > 0)
            {
                sb.append('.');
            }
            StringTokenizer st = new StringTokenizer(m_nameComponents[i], ".%", true);
            while (st.hasMoreTokens())
            {
                String token = st.nextToken();
                if (token.equals(".") || token.equals("%"))
                {
                    sb.append('%');
                }
                sb.append(token);
            }
        }

        return sb.toString();
    }

    /**
     * An individual metric can have a non-zero hash associated with it for faster indexing.
     * If a metric supports such a hash, then this method returns that hash.
     */
    public long getHash() { return m_hash; }

    /**
     * An individual metric can have a non-zero hash associated with it for faster indexing.
     * This method sets that hash.
     */
    public void setHash(long hash) { m_hash = hash; }

    /**
     * Tests if the given identity has matching name components.
     */
    @Override
    public boolean equals(Object id)
    {
        if (!(id instanceof IMetricIdentity))
        {
            return false;
        }

        String[] nameComponents = ((IMetricIdentity)id).getNameComponents();

        if (nameComponents.length != m_nameComponents.length)
        {
            return false;
        }

        for (int i = 0; i < nameComponents.length; i++)
        {
            if (!m_nameComponents[i].equals(nameComponents[i]))
            {
                return false;
            }
        }

        return true;
    }

    /**
     * Tests if the given identity has matching hash.
     */
    public boolean equalsHash(MetricIdentity id) { return id.getHash() == m_hash; }

    /**
     * Tests if the given identity parents this identity.
     */
    @Override
    public boolean isInstanceOf(IMetricIdentity id)
    {
        String[] nameComponents = id.getNameComponents();

        if (nameComponents.length > m_nameComponents.length)
        {
            return false;
        }

        for (int i = 0; i < nameComponents.length; i++)
        {
            if (i > 0 && i == nameComponents.length - 1 && nameComponents[i].indexOf('*') > -1)
            {
                return isPatternMatch(m_nameComponents[i], nameComponents[i]);
            }
            else
            if (!m_nameComponents[i].equals(nameComponents[i]))
            {
                return false;
            }
        }

        return true;
    }

    private static boolean isPatternMatch(String matcher, String pattern)
    {
        if (pattern.length() == 0)
        {
            return false;
        }

        StringTokenizer st = new StringTokenizer(pattern, "*");
        if (st.countTokens() == 0)
        {
            return true;
        }
        boolean startsWithWild = pattern.startsWith("*");
        boolean endsWithWild = pattern.endsWith("*");
        boolean first = true;
        while (st.hasMoreTokens())
        {
            String token = st.nextToken();
            if (first && !startsWithWild)  // special case first token
            {
                 if (!matcher.startsWith(token))
                {
                    return false;
                }
                else
                {
                    matcher = matcher.substring(token.length());
                }
                first = false;
                continue;
            }
            int index = matcher.indexOf(token);
            if (index < 0)
            {
                return false;
            }
            else
            {
                matcher = matcher.substring(index + token.length());
            }
        }

        if (matcher.length() == 0)
        {
            return true;
        }
        else
        {
            return endsWithWild;
        }
    }

    /**
     * Returns the hashcode for this metric for the purpose of indexing in tables (such as HashMap).
     */
    @Override
    public int hashCode() { return getName().hashCode(); }

    @Override
    public String toString() { return getName(); }

    //
    // Serialization
    //

    // You can add to these, but never remove
    private static final short ID_HASH_FIELD = 0;
    private static final short ID_NAME_FIELD = 1;

    // This is done more efficiently than in other places where we employ version
    // handling serialization code .. as there may be many metrics to describe

    private void writeObject(ObjectOutputStream stream)
    throws IOException
    {
        short fieldCount = 2;

        // we know how many fields we will write and serial version will always be written
        stream.writeShort(fieldCount);
        stream.writeShort(m_serialVersion);

        // MetricIdentity specific fields

        stream.writeShort(ID_HASH_FIELD);
        stream.writeLong(m_hash);

        stream.writeShort(ID_NAME_FIELD);
        stream.writeObject(m_nameComponents);
    }

    private void readObject(ObjectInputStream stream)
    throws IOException, ClassNotFoundException
    {
        // read the number of items and stuff them in a hash map
        short numFields = stream.readShort();
        short serialVer = stream.readShort();
        for (int i = 0; i < numFields; i++)
        {
            short field = stream.readShort();
            switch (serialVer)
            {
                // case olderVersion<n> ...
                default:
                {
                    if (field == ID_HASH_FIELD)
                    {
                        m_hash = stream.readLong();
                    }
                    else if (field == ID_NAME_FIELD)
                    {
                        m_nameComponents = (String[]) stream.readObject();
                    }
                }
            }
        }
    }
}
