/**
 * Copyright (c) 1998-2002 Sonic Software Corporation. All Rights Reserved.
 *
 * This software is the confidential and proprietary information of Sonic
 * Software Corpoation. ("Confidential Information").  You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Progress.
 *
 * SONIC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE
 * SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 * PURPOSE, OR NON-INFRINGEMENT. SONIC SHALL NOT BE LIABLE FOR ANY DAMAGES
 * SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR DISTRIBUTING
 * THIS SOFTWARE OR ITS DERIVATIVES.
 *
 * CopyrightVersion 1.0
 */
package com.sonicsw.mf.common.metrics.impl;

import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.Serializable;

import com.sonicsw.mf.common.metrics.IMetricIdentity;
import com.sonicsw.mf.common.metrics.IMetricInfo;
import com.sonicsw.mf.common.metrics.MetricsFactory;

/**
 * Class is not for reference and construction, rather you should use the factory and interface.
 *
 * @see com.sonicsw.mf.common.metrics.MetricsFactory#createMetricInfo(IMetricIdentity, short, String, String, boolean, boolean)
 * @see com.sonicsw.mf.common.metrics.IMetricInfo
 */
public final class MetricInfo
implements IMetricInfo, Serializable
{
    private static final long serialVersionUID = 8276049070158537345L;
    private static final short m_serialVersion = 0;

    private IMetricIdentity m_id;
    private short m_valueType;
    private String m_description;
    private String m_extendedData;
    private boolean m_isInstanceMetric = false;
    private boolean m_isDynamic = false;
    private boolean m_supportsHighAlerts;
    private boolean m_supportsLowAlerts;
    private String m_units;
    private boolean m_isHidden = false; // not serialized

    /**
     * For serialization use only.
     */
    public MetricInfo() { }

    public MetricInfo(IMetricIdentity id, short valueType, String description, String extendedData,
        boolean isInstanceMetric, boolean isDynamic,
        boolean supportsHighAlerts, boolean supportsLowAlerts, String units)
    {
        m_id = id;
        m_valueType = valueType;
        m_description = description;
        m_extendedData = extendedData;
        m_isInstanceMetric = isInstanceMetric;
        m_isDynamic = isDynamic;
        m_supportsHighAlerts = supportsHighAlerts;
        m_supportsLowAlerts = supportsLowAlerts;
        m_units = units;
    }

    /**
     * @see com.sonicsw.mf.common.metrics.IMetricInfo#getMetricIdentity()
     */
    @Override
    public IMetricIdentity getMetricIdentity() { return m_id; }

    /**
     * @see com.sonicsw.mf.common.metrics.IMetricInfo#getValueType()
     */
    @Override
    public short getValueType() { return m_valueType; }

    /**
     * @see com.sonicsw.mf.common.metrics.IMetricInfo#getDescription()
     */
    @Override
    public String getDescription() { return m_description; }

    /**
     * @see com.sonicsw.mf.common.metrics.IMetricInfo#getExtendedData()
     */
    @Override
    public String getExtendedData() { return m_extendedData; }

    /**
     * @see com.sonicsw.mf.common.metrics.IMetricInfo#isInstanceMetric()
     */
    @Override
    public boolean isInstanceMetric() { return m_isInstanceMetric; }

    /**
     * @see com.sonicsw.mf.common.metrics.IMetricInfo#isDynamic()
     */
    @Override
    public boolean isDynamic() { return m_isDynamic; }

    /**
     * @see com.sonicsw.mf.common.metrics.IMetricInfo#supportsHighThresholdAlerts()
     */
    @Override
    public boolean supportsHighThresholdAlerts() { return m_supportsHighAlerts; }

    /**
     * @see com.sonicsw.mf.common.metrics.IMetricInfo#supportsLowThresholdAlerts()
     */
    @Override
    public boolean supportsLowThresholdAlerts() { return m_supportsLowAlerts; }

   /**
     * @see com.sonicsw.mf.common.metrics.IMetricInfo#getUnits()
     */
    @Override
    public String getUnits() { return m_units; }

    /**
     * Set/get internal metric flag
    */
    public void setHidden(boolean t) { m_isHidden = t;}
    public boolean isHidden() { return m_isHidden;}

    /**
     * toString()
     */
    @Override
    public String toString()
    {
        return "MetricInfo for " + m_id + "\n desc = " + m_description
            + "\n type = " + m_valueType + " instance = " + m_isInstanceMetric + " dyn = " + m_isDynamic
            + "\n hiA = " + m_supportsHighAlerts + " lowA = " + m_supportsLowAlerts + " Units = " + m_units;
    }
     //
    // Serialization
    //

    // You can add to these, but never remove
    private static final short ID_NAME_FIELD = 0;
    private static final short ID_HASH_FIELD = 1;
    private static final short VALUE_TYPE_FIELD = 2;
    private static final short DESCRIPTION_FIELD = 3;
    private static final short EXTENDED_DATA_FIELD = 4;
    private static final short IS_INSTANCE_METRIC_FIELD = 5;
    private static final short IS_DYNAMIC_FIELD = 6;
    private static final short SUPPORTS_HIGH_ALERTS_FIELD = 7;
    private static final short SUPPORTS_LOW_ALERTS_FIELD = 8;
    private static final short UNITS_FIELD = 9;

    // This is done more efficiently than in other places where we employ version
    // handling serialization code .. as there may be many metrics to describe

    private void writeObject(ObjectOutputStream stream)
    throws IOException
    {
        short fieldCount = 2;
        if (((MetricIdentity)m_id).getHash() != 0)
        {
            fieldCount++;
        }
        if (m_description != null)
        {
            fieldCount++;
        }
        if (m_extendedData != null)
        {
            fieldCount++;
        }
        if (m_isInstanceMetric)
        {
            fieldCount++;
        }
        if (m_isDynamic)
        {
            fieldCount++;
        }
        if (m_supportsHighAlerts)
        {
            fieldCount++;
        }
        if (m_supportsLowAlerts)
        {
            fieldCount++;
        }
        if (m_units != null)
        {
            fieldCount++;
        }

        // we know how many fields we will write and serial version will always be written
        stream.writeShort(fieldCount);
        stream.writeShort(m_serialVersion);

        // MetricInfo specific fields

        stream.writeShort(ID_NAME_FIELD);
        stream.writeObject(m_id.getNameComponents());

        if (((MetricIdentity)m_id).getHash() != 0)
        {
            stream.writeShort(ID_HASH_FIELD);
            stream.writeLong(((MetricIdentity)m_id).getHash());
        }

        stream.writeShort(VALUE_TYPE_FIELD);
        stream.writeShort(m_valueType);

        if (m_description != null)
        {
            stream.writeShort(DESCRIPTION_FIELD);
            stream.writeUTF(m_description);
        }

        if (m_extendedData != null)
        {
            stream.writeShort(EXTENDED_DATA_FIELD);
            stream.writeUTF(m_extendedData);
        }

        if (m_isInstanceMetric)
        {
            stream.writeShort(IS_INSTANCE_METRIC_FIELD);
            stream.writeBoolean(m_isInstanceMetric);
        }

        if (m_isDynamic)
        {
            stream.writeShort(IS_DYNAMIC_FIELD);
            stream.writeBoolean(m_isDynamic);
        }

        if (m_supportsHighAlerts)
        {
            stream.writeShort(SUPPORTS_HIGH_ALERTS_FIELD);
            stream.writeBoolean(m_supportsHighAlerts);
        }

        if (m_supportsLowAlerts)
        {
            stream.writeShort(SUPPORTS_LOW_ALERTS_FIELD);
            stream.writeBoolean(m_supportsLowAlerts);
        }

        if (m_units != null)
        {
            stream.writeShort(UNITS_FIELD);
            stream.writeUTF(m_units);
        }
    }

    private void readObject(ObjectInputStream stream)
    throws IOException, ClassNotFoundException
    {
        String[] nameComponents = null;
        long hash = 0;

        // read the number of items and stuff them in a hash map
        short numFields = stream.readShort();
        short serialVer = stream.readShort();
        for (int i = 0; i < numFields; i++)
        {
            short field = stream.readShort();
            switch (serialVer)
            {
                // case olderVersion<n> ...
                default:
                    if (field == ID_NAME_FIELD)
                    {
                        nameComponents = (String[])stream.readObject();
                    }
                    else if (field == ID_HASH_FIELD)
                    {
                        hash = stream.readLong();
                    }
                    else if (field == VALUE_TYPE_FIELD)
                    {
                        m_valueType = stream.readShort();
                    }
                    else if (field == DESCRIPTION_FIELD)
                    {
                        m_description = stream.readUTF();
                    }
                    else if (field == EXTENDED_DATA_FIELD)
                    {
                        m_extendedData = stream.readUTF();
                    }
                    else if (field == IS_INSTANCE_METRIC_FIELD)
                    {
                        m_isInstanceMetric = stream.readBoolean();
                    }
                    else if (field == IS_DYNAMIC_FIELD)
                    {
                        m_isDynamic = stream.readBoolean();
                    }
                    else if (field == SUPPORTS_HIGH_ALERTS_FIELD)
                    {
                        m_supportsHighAlerts = stream.readBoolean();
                    }
                    else if (field == SUPPORTS_LOW_ALERTS_FIELD)
                    {
                        m_supportsLowAlerts = stream.readBoolean();
                    }
                    else if (field == UNITS_FIELD)
                    {
                        m_units = stream.readUTF();
                    }
                    break;
            }
        }

        m_id = MetricsFactory.createMetricIdentity(nameComponents);
        ((MetricIdentity)m_id).setHash(hash);
    }
}
